import React, { useState, useEffect } from "react";
import { useNavigate } from "@remix-run/react";
import { authAPI, handleAuthSuccess } from "~/lib/api";
import { Button } from "../ui/button";
import { useLanguage } from "~/hooks/useLanguage";

const SignInEmail: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const [isVisible, setIsVisible] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [emailFormData, setEmailFormData] = useState({
    email: "",
    password: "",
    keepLoggedIn: true,
  });
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    setIsVisible(true);
  }, []);

  const handleEmailInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type, checked } = e.target;
    setEmailFormData((prev) => ({
      ...prev,
      [name]: type === "checkbox" ? checked : value,
    }));
    if (error) setError(null);
  };

  const handleEmailSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);

    try {
      const response = await authAPI.login(
        emailFormData.email,
        emailFormData.password
      );

      if (response.status && response.token) {
        handleAuthSuccess(response);

        // Check if there's a pending booking
        const pendingBooking =
          typeof window !== "undefined"
            ? sessionStorage.getItem("pendingBooking")
            : null;

        setTimeout(() => {
          if (pendingBooking) {
            const bookingData = JSON.parse(pendingBooking);
            // Redirect back to the profile page to continue booking
            window.location.href = bookingData.returnUrl || "/home";
          } else {
            // Regular login, go to home
            window.location.href = "/home"; // Full reload to reset state
          }
        }, 100);
      } else {
        setError(
          response.error ||
            t(
              "auth.signinemail.errors.loginFailed",
              "Login failed. Please try again."
            )
        );
      }
    } catch (err) {
      console.error("Login error:", err);
      setError(
        err instanceof Error
          ? err.message
          : t(
              "auth.signinemail.errors.unexpected",
              "An unexpected error occurred. Please try again."
            )
      );
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div
          className="absolute top-20 left-10 w-2 h-2 bg-orange1-400 rounded-full animate-bounce opacity-60"
          style={{ animationDelay: "0s", animationDuration: "3s" }}
        />
        <div
          className="absolute top-40 right-16 w-1 h-1 bg-white1 rounded-full animate-pulse opacity-40"
          style={{ animationDelay: "1s" }}
        />
        <div
          className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30"
          style={{ animationDelay: "2s" }}
        />
        <div
          className="absolute bottom-20 right-10 w-2 h-2 bg-white1 rounded-full animate-bounce opacity-50"
          style={{ animationDelay: "1.5s", animationDuration: "4s" }}
        />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0">
        {/* Hero Title */}
        <div
          className={`text-center mb-12 transform transition-all duration-1000 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t("auth.signinemail.title", "SIGN IN")}
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light">
            {t("auth.signinemail.subtitle", "Access your creative workspace")}
          </p>
        </div>

        {/* Login Card */}
        <div
          className={`w-full max-w-md transform transition-all duration-1000 delay-300 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            <h2 className="text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider">
              {t("auth.signinemail.emailLogin", "Email Login")}
            </h2>

            {/* Error Message */}
            {error && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {error}
              </div>
            )}

            {/* Email Login Form */}
            <form onSubmit={handleEmailSubmit} className="space-y-6">
              {/* Email */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  {t("auth.signinemail.emailAddress", "Email Address")}
                </label>
                <input
                  type="email"
                  name="email"
                  placeholder={t(
                    "auth.signinemail.emailPlaceholder",
                    "Enter your email"
                  )}
                  value={emailFormData.email}
                  onChange={handleEmailInputChange}
                  className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                  required
                  disabled={isLoading}
                />
              </div>
              {/* Password */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  {t("auth.signinemail.password", "Password")}
                </label>
                <div className="relative">
                  <input
                    type={showPassword ? "text" : "password"}
                    name="password"
                    placeholder={t(
                      "auth.signinemail.passwordPlaceholder",
                      "Enter your password"
                    )}
                    value={emailFormData.password}
                    onChange={handleEmailInputChange}
                    className="w-full px-4 py-4 pr-12 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                    disabled={isLoading}
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword(!showPassword)}
                    className="absolute right-4 top-1/2 transform -translate-y-1/2 text-white/60 hover:text-white/80 transition-colors"
                    disabled={isLoading}
                  >
                    {showPassword ? (
                      <svg
                        className="w-5 h-5"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21"
                        />
                      </svg>
                    ) : (
                      <svg
                        className="w-5 h-5"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                        />
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
                        />
                      </svg>
                    )}
                  </button>
                </div>
              </div>
              {/* Keep me logged in & Forgot Password */}
              <div className="flex items-center justify-between">
                <div className="flex items-center">
                  <input
                    type="checkbox"
                    id="keepLoggedInEmail"
                    name="keepLoggedIn"
                    checked={emailFormData.keepLoggedIn}
                    onChange={handleEmailInputChange}
                    className="w-5 h-5 text-orange-500 bg-white/10 border-white/20 rounded focus:ring-orange-400 focus:ring-2"
                  />
                  <label
                    htmlFor="keepLoggedInEmail"
                    className="ml-3 text-sm text-white/80"
                  >
                    {t("auth.signinemail.keepLoggedIn", "Keep me logged in")}
                  </label>
                </div>
                <Button
                  type="button"
                  variant="outline-white"
                  size="sm"
                  onClick={() => navigate("/reset-password")}
                >
                  {t("auth.signinemail.forgotPassword", "Forgot Password?")}
                </Button>
              </div>{" "}
              {/* Login button */}
              <Button
                type="submit"
                variant="fill-white"
                size="lg"
                className="w-full"
                disabled={isLoading}
                arrowRight={!isLoading}
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                    {t("auth.signinemail.signingIn", "Signing in...")}
                  </>
                ) : (
                  t("auth.signinemail.signIn", "Sign In")
                )}
              </Button>
              {/* Alternative Options */}
              <div className="text-center space-y-4 pt-4">
                <Button
                  type="button"
                  variant="outline-white"
                  size="md"
                  className="w-full"
                  onClick={() => navigate("/login")}
                >
                  {t(
                    "auth.signinemail.loginWithPhone",
                    "Login with phone instead"
                  )}
                </Button>

                <div className="text-white/60 text-sm">
                  {t("auth.signinemail.noAccount", "Don't have an account?")}{" "}
                  <Button
                    type="button"
                    variant="outline-white"
                    size="sm"
                    className="ml-2"
                    onClick={() => navigate("/signup")}
                  >
                    {t("auth.signinemail.signUp", "Sign up")}
                  </Button>
                </div>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
};

export default SignInEmail;
