import React, { useState, useEffect } from "react";
import { useNavigate } from "@remix-run/react";
import { Button } from "../ui/button";
import { useLanguage } from "~/hooks/useLanguage";
import { plansAPI, type Plan, type Service } from "~/lib/api";

const SignupFirst: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const [isVisible, setIsVisible] = useState(false);
  const [selectedTier, setSelectedTier] = useState<number | null>(null);
  const [selectedAddOns, setSelectedAddOns] = useState<number[]>([]);
  const [plans, setPlans] = useState<Plan[]>([]);
  const [service, setService] = useState<Service | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    setIsVisible(true);

    const fetchPlans = async () => {
      try {
        setLoading(true);
        const response = await plansAPI.getAll();
        if (response.status && response.data) {
          setPlans(response.data.plans);
          setService(response.data.services);
        } else {
          setError("Failed to fetch plans");
        }
      } catch (err) {
        setError(err instanceof Error ? err.message : "Failed to fetch plans");
      } finally {
        setLoading(false);
      }
    };

    fetchPlans();
  }, []);

  const generateFeatures = (plan: Plan): string[] => {
    const features: string[] = [];
    if (plan.advance_duration > 0) {
      features.push(
        `Book up ${plan.advance_duration} ${plan.advance_frequency}${
          plan.advance_duration > 1 ? "s" : ""
        } in advance`
      );
    } else {
      features.push("Just one day/one by member");
    }
    if (plan.num_of_guest > 0) {
      features.push(
        `${
          plan.num_of_guest === 1 ? "One guest" : `${plan.num_of_guest} guests`
        }`
      );
    } else {
      features.push("Zero guests");
    }
    return features;
  };

  const handleAddOnToggle = (serviceId: number) => {
    setSelectedAddOns((prev) =>
      prev.includes(serviceId)
        ? prev.filter((id) => id !== serviceId)
        : [...prev, serviceId]
    );
  };

  const handleTierSelect = (index: number) => {
    setSelectedTier(index);
  };

  return (
    <div
      className="min-h-screen bg-[#121212] relative overflow-hidden"
      style={{ fontFamily: "Clash Grotesk Variable, sans-serif" }}
    >
      {/* Main Content */}
      <div className="relative z-10 min-h-screen">
        {/* Header Section - Frame 1000003600 */}
        <div className="w-full max-w-[1440px] mx-auto pt-16 sm:pt-24 md:pt-[134px] pb-12 sm:pb-20 md:pb-[134px] px-4 sm:px-8 md:px-[56px]">
          {/* Hero Text */}
          <div className="text-center mb-8">
            <h1 className="text-white text-[32px] sm:text-[48px] md:text-[64px] lg:text-[80px] font-bold uppercase tracking-tight leading-[1.1] mb-4 sm:mb-6">
              JOIN THE NOMADIC COMMUNITY
            </h1>
            <p className="text-white/80 text-sm sm:text-base font-normal max-w-2xl mx-auto leading-relaxed mb-2 px-4">
              A global membership network for professional creators.
            </p>
            <p className="text-white/80 text-sm sm:text-base font-normal max-w-3xl mx-auto leading-relaxed px-4">
              Access world-class studios, tools, and expert support — without
              the commitment
              <br className="hidden sm:inline" />
              of owning or maintaining a dedicated facility.
            </p>
          </div>

          {/* Learn More Button */}
          <div className="text-center mt-6 sm:mt-8">
            <button
              onClick={() => {
                const element = document.getElementById("pricing-section");
                element?.scrollIntoView({ behavior: "smooth" });
              }}
              className="px-6 sm:px-8 py-2.5 sm:py-3 border border-white text-white text-xs sm:text-sm uppercase tracking-wider font-medium hover:bg-white hover:text-black transition-all"
            >
              LEARN MORE
            </button>
          </div>
        </div>

        {/* Plans Section - Row with 4 cards */}
        <div
          id="pricing-section"
          className="w-full max-w-[1440px] mx-auto border-t border-white/10 px-4 sm:px-6 md:px-[56px]"
        >
          <div className="flex flex-col md:flex-row md:flex-wrap gap-4 p-0">
            {loading ? (
              <div className="w-full text-center py-16">
                <div className="text-white/80">Loading plans...</div>
              </div>
            ) : error ? (
              <div className="w-full text-center py-16">
                <div className="text-red-400">Error: {error}</div>
              </div>
            ) : (
              plans.map((plan, index) => {
                const isPopular = plan.slug === "up-and-coming";
                const isSelected = selectedTier === index;
                const shouldShowPopularStyle = isSelected;

                return (
                  <div
                    key={plan.id}
                    className={`relative flex-1 flex flex-col w-full md:w-[300px] md:h-[510px] py-8 px-6 ${
                      shouldShowPopularStyle ? "bg-[#121212]" : "bg-white"
                    } border-2 ${
                      shouldShowPopularStyle
                        ? "border-[#D64933]"
                        : "border-[#121212]"
                    } cursor-pointer transition-all`}
                    onClick={() => handleTierSelect(index)}
                  >
                    {/* Selected Badge */}
                    {shouldShowPopularStyle && (
                      <div className="absolute -top-0 left-0 right-0">
                        <div className="bg-[#D64933] text-white text-[11px] px-3 py-2 text-center uppercase tracking-wider font-semibold">
                          SELECTED
                        </div>
                      </div>
                    )}

                    {/* Card Content */}
                    <div
                      className={`flex flex-col flex-grow gap-8 ${
                        shouldShowPopularStyle ? "mt-6" : ""
                      }`}
                    >
                      {/* Title */}
                      <h3
                        className={`${
                          shouldShowPopularStyle ? "text-white" : "text-black"
                        } text-sm font-semibold uppercase tracking-wide`}
                      >
                        {plan.name}
                      </h3>

                      {/* Price */}
                      <div>
                        <div className="flex items-baseline gap-1">
                          <span
                            className={`text-[48px] font-bold ${
                              shouldShowPopularStyle
                                ? "text-white"
                                : "text-black"
                            } leading-none`}
                          >
                            ${plan.price}
                          </span>
                          <span
                            className={`${
                              shouldShowPopularStyle
                                ? "text-white/60"
                                : "text-black/60"
                            } text-sm`}
                          >
                            /{plan.frequency}
                          </span>
                        </div>
                      </div>

                      {/* Points */}
                      <div className="flex items-center gap-2">
                        <svg
                          className="w-4 h-4"
                          fill="none"
                          viewBox="0 0 24 24"
                          stroke="#D64933"
                          strokeWidth={2.5}
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            d="M5 13l4 4L19 7"
                          />
                        </svg>
                        <span className="text-[#D64933] text-sm font-normal">
                          Get {plan.points} Nomadic Points
                        </span>
                      </div>

                      {/* Features List */}
                      <ul
                        className={`flex-grow space-y-3 ${
                          shouldShowPopularStyle ? "text-white" : "text-black"
                        } text-sm`}
                      >
                        {index === 0 ? (
                          <>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>Attend as a guest of a member</span>
                            </li>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>
                                Book day-of studio time (subject to
                                availability)
                              </span>
                            </li>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>Join workshops and events</span>
                            </li>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>Upgrade anytime</span>
                            </li>
                            <li className="flex items-start gap-3 opacity-30">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>Cannot invite guests or refer members</span>
                            </li>
                          </>
                        ) : (
                          <>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>Includes all Guest features and...</span>
                            </li>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>
                                {index === 1
                                  ? "Monthly creative balance for studio use and bookings"
                                  : index === 2
                                  ? "Expanded creative balance for frequent use"
                                  : "Premium balance for professionals and teams"}
                              </span>
                            </li>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>
                                Book up to {plan.advance_duration}{" "}
                                {plan.advance_frequency}
                                {plan.advance_duration > 1 ? "s" : ""} in
                                advance
                              </span>
                            </li>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>
                                {plan.num_of_guest} guest
                                {plan.num_of_guest > 1 ? "s" : ""} included
                              </span>
                            </li>
                            <li className="flex items-start gap-3">
                              <div
                                className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                  shouldShowPopularStyle
                                    ? "border-white"
                                    : "border-black"
                                } flex items-center justify-center`}
                              >
                                <svg
                                  className={`w-3 h-3`}
                                  fill="currentColor"
                                  viewBox="0 0 24 24"
                                >
                                  <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                </svg>
                              </div>
                              <span>
                                Usage valid for {plan.advance_duration}{" "}
                                {plan.advance_frequency}
                                {plan.advance_duration > 1 ? "s" : ""}
                              </span>
                            </li>
                          </>
                        )}
                      </ul>

                      {/* Button */}
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          if (selectedTier === index) {
                            // Navigate to next page with selected plan data
                            navigate("/signup-details", {
                              state: {
                                selectedTier: plan,
                                selectedAddOns: [],
                              },
                            });
                          } else {
                            // Select the plan
                            handleTierSelect(index);
                          }
                        }}
                        className={`w-full py-3 text-sm uppercase tracking-wider font-semibold border-2 transition-all cursor-pointer ${
                          shouldShowPopularStyle
                            ? "bg-white text-black border-white hover:bg-white/90"
                            : "bg-white text-black border-black hover:bg-black hover:text-white"
                        }`}
                      >
                        {selectedTier === index ? "APPLY NOW" : "SELECT"}
                      </button>
                    </div>
                  </div>
                );
              })
            )}
          </div>
        </div>

        {/* Services/Benefits Section */}
        <div className="w-full bg-white py-[120px] px-[56px] mt-[-230px]">
          <div className="max-w-[1440px] mx-auto pt-[250px]">
            {/* Header Section */}
            <div className="flex flex-col lg:flex-row gap-[24px] mb-[120px]">
              {/* Left - Title */}
              <div className="lg:w-[538px]">
                <p className="text-black/40 text-[10px] uppercase tracking-[0.2em] mb-6 font-medium">
                  [BENEFITS]
                </p>

                <h2 className="text-black text-[32px] sm:text-[44px] md:text-[68px] font-medium uppercase leading-[1] md:leading-[0.9] tracking-[-0.03em]">
                  MEMBERSHIP
                  <br />
                  THAT MOVES
                  <br />
                  WITH YOU
                </h2>
              </div>

              {/* Right - Description */}
              <div className="flex-1 flex items-center">
                <div>
                  <p className="text-black text-[16px] leading-[32px] mb-6 font-normal">
                    Nomadic Studios gives working creators flexible access to{" "}
                    <span className="font-medium">
                      dedicated studio time, equipment, events,
                    </span>{" "}
                    and <span className="font-medium">technical teams</span> —
                    available in every city within our network.
                  </p>
                  <p className="text-black text-[16px] leading-[32px] font-normal">
                    Each plan includes a monthly creative balance{" "}
                    <span className="text-[#D64933] font-medium">
                      (Nomadic Points)
                    </span>{" "}
                    you can apply across disciplines and locations.
                  </p>
                </div>
              </div>
            </div>

            {/* Benefits Grid - 3 columns */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-[24px]">
              {/* CREATE ANYWHERE */}
              <div className="border border-black/30 p-8">
                <div className="w-10 h-10 mb-6">
                  <svg
                    width="40"
                    height="40"
                    viewBox="0 0 40 40"
                    fill="none"
                    xmlns="http://www.w3.org/2000/svg"
                  >
                    <path
                      d="M20 8V32M20 8L13 15M20 8L27 15"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                    <path
                      d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                  </svg>
                </div>
                <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                  CREATE ANYWHERE
                </h3>
                <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                  Book{" "}
                  <span className="font-medium">dedicated studio time</span> for{" "}
                  <span className="font-medium">music, video</span> and{" "}
                  <span className="font-medium">
                    streaming, fashion, art, photography,
                  </span>{" "}
                  or{" "}
                  <span className="font-medium">immersive/maker projects</span>{" "}
                  — all under one membership.
                </p>
              </div>

              {/* FLEXIBLE ACCESS */}
              <div className="border border-black/30 p-8">
                <div className="w-10 h-10 mb-6">
                  <svg
                    width="40"
                    height="40"
                    viewBox="0 0 40 40"
                    fill="none"
                    xmlns="http://www.w3.org/2000/svg"
                  >
                    <path
                      d="M20 8V32M20 8L13 15M20 8L27 15"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                    <path
                      d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                  </svg>
                </div>
                <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                  FLEXIBLE ACCESS
                </h3>
                <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                  Each session draws from your creative balance. The cost of
                  resources automatically adjusts based on city and type.
                </p>
              </div>

              {/* TIERED BENEFITS */}
              <div className="border border-black/30 p-8">
                <div className="w-10 h-10 mb-6">
                  <svg
                    width="40"
                    height="40"
                    viewBox="0 0 40 40"
                    fill="none"
                    xmlns="http://www.w3.org/2000/svg"
                  >
                    <path
                      d="M20 8V32M20 8L13 15M20 8L27 15"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                    <path
                      d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                  </svg>
                </div>
                <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                  TIERED BENEFITS
                </h3>
                <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                  Higher tiers provide larger balances, extended validity,
                  earlier booking windows, and more included guests.
                </p>
              </div>

              {/* BRING GUESTS */}
              <div className="border border-black/30 p-8">
                <div className="w-10 h-10 mb-6">
                  <svg
                    width="40"
                    height="40"
                    viewBox="0 0 40 40"
                    fill="none"
                    xmlns="http://www.w3.org/2000/svg"
                  >
                    <path
                      d="M20 8V32M20 8L13 15M20 8L27 15"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                    <path
                      d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                  </svg>
                </div>
                <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                  BRING GUESTS
                </h3>
                <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                  Invite collaborators, clients, or assistants to your booked
                  sessions. Additional guests can join for a small usage fee.
                </p>
              </div>

              {/* WORKSHOPS & EVENTS */}
              <div className="border border-black/30 p-8">
                <div className="w-10 h-10 mb-6">
                  <svg
                    width="40"
                    height="40"
                    viewBox="0 0 40 40"
                    fill="none"
                    xmlns="http://www.w3.org/2000/svg"
                  >
                    <path
                      d="M20 8V32M20 8L13 15M20 8L27 15"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                    <path
                      d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                  </svg>
                </div>
                <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                  WORKSHOPS & EVENTS
                </h3>
                <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                  Apply your balance toward hands-on workshops and community
                  events that connect creators and expand skills.
                </p>
              </div>

              {/* REFERRAL-ONLY MEMBERSHIP */}
              <div className="border border-black/30 p-8">
                <div className="w-10 h-10 mb-6">
                  <svg
                    width="40"
                    height="40"
                    viewBox="0 0 40 40"
                    fill="none"
                    xmlns="http://www.w3.org/2000/svg"
                  >
                    <path
                      d="M20 8V32M20 8L13 15M20 8L27 15"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                    <path
                      d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                      stroke="black"
                      strokeWidth="2"
                      strokeLinecap="round"
                      strokeLinejoin="round"
                    />
                  </svg>
                </div>
                <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                  REFERRAL-ONLY
                  <br />
                  MEMBERSHIP
                </h3>
                <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                  New members join by referral to ensure that every addition
                  strengthens the professional community.
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* How It Works Section */}
        <div className="w-full bg-[#121212] py-12 sm:py-20 md:py-[120px] px-4 sm:px-6 md:px-[56px]">
          <div className="max-w-[1440px] mx-auto">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 sm:gap-8 md:gap-[56px] relative">
              {/* Left Column */}
              <div>
                <p className="text-white/60 text-[16px] uppercase tracking-[0.02em] mb-6 font-normal leading-[100%]">
                  [HOW IT WORKS]
                </p>
                <h2 className="text-white text-[32px] sm:text-[44px] md:text-[68px] font-medium uppercase leading-[1.05] md:leading-[1] tracking-[-0.03em] mb-30">
                  ONE MEMBERSHIP.
                  <br />
                  GLOBAL ACCESS
                </h2>
                <div className="space-y-0">
                  <p className="text-white text-[24px] leading-[32px] font-normal tracking-[0%]">
                    Your membership includes a flexible creative balance
                    <br />
                    <span className="text-[#D64933] font-medium">
                      (Nomadic Points)
                    </span>{" "}
                    you can use for studio time, equipment,
                    <br />
                    events, and technical or support teams.
                  </p>
                  <p className="text-white text-[24px] leading-[32px] font-normal tracking-[0%]">
                    The cost of each resource adjusts automatically to reflect
                    <br />
                    local studio rates — so you can create confidently wherever
                    <br />
                    you go.
                  </p>
                </div>
              </div>

              {/* Center - Vertical Line with Dots */}
              <div className="hidden lg:block absolute left-1/2 top-0 bottom-0 -translate-x-1/2">
                <div className="relative h-full w-[2px]">
                  {/* Vertical Line */}
                  <div className="absolute inset-0 bg-[#D64933]"></div>
                  {/* Dots */}
                  {[0, 1, 2, 3, 4].map((index) => (
                    <div
                      key={index}
                      className="absolute left-1/2 -translate-x-1/2"
                      style={{ top: `${20 + index * 18}%` }}
                    >
                      <svg
                        width="12"
                        height="12"
                        viewBox="0 0 12 12"
                        fill="none"
                      >
                        <rect width="12" height="12" fill="#D64933" />
                      </svg>
                    </div>
                  ))}
                  {/* Arrow at bottom */}
                  <div className="absolute bottom-0 left-1/2 -translate-x-1/2">
                    <svg width="12" height="20" viewBox="0 0 12 20" fill="none">
                      <path
                        d="M6 0L6 16M6 16L1 11M6 16L11 11"
                        stroke="#D64933"
                        strokeWidth="2"
                      />
                    </svg>
                  </div>
                </div>
              </div>

              {/* Right Column - Text Items */}
              <div className="flex flex-col justify-center">
                <div className="space-y-0">
                  {[
                    "BOOK STUDIOS, EQUIPMENT, WORKSHOPS, AND SUPPORT TEAMS DIRECTLY THROUGH YOUR DASHBOARD.",
                    "RESOURCE COSTS ARE AUTOMATICALLY CALCULATED PER CITY.",
                    "ADD MORE BALANCE ANYTIME AT YOUR MEMBER RATE.",
                    "HIGHER TIERS PROVIDE LONGER VALIDITY AND EARLY BOOKING PRIVILEGES.",
                    "APPLY YOUR BALANCE TOWARD EVENTS, WORKSHOPS, AND GUEST ACCESS.",
                  ].map((text, index) => (
                    <div
                      key={index}
                      className="py-6 border-t border-white/10 first:border-t-0"
                    >
                      <p className="text-white text-[18px] sm:text-[20px] md:text-[32px] leading-[28px] sm:leading-[32px] md:leading-[40px] uppercase font-normal tracking-[-0.01em]">
                        {text}
                      </p>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* CTA Section */}
        <div className="w-full bg-white relative overflow-hidden py-[134px] px-[56px]">
          {/* Concentric circles background pattern */}
          <div className="absolute inset-0 pointer-events-none overflow-hidden">
            {/* Top-left circles */}
            <div className="absolute -top-[400px] -left-[400px]">
              {[...Array(12)].map((_, i) => (
                <div
                  key={`tl-${i}`}
                  className="absolute rounded-full border"
                  style={{
                    width: `${300 + i * 120}px`,
                    height: `${300 + i * 120}px`,
                    borderColor: `rgba(18, 18, 18, ${0.08 - i * 0.005})`,
                    borderWidth: "1px",
                  }}
                />
              ))}
            </div>
            {/* Bottom-right circles */}
            <div className="absolute -bottom-[400px] -right-[400px]">
              {[...Array(12)].map((_, i) => (
                <div
                  key={`br-${i}`}
                  className="absolute rounded-full border"
                  style={{
                    width: `${300 + i * 120}px`,
                    height: `${300 + i * 120}px`,
                    borderColor: `rgba(18, 18, 18, ${0.08 - i * 0.005})`,
                    borderWidth: "1px",
                  }}
                />
              ))}
            </div>
          </div>

          {/* Content */}
          <div className="relative z-10 max-w-[1100px] mx-auto text-center">
            <div className="mb-8">
              <h2 className="text-[#121212] text-[28px] sm:text-[40px] md:text-[68px] font-semibold uppercase tracking-[-0.03em] leading-[1.05] md:leading-[1.2] mb-0">
                A CURATED NETWORK FOR
              </h2>
              <h2 className="text-[#121212] text-[28px] sm:text-[40px] md:text-[68px] font-semibold uppercase tracking-[-0.03em] leading-[1.05] md:leading-[1.2]">
                SERIOUS CREATIVES
              </h2>
            </div>

            <p className="text-[#52525B] text-[18px] leading-[32px] font-normal max-w-[651px] mx-auto mb-12">
              Nomadic Studios blends the accessibility of co-working with the
              depth of professional production environments. Membership is
              intentionally limited in each city to ensure consistent access to
              top-tier facilities and support.
            </p>

            <button
              onClick={() => {
                const element = document.getElementById("pricing-section");
                element?.scrollIntoView({ behavior: "smooth" });
              }}
              className="inline-flex items-center gap-3 bg-[#121212] text-white px-10 py-5 text-[18px] uppercase tracking-[0.02em] font-medium hover:bg-[#121212]/90 transition-all"
            >
              APPLY FOR MEMBERSHIP
              <svg
                className="w-5 h-5"
                fill="none"
                viewBox="0 0 24 24"
                stroke="currentColor"
                strokeWidth={2}
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  d="M17 8l4 4m0 0l-4 4m4-4H3"
                />
              </svg>
            </button>
          </div>
        </div>

        {/* Footer Links */}
        <div className="text-center py-8">
          <div className="text-white/60 text-sm">
            Already have an account?{" "}
            <button
              onClick={() => navigate("/login")}
              className="text-white underline hover:text-white/80 transition-colors"
            >
              Sign in
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default SignupFirst;
