import React, { useState, useEffect } from 'react';
import { useSearchParams, Link, useNavigate, useLocation } from '@remix-run/react';
import { authAPI } from '~/lib/api';
import { Button } from '../ui/button';
import { useLanguage } from '~/hooks/useLanguage';

interface CountryCode {
  code: string;
  country: string;
  flag: string;
}

const SignupSecond: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [searchParams] = useSearchParams();
  const [isVisible, setIsVisible] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  
  // Get data from location state (from SignupFirst) or fallback to URL params
  const selectedTierData = location.state?.selectedTier;
  const selectedAddOnsData = location.state?.selectedAddOns || [];
  const selectedPlan = selectedTierData?.name || searchParams.get('plan') || 'Guest';
  const selectedPrice = selectedTierData?.price || searchParams.get('price') || '$0';
  
  // Check if returning from OTP page with saved data
  const savedFormData = location.state?.formData;
  const savedCountryCode = location.state?.countryCode;
  
  const [formData, setFormData] = useState({
    firstName: savedFormData?.firstName || '',
    lastName: savedFormData?.lastName || '',
    email: savedFormData?.email || '',
    mobileNumber: savedFormData?.mobileNumber || ''
  });

  const [selectedCountry, setSelectedCountry] = useState<CountryCode>(
    savedCountryCode || {
      code: '+1',
      country: 'United States',
      flag: '🇺🇸'
    }
  );

  const [isDropdownOpen, setIsDropdownOpen] = useState(false);

  const countryCodes: CountryCode[] = [
    { code: "+61", country: "Australia", flag: "🇦🇺" },
    { code: "+43", country: "Austria", flag: "🇦🇹" },
    { code: "+54", country: "Argentina", flag: "🇦🇷" },
    { code: "+32", country: "Belgium", flag: "🇧🇪" },
    { code: "+359", country: "Bulgaria", flag: "🇧🇬" },
    { code: "+55", country: "Brazil", flag: "🇧🇷" },
    { code: "+56", country: "Chile", flag: "🇨🇱" },
    { code: "+86", country: "China", flag: "🇨🇳" },
    { code: "+420", country: "Czech Republic", flag: "🇨🇿" },
    { code: "+385", country: "Croatia", flag: "🇭🇷" },
    { code: "+20", country: "Egypt", flag: "🇪🇬" },
    { code: "+33", country: "France", flag: "🇫🇷" },
    { code: "+358", country: "Finland", flag: "🇫🇮" },
    { code: "+30", country: "Greece", flag: "🇬🇷" },
    { code: "+49", country: "Germany", flag: "🇩🇪" },
    { code: "+353", country: "Ireland", flag: "🇮🇪" },
    { code: "+91", country: "India", flag: "🇮🇳" },
    { code: "+972", country: "Israel", flag: "🇮🇱" },
    { code: "+62", country: "Indonesia", flag: "🇮🇩" },
    { code: "+81", country: "Japan", flag: "🇯🇵" },
    { code: "+254", country: "Kenya", flag: "🇰🇪" },
    { code: "+82", country: "South Korea", flag: "🇰🇷" },
    { code: "+36", country: "Hungary", flag: "🇭🇺" },
    { code: "+385", country: "Croatia", flag: "🇭🇷" },
    { code: "+39", country: "Italy", flag: "🇮🇹" },
    { code: "+353", country: "Ireland", flag: "🇮🇪" },
    { code: "+234", country: "Nigeria", flag: "🇳🇬" },
    { code: "+31", country: "Netherlands", flag: "🇳🇱" },
    { code: "+64", country: "New Zealand", flag: "🇳🇿" },
    { code: "+351", country: "Portugal", flag: "🇵🇹" },
    { code: "+48", country: "Poland", flag: "🇵🇱" },
    { code: "+51", country: "Peru", flag: "🇵🇪" },
    { code: "+63", country: "Philippines", flag: "🇵🇭" },
    { code: "+966", country: "Saudi Arabia", flag: "🇸🇦" },
    { code: "+65", country: "Singapore", flag: "🇸🇬" },
    { code: "+46", country: "Sweden", flag: "🇸🇪" },
    { code: "+41", country: "Switzerland", flag: "🇨🇭" },
    { code: "+90", country: "Turkey", flag: "🇹🇷" },
    { code: "+66", country: "Thailand", flag: "🇹🇭" },
    { code: "+27", country: "South Africa", flag: "🇿🇦" },
    { code: "+44", country: "United Kingdom", flag: "🇬🇧" },
    { code: "+1", country: "United States", flag: "🇺🇸" },
    { code: "+971", country: "UAE", flag: "🇦🇪" },
    { code: "+380", country: "Ukraine", flag: "🇺🇦" },
    { code: "+34", country: "Spain", flag: "🇪🇸" },
    { code: "+91", country: "India", flag: "🇮🇳" },
    { code: "+84", country: "Vietnam", flag: "🇻🇳" },
    { code: "+58", country: "Venezuela", flag: "🇻🇪" },
    { code: "+52", country: "Mexico", flag: "🇲🇽" },
  ];

  useEffect(() => {
    setIsVisible(true);
    
    // If no plan selected and no location state, redirect to signup first
    if (!selectedPlan || selectedPlan === 'Guest' && !location.state?.selectedTier) {
      // Allow Guest plan but show a message or redirect if needed
    }
  }, [selectedPlan, location.state]);

  // Handle keyboard navigation for country dropdown
  useEffect(() => {
    const handleKeyPress = (e: KeyboardEvent) => {
      if (isDropdownOpen) {
        const key = e.key.toLowerCase();
        const matchingCountry = countryCodes.find((country) =>
          country.country.toLowerCase().startsWith(key)
        );

        if (matchingCountry) {
          const countryElement = document.querySelector(
            `[data-country="${matchingCountry.country}"]`
          ) as HTMLElement;
          if (countryElement) {
            countryElement.scrollIntoView({ block: "nearest" });
            countryElement.focus();
          }
        }
      }
    };

    if (isDropdownOpen) {
      document.addEventListener("keydown", handleKeyPress);
    }

    return () => {
      document.removeEventListener("keydown", handleKeyPress);
    };
  }, [isDropdownOpen]);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
    if (error) setError(null);
  };

  const handleCountrySelect = (country: CountryCode) => {
    setSelectedCountry(country);
    setIsDropdownOpen(false);
    if (error) setError(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);
    
    try {
      const fullPhoneNumber = `${selectedCountry.code}${formData.mobileNumber}`;
      
      // Send OTP for signup
      const otpResponse = await authAPI.sendOTP(fullPhoneNumber);
      
      if (otpResponse.status) {
        // Navigate to OTP verification with all signup data
        navigate('/otp-verification', {
          state: {
            phoneNumber: fullPhoneNumber,
            requestType: 'signup',
            signupData: {
              fname: formData.firstName,
              lname: formData.lastName,
              email: formData.email,
              mobile_number: fullPhoneNumber,
              plan: selectedPlan,
              addon_services: selectedAddOnsData.map((addon: any) => addon.id).join(',')
            },
            // Save form data to restore on back
            returnState: {
              formData: formData,
              countryCode: selectedCountry,
              selectedTier: selectedTierData,
              selectedAddOns: selectedAddOnsData
            }
          }
        });
      } else {
        setError(otpResponse.error || t("auth.signupSecond.errors.otpFailed", "Failed to send OTP. Please try again."));
      }
    } catch (error) {
      console.error('Signup error:', error);
      setError(error instanceof Error ? error.message : t("auth.signupSecond.errors.unexpected", "An unexpected error occurred. Please try again."));
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60" style={{ animationDelay: '0s', animationDuration: '3s' }} />
        <div className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40" style={{ animationDelay: '1s' }} />
        <div className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30" style={{ animationDelay: '2s' }} />
        <div className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50" style={{ animationDelay: '1.5s', animationDuration: '4s' }} />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0 py-12">
        {/* Hero Title */}
        <div className={`text-center mb-12 transform transition-all duration-1000 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t("auth.signupSecond.title", "COMPLETE SIGNUP")}
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light">
            {t("auth.signupSecond.subtitle", "You're one step away from joining the community")}
          </p>
        </div>

        {/* Signup Card */}
        <div className={`w-full max-w-lg transform transition-all duration-1000 delay-300 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            
            {/* Selected Plan Info */}
            <div className="mb-8 text-center">
              <div className="inline-flex items-center bg-orange-500/20 border border-orange-400/30 rounded-xl px-6 py-3 mb-4">
                <div className="text-orange-400 text-sm font-semibold uppercase tracking-wider">
                  {t("auth.signupSecond.selectedPlan", "Selected Plan")}: {selectedPlan}
                </div>
              </div>
              <div className="text-white/80 text-lg font-medium">
                {selectedPrice}/{t("auth.signupSecond.month", "month")}
              </div>
              <Link 
                to="/signup" 
                className="text-orange-400 hover:text-orange-300 text-sm font-medium transition-colors mt-2 inline-block"
              >
                {t("auth.signupSecond.changePlan", "Change plan")} →
              </Link>
            </div>
            
            {/* Error Message */}
            {error && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {error}
              </div>
            )}
            
            <form onSubmit={handleSubmit} className="space-y-6">
              {/* Name Fields Row */}
              <div className="grid grid-cols-2 gap-4">
                {/* First Name */}
                <div className="space-y-2">
                  <label className="text-white/80 text-sm font-medium uppercase tracking-wider">{t("auth.signupSecond.firstName", "First Name")}</label>
                  <input
                    type="text"
                    name="firstName"
                    placeholder={t("auth.signupSecond.firstNamePlaceholder", "John")}
                    value={formData.firstName}
                    onChange={handleInputChange}
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                    disabled={isLoading}
                  />
                </div>

                {/* Last Name */}
                <div className="space-y-2">
                  <label className="text-white/80 text-sm font-medium uppercase tracking-wider">{t("auth.signupSecond.lastName", "Last Name")}</label>
                  <input
                    type="text"
                    name="lastName"
                    placeholder={t("auth.signupSecond.lastNamePlaceholder", "Doe")}
                    value={formData.lastName}
                    onChange={handleInputChange}
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                    disabled={isLoading}
                  />
                </div>
              </div>

              {/* Email */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">{t("auth.signupSecond.emailAddress", "Email Address")}</label>
                <input
                  type="email"
                  name="email"
                  placeholder={t("auth.signupSecond.emailPlaceholder", "john.doe@example.com")}
                  value={formData.email}
                  onChange={handleInputChange}
                  className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                  required
                  disabled={isLoading}
                />
              </div>

              {/* Mobile Number */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">{t('signupSecond.mobileNumber')}</label>
                <div className="flex relative">
                  {/* Country Code Dropdown */}
                  <div className="relative">
                    <button
                      type="button"
                      onClick={() => setIsDropdownOpen(!isDropdownOpen)}
                      disabled={isLoading}
                      className="flex items-center px-4 py-4 bg-white/10 border border-white/20 border-r-0 rounded-l-xl hover:bg-white/20 focus:outline-none focus:ring-2 focus:ring-orange-400 backdrop-blur-sm transition-all disabled:opacity-50"
                    >
                      <span className="text-lg mr-2">{selectedCountry.flag}</span>
                      <span className="text-sm text-white/80 mr-1">{selectedCountry.code}</span>
                      <svg className="w-4 h-4 text-white/60" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                      </svg>
                    </button>

                    {/* Dropdown Menu */}
                    {isDropdownOpen && (
                      <div className="absolute z-20 top-full left-0 w-64 bg-black border border-white/20 rounded-xl shadow-2xl max-h-60 overflow-y-auto">
                        {countryCodes.map((country, index) => (
                          <button
                            key={index}
                            type="button"
                            onClick={() => handleCountrySelect(country)}
                            data-country={country.country}
                            className="w-full flex items-center px-4 py-3 text-left hover:bg-white/20 focus:outline-none focus:bg-white/20 transition-colors"
                          >
                            <span className="text-lg mr-3">{country.flag}</span>
                            <span className="text-sm text-white/80 mr-2">{country.code}</span>
                            <span className="text-sm text-white truncate">{country.country}</span>
                          </button>
                        ))}
                      </div>
                    )}
                  </div>

                  <input
                    type="tel"
                    name="mobileNumber"
                    placeholder={t('signupSecond.mobileNumberPlaceholder')}
                    value={formData.mobileNumber}
                    onChange={handleInputChange}
                    className="flex-1 px-4 py-4 bg-white/10 border border-white/20 rounded-r-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                    disabled={isLoading}
                  />
                </div>

                {/* Click outside to close dropdown */}
                {isDropdownOpen && (
                  <div
                    className="fixed inset-0 z-10"
                    onClick={() => setIsDropdownOpen(false)}
                  />
                )}
              </div>

              {/* Terms and Privacy */}
              <div className="bg-white/5 rounded-xl p-4 border border-white/10">
                <div className="text-xs text-white/70 space-y-2">
                  <p>
                    {t('signupSecond.termsAgreement')}{' '}
                    <a href="#" className="text-orange-400 hover:text-orange-300 transition-colors">{t('signupSecond.termsOfService')}</a> {t('common.and')}{' '}
                    <a href="#" className="text-orange-400 hover:text-orange-300 transition-colors">{t('signupSecond.privacyPolicy')}</a>.
                  </p>
                  <p>
                    {t('signupSecond.smsVerification')}
                  </p>
                </div>
              </div>

              {/* Sign up button */}
              <Button
                type="submit"
                variant="fill-white"
                size="lg"
                className="w-full"
                disabled={isLoading}
                arrowRight={!isLoading}
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                    {t('signupSecond.creatingAccount')}
                  </>
                ) : (
                  t('signupSecond.completeSignup')
                )}
              </Button>

              {/* Already have account */}
              <div className="text-center text-white/60 text-sm pt-4">
                {t('signupSecond.alreadyHaveAccount')}{' '}
                <Button
                  type="button"
                  variant="outline-white"
                  size="sm"
                  className="ml-2"
                  onClick={() => navigate('/login')}
                >
                  {t('common.signIn')}
                </Button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
};

export default SignupSecond;