import React, { useState, useEffect, useRef } from 'react';
import { useNavigate, useLocation } from '@remix-run/react';
import { authAPI } from '~/lib/api';
import { Button } from '../ui/button';
import { useLanguage } from '~/hooks/useLanguage';

const UploadProfilePicture: React.FC = () => {
  const navigate = useNavigate();
  const location = useLocation();
  const fileInputRef = useRef<HTMLInputElement>(null);
  const [isVisible, setIsVisible] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [selectedImage, setSelectedImage] = useState<string | null>(null);
  const [imageFile, setImageFile] = useState<File | null>(null);
  const { t } = useLanguage();

  // Get UUID from previous page
  const uuid = location.state?.uuid;
  const userData = location.state?.userData;
  const tierData = location.state?.tierData;
  
  // Get plan data from state or localStorage
  const getPlanData = () => {
    const storedData = typeof window !== 'undefined' ? localStorage.getItem('signup_plan_data') : null;
    if (storedData) {
      const parsed = JSON.parse(storedData);
      return {
        plan: parsed.tierData?.name || parsed.plan || parsed.userData?.plan,
        price: parsed.tierData?.price,
        uuid: parsed.uuid,
        userData: parsed.userData,
        tierData: parsed.tierData
      };
    }
    return {
      plan: tierData?.name || userData?.plan,
      price: tierData?.price,
      uuid: uuid,
      userData: userData,
      tierData: tierData
    };
  };
  
  const planData = getPlanData();
  const selectedPlan = planData.plan || 'STARVING ARTIST';
  const selectedPrice = planData.price ? `${planData.price}` : '1149';

  useEffect(() => {
    // Redirect if no UUID
    if (!uuid) {
      navigate('/signup');
      return;
    }
    
    setIsVisible(true);
  }, [uuid, navigate]);

  const handleImageSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    // Validate file type
    if (!file.type.startsWith('image/')) {
      setError(t('uploadProfile.invalidFileType'));
      return;
    }

    // Validate file size (max 5MB)
    if (file.size > 5 * 1024 * 1024) {
      setError(t('uploadProfile.fileSizeLimit'));
      return;
    }

    setImageFile(file);
    setError(null);

    // Create preview
    const reader = new FileReader();
    reader.onload = (e) => {
      setSelectedImage(e.target?.result as string);
    };
    reader.readAsDataURL(file);
  };

  const handleUpload = async () => {
    if (!imageFile || !uuid) return;

    setIsLoading(true);
    setError(null);

    try {
      const response = await authAPI.saveProfilePicture(uuid, imageFile);
      
      if (response.status) {
        // Navigate to add credit card page
        navigate('/add-credit-card', {
          state: {
            uuid,
            userData,
            selectedPlan,
            selectedPrice
          }
        });
      } else {
        setError(response.error || t('uploadProfile.uploadFailed'));
      }
    } catch (err) {
      console.error('Profile picture upload error:', err);
      setError(err instanceof Error ? err.message : t('uploadProfile.unexpectedError'));
    } finally {
      setIsLoading(false);
    }
  };

  const handleSkip = () => {
    // Navigate to add credit card page without profile picture
    navigate('/add-credit-card', {
      state: {
        uuid,
        userData,
        selectedPlan,
        selectedPrice
      }
    });
  };

  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60" style={{ animationDelay: '0s', animationDuration: '3s' }} />
        <div className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40" style={{ animationDelay: '1s' }} />
        <div className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30" style={{ animationDelay: '2s' }} />
        <div className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50" style={{ animationDelay: '1.5s', animationDuration: '4s' }} />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0">
        {/* Hero Title */}
        <div className={`text-center mb-12 transform transition-all duration-1000 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t('uploadProfile.title')}
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light">
            {t('uploadProfile.subtitle')}
          </p>
        </div>

        {/* Upload Card */}
        <div className={`w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            <h2 className="text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider">
              {t('uploadProfile.profilePicture')}
            </h2>

            {/* Error Message */}
            {error && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {error}
              </div>
            )}

            {/* Image Preview */}
            <div className="mb-8 text-center">
              <div className="relative inline-block">
                <div className="w-32 h-32 mx-auto rounded-full overflow-hidden border-4 border-white/20 bg-white/10 flex items-center justify-center">
                  {selectedImage ? (
                    <img
                      src={selectedImage}
                      alt={t('uploadProfile.profilePreview')}
                      className="w-full h-full object-cover"
                    />
                  ) : (
                    <svg className="w-16 h-16 text-white/40" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                    </svg>
                  )}
                </div>
                
                {selectedImage && (
                  <button
                    onClick={() => {
                      setSelectedImage(null);
                      setImageFile(null);
                      if (fileInputRef.current) {
                        fileInputRef.current.value = '';
                      }
                    }}
                    className="absolute -top-2 -right-2 bg-red-500 hover:bg-red-600 text-white rounded-full p-2 transition-colors"
                  >
                    <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                    </svg>
                  </button>
                )}
              </div>
            </div>

            {/* Upload Section */}
            <div className="space-y-6">
              {/* File Input (Hidden) */}
              <input
                ref={fileInputRef}
                type="file"
                accept="image/*"
                onChange={handleImageSelect}
                className="hidden"
                disabled={isLoading}
              />

              {/* Select Image Button */}
              {!selectedImage && (
                <Button
                  type="button"
                  variant="outline-white"
                  size="lg"
                  className="w-full"
                  onClick={() => fileInputRef.current?.click()}
                  disabled={isLoading}
                >
                  <svg className="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
                  </svg>
                  {t('uploadProfile.chooseImage')}
                </Button>
              )}

              {/* Upload Button */}
              {selectedImage && (
                <Button
                  type="button"
                  variant="fill-white"
                  size="lg"
                  className="w-full"
                  onClick={handleUpload}
                  disabled={isLoading}
                  arrowRight={!isLoading}
                >
                  {isLoading ? (
                    <>
                      <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                      {t('uploadProfile.uploading')}
                    </>
                  ) : (
                    t('uploadProfile.uploadAndContinue')
                  )}
                </Button>
              )}

              {/* Skip Button */}
              <Button
                type="button"
                variant="outline-white"
                size="md"
                className="w-full"
                onClick={handleSkip}
                disabled={isLoading}
              >
                {t('uploadProfile.skipForNow')}
              </Button>

              {/* File Requirements */}
              <div className="text-center text-white/60 text-xs space-y-1">
                <p>{t('uploadProfile.acceptedFormats')}</p>
                <p>{t('uploadProfile.maxFileSize')}</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default UploadProfilePicture;