import { useState } from "react";
import { Button } from "~/components/ui/button";
import { SectionHeader } from "~/components/shared/SectionHeader";
import { PersonCard } from "./components/PersonCard";
import { PeopleToolbar } from "./components/PeopleToolbar";

export interface Person {
  id: string;
  photo: string;
  name: string;
  surname: string;
  role: string;
  location: string;
  href: string;
}

export interface FilterOptions {
  locations: string[];
  roles: string[];
}

const peopleData: Person[] = [
  {
    id: "1",
    photo: "/images/home/people/person-1.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "New York",
    href: "/people/person-1",
  },
  {
    id: "2",
    photo: "/images/home/people/person-2.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Bali",
    href: "/people/person-2",
  },
  {
    id: "3",
    photo: "/images/home/people/person-1.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Bangkok",
    href: "/people/person-3",
  },
  {
    id: "4",
    photo: "/images/home/people/person-2.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Mexico",
    href: "/people/person-4",
  },
  {
    id: "5",
    photo: "/images/home/people/person-1.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Costa Rica",
    href: "/people/person-5",
  },
  {
    id: "6",
    photo: "/images/home/people/person-2.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Mumbai",
    href: "/people/person-6",
  },
  {
    id: "7",
    photo: "/images/home/people/person-1.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "New York",
    href: "/people/person-7",
  },
  {
    id: "8",
    photo: "/images/home/people/person-2.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Bali",
    href: "/people/person-8",
  },
];

const filterOptions: FilterOptions = {
  locations: ["New York", "Bali", "Bangkok", "Mexico", "Costa Rica", "Mumbai"],
  roles: [
    "Photographer",
    "Videographer",
    "Music Producer",
    "Creative Director",
    "Sound Engineer",
  ],
};

export function PeopleSection() {
  const [filters, setFilters] = useState({
    location: "",
    role: "",
  });

  const filteredPeople = peopleData.filter((person) => {
    const locationMatch =
      !filters.location || person.location === filters.location;
    const roleMatch = !filters.role || person.role === filters.role;
    return locationMatch && roleMatch;
  });

  const handleFilterChange = (newFilters: typeof filters) => {
    setFilters(newFilters);
  };

  return (
    <section className="relative w-full bg-white">
      <div className="max-w-[1400px] mx-auto px-4 lg:px-14 py-8 lg:py-24">
        {/* Header */}
        <div className="mb-6 lg:mb-16">
          {/* Mobile Header Layout */}
          <div className="lg:hidden">
            <SectionHeader
              layout="vertical"
              eyebrow="OUR TEAM"
              title="THE PEOPLE TO HELP YOU CREATE"
              className="flex-1 max-w-[991px] [&_span]:text-black/60 [&_h2]:text-black [&_h2]:text-[24px] [&_h2]:tracking-[-0.02em] mb-4"
            />
            <Button
              variant="outline-black"
              size="md"
              className="border-black/30 hover:border-black/50 uppercase tracking-wider text-button-small !text-black"
              arrowRight
            >
              FIND YOUR CREATIVE SUPPORT
            </Button>
          </div>

          {/* Desktop Header Layout */}
          <div className="hidden lg:flex lg:items-end lg:justify-between gap-4">
            <SectionHeader
              layout="vertical"
              eyebrow="OUR TEAM"
              title="THE PEOPLE TO HELP YOU CREATE"
              className="flex-1 max-w-[991px] [&_span]:text-black/60 [&_h2]:text-black"
            />

            {/* CTA Button */}
            <div className="flex items-center">
              <Button
                variant="outline-black"
                size="lg"
                className="border-black/30 hover:border-black/50 uppercase tracking-wider font-medium !text-black"
                arrowRight
              >
                FIND YOUR CREATIVE SUPPORT
              </Button>
            </div>
          </div>
        </div>

        {/* Toolbar - Hidden on mobile */}
        <div className="hidden lg:block">
          <PeopleToolbar
            filters={filters}
            filterOptions={filterOptions}
            onChange={handleFilterChange}
          />
        </div>

        {/* People Grid */}
        <div className="grid grid-cols-2 lg:grid-cols-4 gap-3 lg:gap-6">
          {filteredPeople.map((person) => (
            <PersonCard key={person.id} {...person} />
          ))}
        </div>
      </div>
    </section>
  );
}
