import { ReactNode, useRef } from "react";
import { ArrowLeft, ArrowRight } from "lucide-react";

interface CarouselProps {
  children: ReactNode;
  showArrows?: boolean;
  className?: string;
}

export function Carousel({
  children,
  showArrows = true,
  className = "",
}: CarouselProps) {
  const scrollerRef = useRef<HTMLDivElement>(null);

  function scrollByCards(dir: "prev" | "next") {
    const el = scrollerRef.current;
    if (!el) return;
    const card = el.querySelector<HTMLElement>("[data-card]");
    const amount = card ? card.offsetWidth + 24 : el.clientWidth * 0.9;
    el.scrollBy({
      left: dir === "next" ? amount : -amount,
      behavior: "smooth",
    });
  }

  return (
    <div className={`relative ${className}`}>
      <div
        ref={scrollerRef}
        className="flex snap-x snap-mandatory gap-6 overflow-x-auto pb-4 [-ms-overflow-style:none] [scrollbar-width:none] [&::-webkit-scrollbar]:hidden max-w-full"
        aria-label="Carousel items"
        style={{ overflowX: 'auto', overscrollBehaviorX: 'contain' }}
      >
        {children}
      </div>

      {showArrows && (
        <>
          {/* Desktop arrows */}
          <div className="hidden lg:flex absolute -top-[120px] right-0 items-center gap-4">
            <button
              className="w-[60px] h-[60px] rounded-full border border-neutral-900 opacity-50 flex items-center justify-center transition-opacity hover:opacity-100 !text-black"
              aria-label="Previous"
              onClick={() => scrollByCards("prev")}
            >
              <ArrowLeft className="h-5 w-5" />
            </button>
            <button
              className="w-[60px] h-[60px] rounded-full border border-neutral-900 flex items-center justify-center transition-opacity hover:opacity-75 !text-black"
              aria-label="Next"
              onClick={() => scrollByCards("next")}
            >
              <ArrowRight className="h-5 w-5" />
            </button>
          </div>

          {/* Mobile arrows */}
          <div className="lg:hidden mt-4 flex justify-end items-center gap-3">
            <button
              className="w-10 h-10 rounded-full border border-neutral-900 opacity-50 flex items-center justify-center transition-opacity hover:opacity-100"
              aria-label="Previous"
              onClick={() => scrollByCards("prev")}
            >
              <ArrowLeft className="h-4 w-4" />
            </button>
            <button
              className="w-10 h-10 rounded-full border border-neutral-900 flex items-center justify-center transition-opacity hover:opacity-75"
              aria-label="Next"
              onClick={() => scrollByCards("next")}
            >
              <ArrowRight className="h-4 w-4" />
            </button>
          </div>
        </>
      )}
    </div>
  );
}
