import { Link } from "@remix-run/react";
import type { Event } from "../EventsGrid";

interface EventCardProps extends Event {
  eventData?: any; // Full event data to pass via state
}

export function EventCard({ image, date, time, title, href, id, eventData }: EventCardProps) {
  // Prepare event object for state - convert to API format
  const event = eventData || {
    event_id: parseInt(id),
    event_title: title,
    event_date: `${date.month}-${date.day}`,
    event_time: time,
    thumbnail: image,
    picture: image,
  };

  // Extract location and interested/going counts from eventData
  const location = eventData?.address || "Test address";
  const interestedCount = eventData?.interested_count || 1;
  const goingCount = eventData?.going_count || 0;

  return (
    <Link 
      to={href} 
      state={{ event }} 
      className="group cursor-pointer bg-white border border-gray-200 overflow-hidden hover:shadow-lg transition-all duration-300 block"
    >
      {/* Event Image */}
      <div className="relative overflow-hidden aspect-[426.67/240]">
        <img
          src={image}
          alt={title}
          className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
          loading="lazy"
        />
      </div>

      {/* Event Details */}
      <div className="flex items-start gap-4 p-4 bg-white">
        {/* Date Box */}
        <div className="text-center min-w-[48px] flex-shrink-0">
          <div className="text-[24px] font-semibold leading-tight" style={{ color: '#000000' }}>
            {date.day}
          </div>
          <div className="text-[14px] font-normal uppercase" style={{ color: '#000000' }}>
            {date.month}
          </div>
        </div>

        {/* Event Info */}
        <div className="flex-1 min-w-0">
          <h3 className="text-body-2 font-normal mb-1 line-clamp-2" style={{ color: '#000000' }}>
            {title}
          </h3>
          <p className="text-body-3 mb-1" style={{ color: '#000000' }}>
            {time}
          </p>
          <p className="text-body-3 line-clamp-1" style={{ color: '#000000' }}>
            {location}
          </p>
          {(interestedCount > 0 || goingCount > 0) && (
            <p className="text-body-3 mt-2" style={{ color: '#000000' }}>
              {interestedCount + goingCount} interested • {goingCount} going
            </p>
          )}
        </div>
      </div>
    </Link>
  );
}