import { cn } from "~/lib/utils";

export interface ImageGridItem {
  id: string;
  image: string;
  title: string;
  onClick?: () => void;
}

interface ImageGridProps {
  items: ImageGridItem[];
  className?: string;
  variant?: "single" | "double" | "triple";
  showBorders?: boolean;
}

export function ImageGrid({
  items,
  className,
  variant = "triple",
  showBorders = true,
}: ImageGridProps) {
  // Responsive grid columns based on variant and screen size
  const getGridCols = () => {
    switch (variant) {
      case "single":
        return "grid-cols-1";
      case "double":
        return "grid-cols-1 sm:grid-cols-2";
      case "triple":
      default:
        return "grid-cols-1 sm:grid-cols-2 lg:grid-cols-3";
    }
  };

  return (
    <div
      className={cn(
        showBorders && "border-y border-[rgba(18,18,18,0.2)]",
        className
      )}
    >
      <div className={cn("grid gap-4 sm:gap-6 lg:gap-8 py-8 lg:py-12", getGridCols())}>
        {items.map((item, index) => (
          <div
            key={item.id}
            onClick={item.onClick}
            className={cn(
              "group flex flex-col items-center",
              item.onClick && "cursor-pointer"
            )}
          >
            <div className="w-full aspect-[4/3] sm:aspect-square overflow-hidden">
              <img
                src={item.image}
                alt={item.title}
                className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
              />
            </div>

            <div className="p-3 sm:p-4 lg:p-6">
              <h3 className="text-xs sm:text-sm lg:text-lg 2xl:text-xl font-medium uppercase tracking-[0.02em] text-[#121212] font-clash-grotesk text-center">
                {item.title}
              </h3>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}
