import { useState, useEffect, useCallback } from "react";
import { cn } from "~/lib/utils";
import { Button } from "~/components/ui/button";

export interface PhotoGalleryImage {
  id: string;
  src: string;
  alt: string;
  title?: string;
  description?: string;
}

interface PhotoGalleryProps {
  images: PhotoGalleryImage[];
  className?: string;
  showAllButtonText?: string;
}

export function PhotoGallery({
  images,
  className,
  showAllButtonText = "SHOW ALL PHOTOS",
}: PhotoGalleryProps) {
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);
  const visibleImages = images.slice(0, 3);

  const openModal = (index: number = 0) => {
    setCurrentImageIndex(index);
    setIsModalOpen(true);
  };

  const closeModal = () => {
    setIsModalOpen(false);
  };

  const goToPrevious = useCallback(() => {
    setCurrentImageIndex((prev) => (prev - 1 + images.length) % images.length);
  }, [images.length]);

  const goToNext = useCallback(() => {
    setCurrentImageIndex((prev) => (prev + 1) % images.length);
  }, [images.length]);

  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      if (!isModalOpen) return;

      switch (e.key) {
        case "Escape":
          closeModal();
          break;
        case "ArrowLeft":
          goToPrevious();
          break;
        case "ArrowRight":
          goToNext();
          break;
      }
    };

    window.addEventListener("keydown", handleKeyDown);
    return () => window.removeEventListener("keydown", handleKeyDown);
  }, [isModalOpen, goToPrevious, goToNext]);

  useEffect(() => {
    if (isModalOpen) {
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "";
    }

    return () => {
      document.body.style.overflow = "";
    };
  }, [isModalOpen]);

  return (
    <>
      <div className={cn("grid grid-cols-3 gap-2", className)}>
        {/* Left column - Main large image (2/3 width) */}
        <div className="col-span-2 relative group">
          <img
            src={visibleImages[0]?.src}
            alt={visibleImages[0]?.alt}
            className="w-full h-[400px] object-cover cursor-pointer transition-transform duration-300 group-hover:scale-[1.01]"
            onClick={() => openModal(0)}
          />
        </div>

        {/* Right column - 2 images stacked */}
        {visibleImages.length > 1 && (
          <div className="space-y-2">
            <div className="relative group">
              <img
                src={visibleImages[1]?.src}
                alt={visibleImages[1]?.alt}
                className="w-full h-[196px] object-cover cursor-pointer transition-transform duration-300 group-hover:scale-[1.01]"
                onClick={() => openModal(1)}
              />
            </div>
            {visibleImages[2] && (
              <div className="relative group">
                <img
                  src={visibleImages[2].src}
                  alt={visibleImages[2].alt}
                  className="w-full h-[196px] object-cover cursor-pointer transition-transform duration-300 group-hover:scale-[1.01]"
                  onClick={() => openModal(2)}
                />
                {images.length > 3 && (
                  <Button
                    onClick={() => openModal(0)}
                    className="absolute bottom-4 right-4 shadow-sm"
                    variant="fill-white"
                    size="sm"
                  >
                    {showAllButtonText}
                  </Button>
                )}
              </div>
            )}
          </div>
        )}
      </div>

      {/* Modal/Lightbox */}
      {isModalOpen && (
        <div className="fixed inset-0 z-50 bg-black/95 flex items-center justify-center">
          {/* Close button */}
          <button
            onClick={closeModal}
            className="absolute top-4 right-4 text-white/80 hover:text-white p-2 transition-colors z-10"
            aria-label="Close"
          >
            <svg
              className="w-8 h-8"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M6 18L18 6M6 6l12 12"
              />
            </svg>
          </button>

          {/* Previous button */}
          <button
            onClick={goToPrevious}
            className="absolute left-4 top-1/2 -translate-y-1/2 text-white/80 hover:text-white p-2 transition-colors"
            aria-label="Previous image"
          >
            <svg
              className="w-8 h-8"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M15 19l-7-7 7-7"
              />
            </svg>
          </button>

          {/* Next button */}
          <button
            onClick={goToNext}
            className="absolute right-4 top-1/2 -translate-y-1/2 text-white/80 hover:text-white p-2 transition-colors"
            aria-label="Next image"
          >
            <svg
              className="w-8 h-8"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M9 5l7 7-7 7"
              />
            </svg>
          </button>

          {/* Main image display */}
          <div className="relative max-w-[90vw] max-h-[90vh] flex flex-col items-center">
            <img
              src={images[currentImageIndex]?.src}
              alt={images[currentImageIndex]?.alt}
              className="max-w-full max-h-[70vh] object-contain"
            />

            {/* Image info */}
            {(images[currentImageIndex]?.title ||
              images[currentImageIndex]?.description) && (
              <div className="mt-4 text-center text-white">
                {images[currentImageIndex]?.title && (
                  <h3 className="text-xl font-medium mb-2">
                    {images[currentImageIndex].title}
                  </h3>
                )}
                {images[currentImageIndex]?.description && (
                  <p className="text-white/80">
                    {images[currentImageIndex].description}
                  </p>
                )}
              </div>
            )}

            {/* Image counter */}
            <div className="mt-4 text-white/60 text-sm">
              {currentImageIndex + 1} / {images.length}
            </div>

            {/* Thumbnail strip */}
            <div className="mt-6 flex gap-2 overflow-x-auto max-w-full px-4">
              {images.map((image, index) => (
                <button
                  key={image.id}
                  onClick={() => setCurrentImageIndex(index)}
                  className={cn(
                    "flex-shrink-0 transition-all duration-200",
                    index === currentImageIndex
                      ? "ring-2 ring-white opacity-100"
                      : "opacity-60 hover:opacity-80"
                  )}
                >
                  <img
                    src={image.src}
                    alt={image.alt}
                    className="w-20 h-20 object-cover"
                  />
                </button>
              ))}
            </div>
          </div>
        </div>
      )}
    </>
  );
}
