import { Slot } from "@radix-ui/react-slot";
import { cva, type VariantProps } from "class-variance-authority";
import * as React from "react";

import { cn } from "~/lib/utils";

const ArrowRightIcon = ({ size = 20 }: { size?: number }) => (
  <svg
    width={size}
    height={size}
    fill="none"
    stroke="currentColor"
    viewBox="0 0 24 24"
  >
    <path
      strokeLinecap="round"
      strokeLinejoin="round"
      strokeWidth={2}
      d="M13 7l5 5m0 0l-5 5m5-5H6"
    />
  </svg>
);

const buttonVariants = cva(
  "inline-flex items-center justify-center transition-all disabled:pointer-events-none disabled:opacity-50 box-border cursor-pointer uppercase whitespace-nowrap",
  {
    variants: {
      variant: {
        // Fill variants
        "fill-white": "bg-white !text-black border border-white",
        "fill-black": "bg-black !text-white border border-black",
        // Outline variants
        "outline-white": "border !border-white/30 bg-transparent !text-white",
        "outline-black": "border !border-black bg-transparent text-black",
      },
      size: {
        // Small: 32px height, 14px font
        sm: "h-8 text-button-small",
        // Medium: 48px height, 16px font
        md: "h-12 text-button-medium",
        // Large: 60px height, 18px font
        lg: "h-[60px] text-button-big",
      },
      buttonType: {
        // Label buttons (with text)
        label: "",
        // Icon square buttons
        "icon-square": "aspect-square p-0",
        // Icon circle buttons
        "icon-circle": "aspect-square p-0 rounded-full",
      },
    },
    compoundVariants: [
      // Small label buttons - padding based on CSS: 12px
      {
        size: "sm",
        buttonType: "label",
        class: "px-3 gap-1.5",
      },
      // Medium label buttons - padding based on CSS: 20px
      {
        size: "md",
        buttonType: "label",
        class: "px-5 gap-2",
      },
      // Large label buttons - padding based on CSS: 24px
      {
        size: "lg",
        buttonType: "label",
        class: "px-6 gap-2",
      },
      // Small icon buttons
      {
        size: "sm",
        buttonType: ["icon-square", "icon-circle"],
        class: "w-8 h-8",
      },
      // Medium icon buttons
      {
        size: "md",
        buttonType: ["icon-square", "icon-circle"],
        class: "w-12 h-12",
      },
      // Large icon buttons
      {
        size: "lg",
        buttonType: ["icon-square", "icon-circle"],
        class: "w-[60px] h-[60px]",
      },
      // All variants hover states - consistent border animation
      {
        variant: "fill-white",
        class: "hover:text-gray-600",
      },
      {
        variant: "fill-black",
        class: "hover:text-gray-300",
      },
      {
        variant: "outline-white",
        class: "hover:text-gray-300",
      },
      {
        variant: "outline-black",
        class: "hover:text-gray-600",
      },
    ],
    defaultVariants: {
      variant: "fill-white",
      size: "md",
      buttonType: "label",
    },
  }
);

interface ButtonProps
  extends React.ComponentProps<"button">,
    VariantProps<typeof buttonVariants> {
  asChild?: boolean;
  arrowRight?: boolean;
  responsive?: boolean;
}

function Button({
  className,
  variant,
  size,
  buttonType,
  asChild = false,
  arrowRight = false,
  responsive = false,
  children,
  ...props
}: ButtonProps) {
  const Comp = asChild ? Slot : "button";

  // If responsive is true and no size is explicitly provided, use md on mobile and lg on desktop
  const computedSize = responsive && !size ? undefined : size;
  const responsiveClasses = responsive && !size 
    ? buttonType === "label" 
      ? "h-12 text-button-medium px-5 gap-2 lg:h-[60px] lg:text-button-big lg:px-6"
      : "w-12 h-12 lg:w-[60px] lg:h-[60px]"
    : "";

  const content = arrowRight ? (
    <>
      {children}
      <ArrowRightIcon size={
        responsive && !size 
          ? 20  // Default to md size (20) for responsive buttons on mobile, will be overridden by lg size on desktop
          : computedSize === "sm" ? 16 : computedSize === "lg" ? 24 : 20
      } />
    </>
  ) : (
    children
  );

  return (
    <Comp
      data-slot="button"
      className={cn(
        buttonVariants({ variant, size: computedSize, buttonType }),
        responsiveClasses,
        className
      )}
      {...props}
    >
      {content}
    </Comp>
  );
}

export { Button, buttonVariants };
