export interface PersonData {
  id: string;
  photo: string;
  name: string;
  surname: string;
  role: string;
  location: string;
  href: string;
}

export interface FilterOptions {
  locations: string[];
  roles: string[];
}

export interface LocationAddresses {
  [key: string]: string;
}

export interface EquipmentItem {
  id: string;
  name: string;
  image: string;
  href: string;
}

export interface StudioPageData {
  meta: {
    title: string;
    description: string;
  };
  hero: {
    studioType: string;
    location: string;
    country: string;
    address: string;
    description: string;
    images: Array<{
      id: string;
      src: string;
      alt: string;
      title: string;
      description: string;
    }>;
  };
  crew: {
    title: string;
    people: PersonData[];
    filterOptions: FilterOptions;
  };
  equipment: {
    title: string;
    items: EquipmentItem[];
  };
  planVisit: {
    address: string;
    hours: string;
    phone: string;
    email: string;
    additionalInfo: string;
  };
}

export const locationAddresses: LocationAddresses = {
  bali: "Jl. Raya Ubud No.88, Ubud, Kecamatan Ubud, Kabupaten Gianyar, Bali 80571, Indonesia",
  bangkok: "123 Sukhumvit Rd, Khlong Toei, Bangkok 10110, Thailand",
  mexico: "Av. Revolución 123, Roma Norte, Ciudad de México, Mexico",
  "costa-rica": "Avenida Central 456, San José, Costa Rica",
  "new-york": "123 Broadway, New York, NY 10001, USA",
  mumbai: "456 Linking Road, Bandra West, Mumbai 400050, India",
};

export const peopleData: PersonData[] = [
  {
    id: "1",
    photo: "/images/home/people/person-1.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "New York",
    href: "/people/person-1",
  },
  {
    id: "2",
    photo: "/images/home/people/person-2.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Bangkok",
    href: "/people/person-2",
  },
  {
    id: "3",
    photo: "/images/home/people/person-3.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Costa Rica",
    href: "/people/person-3",
  },
  {
    id: "4",
    photo: "/images/home/people/person-4.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Bali",
    href: "/people/person-4",
  },
];

export const filterOptions: FilterOptions = {
  locations: ["New York", "Bangkok", "Costa Rica", "Bali"],
  roles: ["Profession", "Creative Director", "Sound Engineer"],
};

export const studioImages = [
  {
    id: "1",
    src: "/images/studios/studio-1.png",
    alt: "Studio main view",
    title: "Main Studio Space",
    description: "Professional setup with state-of-the-art equipment",
  },
  {
    id: "2",
    src: "/images/studios/music-1.png",
    alt: "Studio equipment view",
    title: "Recording Setup",
    description: "High-quality audio and video recording capabilities",
  },
  {
    id: "3",
    src: "/images/studios/photo-1.png",
    alt: "Studio lighting setup",
    title: "Lighting & Backdrop",
    description: "Professional lighting with customizable backdrops",
  },
  {
    id: "4",
    src: "/images/studios/studio-2.png",
    alt: "Studio workspace",
    title: "Creative Workspace",
    description: "Comfortable workspace for editing and post-production",
  },
  {
    id: "5",
    src: "/images/studios/music-2.png",
    alt: "Studio control room",
    title: "Control Room",
    description: "Advanced control room with mixing capabilities",
  },
  {
    id: "6",
    src: "/images/studios/studio-3.png",
    alt: "Studio lounge area",
    title: "Lounge Area",
    description: "Relaxation space for breaks and client meetings",
  },
];

const equipmentItems: EquipmentItem[] = [
  {
    id: "dj-equipment",
    name: "DJ EQUIPMENT",
    image: "/images/equipment/dj-equipment.png",
    href: "/equipment/dj-equipment",
  },
  {
    id: "cameras",
    name: "CAMERAS",
    image: "/images/equipment/cameras.png",
    href: "/equipment/cameras",
  },
  {
    id: "light",
    name: "LIGHT",
    image: "/images/equipment/light.png",
    href: "/equipment/light",
  },
];

const countryMap: Record<string, string> = {
  bali: "INDONESIA",
  bangkok: "THAILAND",
  mexico: "MEXICO",
  "costa-rica": "COSTA RICA",
  "new-york": "USA",
  mumbai: "INDIA",
};

export function getStudioPageData(studioId: string, location: string, studioType: string): StudioPageData {
  const country = countryMap[location] || "UNKNOWN";
  const address = locationAddresses[location] || "Location address";
  
  const studioImagesWithAlt = studioImages.map(image => ({
    ...image,
    alt: `${studioType} ${image.alt.split(' ').slice(1).join(' ')}`
  }));

  const description = `${location.charAt(0).toUpperCase() + location.slice(1)} is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.`;

  return {
    meta: {
      title: `Book ${studioType} Studio | Nomadics`,
      description: `Book your ${studioType.toLowerCase()} studio session at Nomadics`,
    },
    hero: {
      studioType,
      location,
      country,
      address,
      description,
      images: studioImagesWithAlt,
    },
    crew: {
      title: "OUR CREW",
      people: peopleData,
      filterOptions,
    },
    equipment: {
      title: "EQUIPMENT",
      items: equipmentItems,
    },
    planVisit: {
      address: "123 Linking Road, Bandra West, Mumbai 400050, India",
      hours: "7:00 AM - 11:00 PM",
      phone: "+91 22 1234 5678",
      email: "mumbai@nomadics.com",
      additionalInfo: "Near Bandra railway station...",
    },
  };
}

// Legacy exports for backward compatibility
export const planVisitData = {
  address: "123 Linking Road, Bandra West, Mumbai 400050, India",
  hours: "7:00 AM - 11:00 PM",
  phone: "+91 22 1234 5678",
  email: "mumbai@nomadics.com",
  additionalInfo: "Near Bandra railway station...",
};