import { useState, useEffect, useCallback } from 'react';
import { API_BASE_URL, IMAGE_BASE_URL } from '~/lib/api';

type Language = 'en' | 'sp';

interface LanguageData {
  [key: string]: any;
}

const LANGUAGE_STORAGE_KEY = 'nomadic-language';
const DEFAULT_LANGUAGE: Language = 'en';

// Replace hardcoded URLs in translation data with API constants
const replaceUrlsInTranslations = (data: any): any => {
  if (typeof data === 'string') {
    // Replace hardcoded URLs with constants
    let result = data
      .replace(/https:\/\/nomadicstudios\.net:444/g, IMAGE_BASE_URL)
      .replace(/https:\/\/dashboard\.nomadicstudios\.net/g, IMAGE_BASE_URL);
    
    // Also handle cases where the string starts with /images but needs the base URL
    // If it's just a path starting with /images and doesn't already have a protocol
    if (result.startsWith('/images') && !result.includes('http')) {
      result = IMAGE_BASE_URL + result;
    }
    
    return result;
  } else if (Array.isArray(data)) {
    return data.map(replaceUrlsInTranslations);
  } else if (data && typeof data === 'object') {
    const result: any = {};
    for (const [key, value] of Object.entries(data)) {
      result[key] = replaceUrlsInTranslations(value);
    }
    return result;
  }
  return data;
};

export function useLanguage() {
  const [currentLanguage, setCurrentLanguage] = useState<Language>(DEFAULT_LANGUAGE);
  const [translations, setTranslations] = useState<LanguageData>({});
  const [loading, setLoading] = useState(true);

  // Load saved language from localStorage on mount
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const savedLanguage = localStorage.getItem(LANGUAGE_STORAGE_KEY) as Language;
      if (savedLanguage && (savedLanguage === 'en' || savedLanguage === 'sp')) {
        setCurrentLanguage(savedLanguage);
      }
    }
  }, []);

  // Load translations when language changes
  useEffect(() => {
    const loadTranslations = async () => {
      try {
        setLoading(true);
        console.log(`Loading translations for language: ${currentLanguage}`);
        const response = await import(`../config/lang/${currentLanguage}.json`);
        const rawTranslations = response.default || response;
        const processedTranslations = replaceUrlsInTranslations(rawTranslations);
        console.log(`Loaded translations for ${currentLanguage}:`, Object.keys(processedTranslations));
        setTranslations(processedTranslations);
        setLoading(false);
      } catch (error) {
        console.error(`Failed to load translations for ${currentLanguage}:`, error);
        // Fallback to English if current language fails
        if (currentLanguage !== 'en') {
          try {
            const fallback = await import('../config/lang/en.json');
            const rawFallbackTranslations = fallback.default || fallback;
            const processedFallbackTranslations = replaceUrlsInTranslations(rawFallbackTranslations);
            console.log(`Loaded fallback translations:`, Object.keys(processedFallbackTranslations));
            setTranslations(processedFallbackTranslations);
          } catch (fallbackError) {
            console.error('Failed to load fallback English translations:', fallbackError);
          }
        }
        setLoading(false);
      }
    };

    loadTranslations();
  }, [currentLanguage]);

  // Change language function
  const changeLanguage = useCallback((newLanguage: Language) => {
    if (newLanguage === currentLanguage) {
      console.log(`Language already set to ${newLanguage}, skipping change`);
      return; // Avoid unnecessary changes
    }
    
    console.log(`Changing language from ${currentLanguage} to ${newLanguage}`);
    setCurrentLanguage(newLanguage);
    if (typeof window !== 'undefined') {
      localStorage.setItem(LANGUAGE_STORAGE_KEY, newLanguage);
      // Reload the page after changing language
      window.location.reload();
    }
  }, [currentLanguage]);

  // Get translation by key with dot notation support
  const t = useCallback((key: string, defaultValue?: string): string => {
    if (!translations || Object.keys(translations).length === 0) {
      return defaultValue || key;
    }
    
    const keys = key.split('.');
    let value = translations;
    
    for (const k of keys) {
      if (value && typeof value === 'object' && k in value) {
        value = value[k];
      } else {
        return defaultValue || key;
      }
    }
    
    return typeof value === 'string' ? value : defaultValue || key;
  }, [translations]);

  // Get nested object by key
  const getSection = useCallback((key: string): any => {
    if (!translations || Object.keys(translations).length === 0) {
      return {};
    }
    
    const keys = key.split('.');
    let value = translations;
    
    for (const k of keys) {
      if (value && typeof value === 'object' && k in value) {
        value = value[k];
      } else {
        return {};
      }
    }
    
    return value || {};
  }, [translations]);

  return {
    currentLanguage,
    changeLanguage,
    t,
    getSection,
    loading,
    translations
  };
}

export type { Language };