import {
  Links,
  Meta,
  Outlet,
  Scripts,
  ScrollRestoration,
  isRouteErrorResponse,
  useRouteError,
  useLocation,
  useNavigate,
} from "@remix-run/react";
import type { LinksFunction } from "@remix-run/node";
import stylesheet from "./tailwind.css?url";
import { AppHeader } from "./components/AppHeader";
import { AppFooter } from "./components/AppFooter";
import { useEffect } from "react";

export const links: LinksFunction = () => [
  { rel: "stylesheet", href: stylesheet },
  { rel: "icon", href: "/favicon.ico" },
  { rel: "manifest", href: "/manifest.json" },
  { rel: "apple-touch-icon", href: "/logo192.png" },
  {
    rel: "preload",
    href: "/fonts/ClashGrotesk-Variable.woff2",
    as: "font",
    type: "font/woff2",
    crossOrigin: "anonymous",
  },
  {
    rel: "preload",
    href: "/fonts/ClashGrotesk-Bold.woff2",
    as: "font",
    type: "font/woff2",
    crossOrigin: "anonymous",
  },
  {
    rel: "preload",
    href: "/fonts/ClashGrotesk-Semibold.woff2",
    as: "font",
    type: "font/woff2",
    crossOrigin: "anonymous",
  },
];

export function Layout({ children }: { children: React.ReactNode }) {
  return (
    <html lang="en">
      <head>
        <meta charSet="utf-8" />
        <meta name="viewport" content="width=device-width, initial-scale=1" />
        <Meta />
        <Links />
      </head>
      <body className="overflow-x-hidden">
        {children}
        <ScrollRestoration />
        <Scripts />
      </body>
    </html>
  );
}

export default function App() {
  const location = useLocation();
  const navigate = useNavigate();

  // Check token expiration on mount and periodically
  useEffect(() => {
    const checkTokenExpiration = () => {
      if (typeof window === "undefined") return;
      
      const tokenExpiration = localStorage.getItem("tokenExpiration");
      const authToken = localStorage.getItem("authToken") || localStorage.getItem("auth_token") || localStorage.getItem("token");
      
      // Skip check if no token or on public routes
      if (!authToken || location.pathname === "/login" || location.pathname === "/signup" || location.pathname === "/") {
        return;
      }
      
      if (tokenExpiration) {
        const expirationTime = parseInt(tokenExpiration, 10);
        const currentTime = Date.now();
        
        if (currentTime >= expirationTime) {
          // Token expired - logout
          console.log("Token expired, logging out...");
          localStorage.clear();
          sessionStorage.clear();
          navigate("/login");
        }
      }
    };

    // Check immediately
    checkTokenExpiration();
    
    // Check every minute
    const interval = setInterval(checkTokenExpiration, 60000);
    
    return () => clearInterval(interval);
  }, [location.pathname, navigate]);

  // Define which pages should use white theme
  // const whiteThemeRoutes = ["/equipment"];
  // const headerTheme = whiteThemeRoutes.some((route) =>
  //   location.pathname.startsWith(route)
  // )
  //   ? "white"
  //   : "dark";

  // Define routes that need full-screen hero sections (no top padding)
  const fullScreenRoutes = ["/", "/home"];
  const needsTopPadding = !fullScreenRoutes.includes(location.pathname);

  return (
    <div className="min-h-screen bg-black">
      <AppHeader theme="dark" />

      <main className={needsTopPadding ? "pt-16 sm:pt-16 lg:pt-20" : ""}>
        <Outlet />
      </main>

      <AppFooter />
    </div>
  );
}

export function ErrorBoundary() {
  const error = useRouteError();

  if (isRouteErrorResponse(error)) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-4xl font-bold mb-4">
            {error.status} {error.statusText}
          </h1>
          {error.data && <p className="text-gray-600">{error.data}</p>}
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen flex items-center justify-center">
      <div className="text-center">
        <h1 className="text-4xl font-bold mb-4">Oops!</h1>
        <p className="text-gray-600">Something went wrong.</p>
      </div>
    </div>
  );
}
