import type { MetaFunction } from "@remix-run/node";
import { useState, useEffect } from "react";
import { useNavigate, useLocation } from "@remix-run/react";
import { ProtectedRoute } from "~/components/RouteProtection";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { Input } from "~/components/ui/input";
import { Camera, Upload, ChevronDown, Plus, Trash2, X } from "lucide-react";
import { useLanguage } from "~/hooks/useLanguage";
import { API_BASE_URL } from "~/lib/api";
export const meta: MetaFunction = () => {
  return [
    { title: "Account Settings - Nomadics Studios" },
    {
      name: "description",
      content: "Manage your account settings and preferences.",
    },
  ];
};

type TabType = "profile" | "membership" | "points" | "cards" | "security" | "calendar";

interface CountryCode {
  code: string;
  country: string;
  flag: string;
}

export default function AccountSettings() {
  const navigate = useNavigate();
  const location = useLocation();
  const { t } = useLanguage();
  
  // Check for tab query parameter
  const searchParams = new URLSearchParams(location.search);
  const tabParam = searchParams.get('tab') as TabType;
  
  // Get previously active tab from localStorage or use default
  const getInitialTab = (): TabType => {
    if (tabParam && ['profile', 'membership', 'points', 'cards', 'security', 'calendar'].includes(tabParam)) {
      return tabParam;
    }
    
    // Try to get last active tab from localStorage (client-side only)
    if (typeof window !== 'undefined') {
      const savedTab = localStorage.getItem('accountSettingsActiveTab') as TabType;
      if (savedTab && ['profile', 'membership', 'points', 'cards', 'security', 'calendar'].includes(savedTab)) {
        return savedTab;
      }
    }
    
    return "profile";
  };
  
  const [activeTab, setActiveTab] = useState<TabType>(getInitialTab());
  const [formData, setFormData] = useState({
    firstName: "",
    lastName: "", 
    bio: "",
    hourlyRate: "0.00",
    projectRate: "0.00",
    homeCityId: ""
  });
  const [profilePicture, setProfilePicture] = useState<File | null>(null);
  const [profilePictureUrl, setProfilePictureUrl] = useState<string | null>(null);
  const [isUploading, setIsUploading] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [selectedCountry, setSelectedCountry] = useState<CountryCode>({
    code: "+1",
    country: "United States",
    flag: "🇺🇸",
  });
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [showBuyPointsModal, setShowBuyPointsModal] = useState(false);
  const [pointsToBuy, setPointsToBuy] = useState("");
  const [selectedPaymentMethodId, setSelectedPaymentMethodId] = useState("");
  const [isBuyingPoints, setIsBuyingPoints] = useState(false);
  
  // Location states
  const [locations, setLocations] = useState<any[]>([]);
  const [isLoadingLocations, setIsLoadingLocations] = useState(false);
  
  // Points balance states
  const [pointsBalance, setPointsBalance] = useState({
    lastMonthBookingCost: 0,
    currentMonthBookingCost: 0,
    currentBalance: 0,
    points: [] as any[],
    pointsAction: {} as Record<string, string>
  });
  const [isLoadingPoints, setIsLoadingPoints] = useState(false);

  // Payment methods states
  const [paymentMethods, setPaymentMethods] = useState<any[]>([]);
  const [defaultPaymentMethod, setDefaultPaymentMethod] = useState<any>(null);
  const [isLoadingPaymentMethods, setIsLoadingPaymentMethods] = useState(false);

  // Membership states
  const [membershipData, setMembershipData] = useState({
    plan: {
      name: 'Guest',
      price: 0,
      points: 0,
      pm_type: '',
      pm_last_four: ''
    },
    transactions: [] as any[]
  });
  const [isLoadingMembership, setIsLoadingMembership] = useState(false);

  // Security states
  const [userProfile, setUserProfile] = useState({
    email: '',
    phoneNumber: '',
    countryCode: '+91',
    profilePicture: ''
  });
  const [originalProfile, setOriginalProfile] = useState({
    email: '',
    phoneNumber: '',
    countryCode: '+91'
  });
  const [isLoadingProfile, setIsLoadingProfile] = useState(true);

  
  // Legacy OTP states (keeping for backward compatibility)
  const [showOtpModal, setShowOtpModal] = useState(false);
  const [otpData, setOtpData] = useState({
    emailOtp: '',
    mobileOtp: ''
  });
  const [isSendingOtp, setIsSendingOtp] = useState(false);
  const [isVerifyingOtp, setIsVerifyingOtp] = useState(false);
  const [otpSent, setOtpSent] = useState(false);
  const [resendTimer, setResendTimer] = useState(0);
  const [changedFields, setChangedFields] = useState<('email' | 'phone')[]>([]);
  
  // Calendar states
  const [currentDate, setCurrentDate] = useState(new Date(2025, 10, 1)); // November 2025
  const [calendarView, setCalendarView] = useState<'schedule' | 'month' | 'week' | 'day'>('schedule');
  const [availabilityType, setAvailabilityType] = useState<'daily' | 'recurring'>('daily');
  const [availabilityRows, setAvailabilityRows] = useState([
    { id: 1, date: '', startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: '0.00', hourlyRate: '0.00' }
  ]);
  const [recurringRows, setRecurringRows] = useState([
    { id: 0, day: 'Mon', dayIndex: 0, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: '0.00', hourlyRate: '0.00' }] },
    { id: 1, day: 'Tue', dayIndex: 1, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: '0.00', hourlyRate: '0.00' }] },
    { id: 2, day: 'Wed', dayIndex: 2, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: '0.00', hourlyRate: '0.00' }] },
    { id: 3, day: 'Thu', dayIndex: 3, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: '0.00', hourlyRate: '0.00' }] },
    { id: 4, day: 'Fri', dayIndex: 4, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: '0.00', hourlyRate: '0.00' }] },
    { id: 5, day: 'Sat', dayIndex: 5, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: '0.00', hourlyRate: '0.00' }] },
    { id: 6, day: 'Sun', dayIndex: 6, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: '0.00', hourlyRate: '0.00' }] },
  ]);
  
  // Calendar API states
  const [calendarSettings, setCalendarSettings] = useState({
    locations: {} as Record<string, string>,
    refrence_id: 0,
    min_booking: { min_days: '0', min_hours: '02', min_minutes: '00' },
    max_booking: { max_days: '05', max_hours: '00', max_minutes: '00' },
    schedule_location: ''
  });
  const [calendarEvents, setCalendarEvents] = useState<any[]>([]);
  const [isLoadingCalendar, setIsLoadingCalendar] = useState(false);
  const [isSavingCalendar, setIsSavingCalendar] = useState(false);

  const tabs = [
    { id: "profile" as TabType, label: t("accountSettings.tabs.profile") },
    { id: "membership" as TabType, label: "Membership" },
    { id: "points" as TabType, label: t("accountSettings.tabs.points") },
    { id: "cards" as TabType, label: t("accountSettings.tabs.paymentMethods") },
    { id: "security" as TabType, label: "Security" },
    { id: "calendar" as TabType, label: "Calendar" },
  ];

  const countryCodes: CountryCode[] = [
    { code: "+61", country: "Australia", flag: "🇦🇺" },
    { code: "+43", country: "Austria", flag: "🇦🇹" },
    { code: "+54", country: "Argentina", flag: "🇦🇷" },
    { code: "+32", country: "Belgium", flag: "🇧🇪" },
    { code: "+359", country: "Bulgaria", flag: "🇧🇬" },
    { code: "+55", country: "Brazil", flag: "🇧🇷" },
    { code: "+56", country: "Chile", flag: "🇨🇱" },
    { code: "+86", country: "China", flag: "🇨🇳" },
    { code: "+420", country: "Czech Republic", flag: "🇨🇿" },
    { code: "+385", country: "Croatia", flag: "🇭🇷" },
    { code: "+20", country: "Egypt", flag: "🇪🇬" },
    { code: "+33", country: "France", flag: "🇫🇷" },
    { code: "+358", country: "Finland", flag: "🇫🇮" },
    { code: "+30", country: "Greece", flag: "🇬🇷" },
    { code: "+49", country: "Germany", flag: "🇩🇪" },
    { code: "+353", country: "Ireland", flag: "🇮🇪" },
    { code: "+91", country: "India", flag: "🇮🇳" },
    { code: "+972", country: "Israel", flag: "🇮🇱" },
    { code: "+62", country: "Indonesia", flag: "🇮🇩" },
    { code: "+81", country: "Japan", flag: "🇯🇵" },
    { code: "+254", country: "Kenya", flag: "🇰🇪" },
    { code: "+82", country: "South Korea", flag: "🇰🇷" },
    { code: "+36", country: "Hungary", flag: "🇭🇺" },
    { code: "+39", country: "Italy", flag: "🇮🇹" },
    { code: "+234", country: "Nigeria", flag: "🇳🇬" },
    { code: "+31", country: "Netherlands", flag: "🇳🇱" },
    { code: "+64", country: "New Zealand", flag: "🇳🇿" },
    { code: "+351", country: "Portugal", flag: "🇵🇹" },
    { code: "+48", country: "Poland", flag: "🇵🇱" },
    { code: "+51", country: "Peru", flag: "🇵🇪" },
    { code: "+63", country: "Philippines", flag: "🇵🇭" },
    { code: "+966", country: "Saudi Arabia", flag: "🇸🇦" },
    { code: "+65", country: "Singapore", flag: "🇸🇬" },
    { code: "+46", country: "Sweden", flag: "🇸🇪" },
    { code: "+41", country: "Switzerland", flag: "🇨🇭" },
    { code: "+90", country: "Turkey", flag: "🇹🇷" },
    { code: "+66", country: "Thailand", flag: "🇹🇭" },
    { code: "+27", country: "South Africa", flag: "🇿🇦" },
    { code: "+44", country: "United Kingdom", flag: "🇬🇧" },
    { code: "+1", country: "United States", flag: "🇺🇸" },
    { code: "+971", country: "UAE", flag: "🇦🇪" },
    { code: "+380", country: "Ukraine", flag: "🇺🇦" },
    { code: "+34", country: "Spain", flag: "🇪🇸" },
    { code: "+84", country: "Vietnam", flag: "🇻🇳" },
    { code: "+58", country: "Venezuela", flag: "🇻🇪" },
    { code: "+52", country: "Mexico", flag: "🇲🇽" },
  ];

  // Update tab when URL changes
  useEffect(() => {
    const searchParams = new URLSearchParams(location.search);
    const tabParam = searchParams.get('tab') as TabType;
    
    if (tabParam && ['profile', 'membership', 'points', 'cards', 'security', 'calendar'].includes(tabParam)) {
      setActiveTab(tabParam);
    }
  }, [location.search]);

  // Save active tab to localStorage whenever it changes
  useEffect(() => {
    localStorage.setItem('accountSettingsActiveTab', activeTab);
  }, [activeTab]);

  // Fetch points balance when points tab is selected
  useEffect(() => {
    if (activeTab === 'points') {
      fetchPointsBalance();
      fetchPaymentMethods(); // Also fetch payment methods for Buy Points modal
    }
  }, [activeTab]);

  // Fetch payment methods when cards tab is selected
  useEffect(() => {
    if (activeTab === 'cards') {
      fetchPaymentMethods();
    }
  }, [activeTab]);

  // Fetch membership data when membership tab is selected
  useEffect(() => {
    if (activeTab === 'membership') {
      fetchMembershipData();
    }
  }, [activeTab]);

  // Fetch calendar data when calendar tab is selected
  useEffect(() => {
    if (activeTab === 'calendar') {
      // Get user rates from localStorage
      const getUserRates = () => {
        try {
          const possibleKeys = ['userData', 'userFullResponse'];
          for (const key of possibleKeys) {
            const stored = localStorage.getItem(key);
            if (stored) {
              const parsed = JSON.parse(stored);
              const userData = key === 'userFullResponse' ? parsed.user : parsed;
              if (userData && (userData.daily_rate || userData.hourly_rate)) {
                return {
                  dailyRate: userData.daily_rate || '0.00',
                  hourlyRate: userData.hourly_rate || '0.00'
                };
              }
            }
          }
        } catch (error) {
          console.error('Error getting user rates:', error);
        }
        return { dailyRate: '0.00', hourlyRate: '0.00' };
      };

      const rates = getUserRates();
      console.log('Loaded user rates for calendar:', rates);
      
      // Update availability rows with user's rates
      setAvailabilityRows([
        { id: 1, date: '', startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: rates.dailyRate, hourlyRate: rates.hourlyRate }
      ]);
      
      // Update recurring rows with user's rates
      setRecurringRows([
        { id: 0, day: 'Mon', dayIndex: 0, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: rates.dailyRate, hourlyRate: rates.hourlyRate }] },
        { id: 1, day: 'Tue', dayIndex: 1, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: rates.dailyRate, hourlyRate: rates.hourlyRate }] },
        { id: 2, day: 'Wed', dayIndex: 2, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: rates.dailyRate, hourlyRate: rates.hourlyRate }] },
        { id: 3, day: 'Thu', dayIndex: 3, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: rates.dailyRate, hourlyRate: rates.hourlyRate }] },
        { id: 4, day: 'Fri', dayIndex: 4, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: rates.dailyRate, hourlyRate: rates.hourlyRate }] },
        { id: 5, day: 'Sat', dayIndex: 5, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: rates.dailyRate, hourlyRate: rates.hourlyRate }] },
        { id: 6, day: 'Sun', dayIndex: 6, enabled: false, slots: [{ startTime: '09:00 AM', endTime: '05:00 PM', dailyRate: rates.dailyRate, hourlyRate: rates.hourlyRate }] },
      ]);
      
      fetchCalendarSettings();
    }
  }, [activeTab]);

  // Utility function to parse mobile number correctly
  const parsePhoneNumber = (mobileNumber: string) => {
    if (!mobileNumber) {
      return { countryCode: '+91', phoneNumber: '' };
    }

    console.log('Parsing mobile number:', mobileNumber);
    
    // Clean the input
    let cleanNumber = mobileNumber.toString().trim().replace(/[\s\-]/g, '');
    
    // Common patterns for Indian numbers and others
    const patterns = [
      // Indian numbers: +919801023903 -> +91 and 9801023903
      { regex: /^(\+91)(\d{10})$/, desc: 'Indian 10-digit' },
      // US numbers: +19801023903 -> +1 and 9801023903  
      { regex: /^(\+1)(\d{10})$/, desc: 'US 10-digit' },
      // Generic international: +XX...XXXXXXXXXX
      { regex: /^(\+\d{1,3})(\d{8,12})$/, desc: 'International' },
      // Without plus: 919801023903 -> +91 and 9801023903
      { regex: /^(91)(\d{10})$/, desc: 'Indian without +' },
      // Just digits: 9801023903 -> +91 and 9801023903
      { regex: /^(\d{10})$/, desc: 'Just 10 digits' }
    ];

    for (const pattern of patterns) {
      const match = cleanNumber.match(pattern.regex);
      if (match) {
        let countryCode = match[1];
        let phoneNumber = match[2];
        
        // Add + if missing
        if (!countryCode.startsWith('+')) {
          countryCode = '+' + countryCode;
        }
        
        // Special case for just 10 digits - assume India
        if (pattern.desc === 'Just 10 digits') {
          countryCode = '+91';
          phoneNumber = match[1];
        }
        
        console.log(`Matched ${pattern.desc}: Country: ${countryCode}, Phone: ${phoneNumber}`);
        return { countryCode, phoneNumber };
      }
    }
    
    // Fallback
    console.log('No pattern matched, using fallback');
    return { countryCode: '+91', phoneNumber: cleanNumber.replace(/^\+/, '') };
  };

  // Initialize profile data from stored authentication data
  const initializeProfileData = () => {
    setIsLoadingProfile(true);
    
    // Check for auth token from different possible keys
    const authToken = localStorage.getItem('authToken') || 
                     localStorage.getItem('auth_token') || 
                     localStorage.getItem('token');
    
    if (!authToken) {
      console.log('No auth token found - user may need to login');
    } else {
      console.log('Auth token found, loading profile data...');
    }
    
    // Load user data from localStorage (set during login via handleAuthSuccess)
    loadUserProfileFromStorage();
    
    setIsLoadingProfile(false);
  };

  // Save login response data (call this after successful login)
  const saveLoginResponse = (loginResponse: any) => {
    try {
      console.log('Saving login response:', loginResponse);
      
      if (loginResponse.status && loginResponse.user) {
        const userData = loginResponse.user;
        
        // Save all relevant data
        localStorage.setItem('userData', JSON.stringify(userData));
        localStorage.setItem('userFullResponse', JSON.stringify(loginResponse));
        
        // Save the token
        if (loginResponse.token) {
          localStorage.setItem('authToken', loginResponse.token);
        }
        
        // Immediately update the UI with the new data
        updateUIWithUserData(userData);
        
        console.log('Login data saved successfully');
        return true;
      }
      return false;
    } catch (error) {
      console.error('Error saving login response:', error);
      return false;
    }
  };

  // Helper function to update UI with user data
  const updateUIWithUserData = (userData: any) => {
    // Update form data
    setFormData(prev => ({
      ...prev,
      firstName: userData.fname || '',
      lastName: userData.lname || '',
      bio: userData.bio || '',
      hourlyRate: userData.hourly_rate || '0.00',
      projectRate: userData.daily_rate || '0.00',
      homeCityId: userData.home_city_id ? userData.home_city_id.toString() : ''
    }));
    
    // Extract country code and phone number from mobile_number
    const parsed = parsePhoneNumber(userData.mobile_number || '');
    const countryCode = parsed.countryCode;
    const phoneNumber = parsed.phoneNumber;
    
    // Update user profile state
    const profileData = {
      email: userData.email || '',
      phoneNumber: phoneNumber,
      countryCode: countryCode,
      profilePicture: userData.profile_pic || ''
    };
    
    setUserProfile(profileData);
    
    // Also set original profile for comparison
    setOriginalProfile({
      email: userData.email || '',
      phoneNumber: phoneNumber,
      countryCode: countryCode
    });
    
    // Set profile picture URL if available
    if (userData.profile_pic) {
      setProfilePictureUrl(userData.profile_pic);
    }
    
    // Set the country dropdown
    const matchingCountry = countryCodes.find(country => 
      country.code === countryCode
    );
    if (matchingCountry) {
      setSelectedCountry(matchingCountry);
      console.log('Updated selectedCountry:', matchingCountry);
    } else {
      console.log('No matching country found for code:', countryCode);
      // Set a default country if no match found
      const defaultCountry = countryCodes.find(country => country.code === '+91');
      if (defaultCountry) {
        setSelectedCountry(defaultCountry);
      }
    }
  };

  // Load user profile data from localStorage (fallback)
  const loadUserProfileFromStorage = () => {
    try {
      // Try different possible keys including the ones from the auth system
      const possibleKeys = ['userData', 'user_data', 'user', 'userInfo', 'profile', 'userFullResponse'];
      let userData = null;
      
      for (const key of possibleKeys) {
        const stored = localStorage.getItem(key);
        if (stored) {
          try {
            const parsed = JSON.parse(stored);
            
            // If it's the full response, extract the user data
            if (key === 'userFullResponse' && parsed.user) {
              userData = parsed.user;
              console.log(`Found user data in full response:`, userData);
              break;
            } else if (key !== 'userFullResponse') {
              userData = parsed;
              console.log(`Found user data in localStorage key: ${key}`, userData);
              break;
            }
          } catch (e) {
            console.log(`Failed to parse data from key: ${key}`);
          }
        }
      }

      if (userData) {
        console.log('Found user data:', userData);
        console.log('Mobile number in data:', userData.mobile_number);
        // Use the helper function to update UI
        updateUIWithUserData(userData);
      } else {
        console.log('No user data found in localStorage under any key');
      }

      // Debug: Log what's in localStorage
      console.log('localStorage keys:', Object.keys(localStorage));
      console.log('authToken:', localStorage.getItem('authToken'));
      console.log('auth_token:', localStorage.getItem('auth_token'));
      console.log('token:', localStorage.getItem('token'));
    } catch (error) {
      console.error('Error loading user data from localStorage:', error);
    }
  };



  // Debug function to check current state
  const debugProfileData = () => {
    console.log('=== PROFILE DEBUG INFO ===');
    console.log('userProfile:', userProfile);
    console.log('originalProfile:', originalProfile);
    console.log('selectedCountry:', selectedCountry);
    
    // Check localStorage data
    const keys = ['userData', 'user_data', 'userFullResponse'];
    keys.forEach(key => {
      const data = localStorage.getItem(key);
      if (data) {
        try {
          const parsed = JSON.parse(data);
          console.log(`${key}:`, parsed);
          if (parsed.mobile_number) {
            console.log(`${key} mobile_number:`, parsed.mobile_number);
          }
        } catch (e) {
          console.log(`${key} (raw):`, data);
        }
      }
    });
    
    return {
      userProfile,
      originalProfile,
      selectedCountry,
      otpStates: {
        showOtpModal,
        otpData,
        isSendingOtp,
        isVerifyingOtp,
        otpSent,
        resendTimer
      },
      localStorage: Object.keys(localStorage).reduce((acc, key) => {
        if (['userData', 'user_data', 'userFullResponse'].includes(key)) {
          try {
            acc[key] = JSON.parse(localStorage.getItem(key) || '{}');
          } catch (e) {
            acc[key] = localStorage.getItem(key);
          }
        }
        return acc;
      }, {} as any)
    };
  };

  // Fetch locations
  const fetchLocations = async () => {
    setIsLoadingLocations(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        console.error('No auth token found');
        return;
      }

      const response = await fetch(`${API_BASE_URL}/locations`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status && result.data) {
        // Remove duplicates by id
        const uniqueLocations = result.data.reduce((acc: any[], current: any) => {
          const exists = acc.find(item => item.id === current.id);
          if (!exists) {
            acc.push(current);
          }
          return acc;
        }, []);
        setLocations(uniqueLocations);
      } else {
        console.error('Failed to fetch locations:', result);
      }
    } catch (error) {
      console.error('Error fetching locations:', error);
    } finally {
      setIsLoadingLocations(false);
    }
  };

  // Load user profile data on component mount
  useEffect(() => {
    // Expose saveLoginResponse function globally for use after login
    (window as any).saveLoginResponse = saveLoginResponse;
    (window as any).debugProfileData = debugProfileData;
    console.log('Available functions:');
    console.log('- window.saveLoginResponse(loginResponse) - Call this after successful login');
    console.log('- window.debugProfileData() - Debug current profile state');
    
    // Initialize profile data from authentication system
    initializeProfileData();
    
    // Fetch locations
    fetchLocations();
  }, []);

  // OTP resend timer effect
  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (resendTimer > 0) {
      interval = setInterval(() => {
        setResendTimer(prev => prev - 1);
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [resendTimer]);



  useEffect(() => {
    const handleKeyPress = (e: KeyboardEvent) => {
      if (isDropdownOpen) {
        const key = e.key.toLowerCase();
        const matchingCountry = countryCodes.find((country) =>
          country.country.toLowerCase().startsWith(key)
        );

        if (matchingCountry) {
          const countryElement = document.querySelector(
            `[data-country="${matchingCountry.country}"]`
          ) as HTMLElement;
          if (countryElement) {
            countryElement.scrollIntoView({ block: "nearest" });
            countryElement.focus();
          }
        }
      }
    };

    document.addEventListener("keydown", handleKeyPress);
    return () => document.removeEventListener("keydown", handleKeyPress);
  }, [isDropdownOpen, countryCodes]);

  const handleInputChange = (field: string, value: string) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  // Profile picture upload handler
  const handleProfilePictureUpload = async (file: File) => {
    if (!file) return;
    
    setIsUploading(true);
    try {
      const formData = new FormData();
      formData.append('type', 'profile');
      formData.append('file', file);

      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token') || '';
      const response = await fetch(`${API_BASE_URL}/save-profile-picture`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      const result = await response.json();
      
      if (response.ok) {
        setProfilePictureUrl(result.url || URL.createObjectURL(file));
        setProfilePicture(file);
        console.log('Profile picture uploaded successfully:', result);
      } else {
        console.error('Upload failed:', result);
        alert('Failed to upload profile picture. Please try again.');
      }
    } catch (error) {
      console.error('Upload error:', error);
      alert('Error uploading profile picture. Please try again.');
    } finally {
      setIsUploading(false);
    }
  };

  // Remove profile picture
  const handleRemoveProfilePicture = async () => {
    setIsUploading(true);
    try {
      const formData = new FormData();
      
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token') || '';
      const response = await fetch(`${API_BASE_URL}/remove-profile-picture`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      const result = await response.json();
      
      if (response.ok) {
        setProfilePictureUrl(null);
        setProfilePicture(null);
        console.log('Profile picture removed successfully:', result);
      } else {
        console.error('Remove failed:', result);
        alert('Failed to remove profile picture. Please try again.');
      }
    } catch (error) {
      console.error('Remove error:', error);
      alert('Error removing profile picture. Please try again.');
    } finally {
      setIsUploading(false);
    }
  };

  // Save account settings
  const handleSaveAccountSettings = async () => {
    setIsSaving(true);
    try {
      const formDataToSend = new FormData();
      formDataToSend.append('fname', formData.firstName);
      formDataToSend.append('lname', formData.lastName);
      formDataToSend.append('description', formData.bio);
      formDataToSend.append('daily_rate', formData.projectRate);
      formDataToSend.append('hourly_rate', formData.hourlyRate);
      if (formData.homeCityId) {
        formDataToSend.append('home_city', formData.homeCityId);
      }

      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token') || '';
      const response = await fetch(`${API_BASE_URL}/account-setting-save`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formDataToSend
      });

      const result = await response.json();
      
      if (response.ok && result.status) {
        console.log('Account settings saved successfully:', result);
        
        // Update localStorage with new home_city_id if it was changed
        if (formData.homeCityId) {
          try {
            const newHomeCityId = parseInt(formData.homeCityId, 10);
            
            // Fetch locations to get the banner for the new home city
            const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
            if (token) {
              try {
                const locationsResponse = await fetch(`${API_BASE_URL}/locations`, {
                  method: 'GET',
                  headers: {
                    'Authorization': `Bearer ${token}`,
                    'Accept': 'application/json',
                  },
                });
                
                if (locationsResponse.ok) {
                  const locationsResult = await locationsResponse.json();
                  if (locationsResult.status && Array.isArray(locationsResult.data)) {
                    // Find the banner for the new home city
                    const newHomeCity = locationsResult.data.find((loc: any) => loc.id === newHomeCityId);
                    if (newHomeCity && newHomeCity.banner_picture) {
                      console.log('Found banner for new home city:', newHomeCity.banner_picture);
                      // Store banner in localStorage for quick access
                      localStorage.setItem('home_city_banner', newHomeCity.banner_picture);
                    }
                  }
                }
              } catch (locError) {
                console.error('Error fetching location banner:', locError);
              }
            }
            
            // Update user_data in localStorage
            const userData = localStorage.getItem('user_data');
            if (userData) {
              const parsedUserData = JSON.parse(userData);
              parsedUserData.home_city_id = newHomeCityId;
              parsedUserData.home_city = newHomeCityId;
              localStorage.setItem('user_data', JSON.stringify(parsedUserData));
              console.log('Updated home_city_id in localStorage:', parsedUserData.home_city_id);
            }
            
            // Also update userData if it exists (another common key)
            const userDataAlt = localStorage.getItem('userData');
            if (userDataAlt) {
              const parsedUserDataAlt = JSON.parse(userDataAlt);
              parsedUserDataAlt.home_city_id = newHomeCityId;
              parsedUserDataAlt.home_city = newHomeCityId;
              localStorage.setItem('userData', JSON.stringify(parsedUserDataAlt));
            }
            
            // Dispatch custom event to notify other components (like home page)
            window.dispatchEvent(new Event('accountSettingsUpdated'));
          } catch (error) {
            console.error('Error updating localStorage:', error);
          }
        }
        
        alert('Account settings saved successfully!');
      } else {
        console.error('Save failed:', result);
        
        // Display specific error message
        let errorMessage = 'Failed to save account settings.';
        
        if (result.error) {
          // Handle validation errors object
          if (typeof result.error === 'object') {
            const errors = Object.values(result.error).flat();
            errorMessage = errors.join('\n');
          } else {
            errorMessage = result.error;
          }
        } else if (result.message) {
          errorMessage = result.message;
        }
        
        alert(errorMessage);
      }
    } catch (error) {
      console.error('Save error:', error);
      alert('Error saving account settings. Please try again.');
    } finally {
      setIsSaving(false);
    }
  };

  // File input handler
  const handleFileInputChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      // Check file size (max 5MB)
      if (file.size > 5 * 1024 * 1024) {
        alert('File size must be less than 5MB');
        return;
      }
      
      // Check file type
      if (!file.type.startsWith('image/')) {
        alert('Please select an image file');
        return;
      }
      
      handleProfilePictureUpload(file);
    }
  };

  // Fetch points balance
  const fetchPointsBalance = async () => {
    setIsLoadingPoints(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        console.error('No auth token found');
        return;
      }

      const response = await fetch(`${API_BASE_URL}/point-balance`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        setPointsBalance({
          lastMonthBookingCost: result.data.last_month_booking_cost || 0,
          currentMonthBookingCost: result.data.current_month_booking_cost || 0,
          currentBalance: result.data.current_balance || 0,
          points: result.data.points || [],
          pointsAction: result.data.points_action || {}
        });
      } else {
        console.error('Failed to fetch points balance:', result);
      }
    } catch (error) {
      console.error('Error fetching points balance:', error);
    } finally {
      setIsLoadingPoints(false);
    }
  };

  // Buy points handler
  const handleBuyPoints = async () => {
    if (!pointsToBuy || pointsToBuy === '0') {
      alert('Please enter a valid number of points to buy.');
      return;
    }

    if (!selectedPaymentMethodId) {
      alert('Please select a payment method.');
      return;
    }

    try {
      setIsBuyingPoints(true);
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication required.');
        return;
      }

      const formData = new FormData();
      formData.append('points', pointsToBuy);
      formData.append('token', selectedPaymentMethodId);

      const response = await fetch(`${API_BASE_URL}/buy-more-points`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData,
      });

      if (response.ok) {
        const result = await response.json();
        alert(`Successfully purchased ${pointsToBuy} points!`);
        setShowBuyPointsModal(false);
        setPointsToBuy('');
        setSelectedPaymentMethodId('');
        // Refresh points balance
        fetchPointsBalance();
      } else {
        const errorData = await response.json();
        alert(errorData.message || 'Failed to purchase points. Please try again.');
      }
    } catch (error) {
      console.error('Buy points error:', error);
      alert('Failed to purchase points. Please try again.');
    } finally {
      setIsBuyingPoints(false);
    }
  };

  // Fetch payment methods
  const fetchPaymentMethods = async () => {
    setIsLoadingPaymentMethods(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        console.error('No auth token found');
        return;
      }

      const response = await fetch(`${API_BASE_URL}/payment-methods`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        setPaymentMethods(result.payment_methods || []);
        setDefaultPaymentMethod(result.default_payment_method || null);
      } else {
        console.error('Failed to fetch payment methods:', result);
      }
    } catch (error) {
      console.error('Error fetching payment methods:', error);
    } finally {
      setIsLoadingPaymentMethods(false);
    }
  };

  // Delete payment method
  const handleDeletePaymentMethod = async (pmid: string) => {
    if (!confirm('Are you sure you want to delete this payment method?')) {
      return;
    }

    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication token not found');
        return;
      }

      const formData = new FormData();
      formData.append('pmid', pmid);

      const response = await fetch(`${API_BASE_URL}/payment-method-delete`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      const result = await response.json();

      if (response.ok && result.status) {
        // Refresh payment methods list
        await fetchPaymentMethods();
        alert('Payment method deleted successfully');
      } else {
        alert(result.message || 'Failed to delete payment method. Please try again.');
      }
    } catch (error) {
      console.error('Delete payment method error:', error);
      alert('An error occurred while deleting the payment method.');
    }
  };

  // Fetch membership data and transactions
  const fetchMembershipData = async () => {
    setIsLoadingMembership(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        console.error('No auth token found');
        return;
      }

      // Fetch transactions from membership API
      const response = await fetch(`${API_BASE_URL}/membership`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        // Get user data from localStorage to get plan info
        // Check multiple possible localStorage keys, same as change-plan page
        const possibleKeys = ['userData', 'userFullResponse', 'user_data'];
        let planData = {
          name: 'Guest',
          price: 0,
          points: 0,
          pm_type: '',
          pm_last_four: ''
        };

        for (const key of possibleKeys) {
          const stored = localStorage.getItem(key);
          if (stored) {
            try {
              const parsed = JSON.parse(stored);
              // Handle different data structures
              const userData = key === 'userFullResponse' ? parsed.user : parsed;
              
              if (userData?.plan) {
                planData = {
                  name: userData.plan.name || 'Guest',
                  price: userData.plan.price || 0,
                  points: userData.plan.points || 0,
                  pm_type: userData.plan.pm_type || '',
                  pm_last_four: userData.plan.pm_last_four || ''
                };
                console.log('Loaded plan from', key, ':', planData.name);
                break; // Stop after finding plan data
              }
            } catch (e) {
              console.error(`Error parsing ${key}:`, e);
            }
          }
        }

        setMembershipData({
          plan: planData,
          transactions: result.data.points || []
        });
      } else {
        console.error('Failed to fetch membership data:', result);
      }
    } catch (error) {
      console.error('Error fetching membership data:', error);
    } finally {
      setIsLoadingMembership(false);
    }
  };

  // Cancel plan
  const handleCancelPlan = async () => {
    if (!confirm('Are you sure you want to cancel your plan?')) {
      return;
    }

    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication token not found');
        return;
      }

      const response = await fetch(`${API_BASE_URL}/subscription-cancel`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        alert('Plan cancelled successfully');
        // Refresh membership data
        fetchMembershipData();
      } else {
        alert(result.message || 'Failed to cancel plan. Please try again.');
      }
    } catch (error) {
      console.error('Error cancelling plan:', error);
      alert('An error occurred while cancelling the plan.');
    }
  };

  // Fetch calendar settings
  const fetchCalendarSettings = async () => {
    setIsLoadingCalendar(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        console.error('No auth token found');
        setIsLoadingCalendar(false);
        return;
      }

      const response = await fetch(`${API_BASE_URL}/account-setting-calendar`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status && result.data) {
        const locationsObj = result.data.locations || {};
        const firstLocationId = Object.keys(locationsObj)[0] || '';
        
        setCalendarSettings({
          locations: locationsObj,
          refrence_id: result.data.refrence_id || 0,
          min_booking: result.data.min_booking || { min_days: '0', min_hours: '02', min_minutes: '00' },
          max_booking: result.data.max_booking || { max_days: '05', max_hours: '00', max_minutes: '00' },
          schedule_location: firstLocationId
        });
        
        // Fetch events after settings are loaded
        if (result.data.refrence_id && firstLocationId) {
          await fetchCalendarEvents(result.data.refrence_id, firstLocationId);
        }
      } else {
        console.error('Failed to fetch calendar settings:', result);
      }
    } catch (error) {
      console.error('Error fetching calendar settings:', error);
    } finally {
      setIsLoadingCalendar(false);
    }
  };

  // Fetch calendar events
  const fetchCalendarEvents = async (refrence_id?: number, schedule_location?: string) => {
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        console.error('No auth token found');
        return;
      }

      // Calculate date range for current month
      const now = currentDate || new Date();
      const startOfMonth = new Date(now.getFullYear(), now.getMonth(), 1);
      const endOfMonth = new Date(now.getFullYear(), now.getMonth() + 1, 0);
      
      // Format dates as ISO strings
      const startDate = startOfMonth.toISOString().split('T')[0] + 'T00:00:00Z';
      const endDate = endOfMonth.toISOString().split('T')[0] + 'T23:59:59Z';

      const formData = new FormData();
      formData.append('category', 'schedule');
      formData.append('type', 'resource');
      formData.append('refrence_id', String(refrence_id || calendarSettings.refrence_id));
      formData.append('schedule_location', schedule_location || calendarSettings.schedule_location);
      formData.append('start', startDate);
      formData.append('end', endDate);
      formData.append('timeZone', 'UTC');

      console.log('Fetching calendar events with params:', {
        refrence_id: refrence_id || calendarSettings.refrence_id,
        schedule_location: schedule_location || calendarSettings.schedule_location,
        startDate,
        endDate
      });

      const response = await fetch(`${API_BASE_URL}/calendar-sch-events`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      console.log('Calendar events response:', result);
      
      // Handle both response formats - array directly or wrapped in data
      if (Array.isArray(result)) {
        setCalendarEvents(result);
        console.log('Set calendar events (array):', result.length, 'events');
      } else if (result.status && result.data) {
        setCalendarEvents(result.data || []);
        console.log('Set calendar events (wrapped):', result.data.length, 'events');
      } else {
        console.error('Unexpected response format:', result);
        setCalendarEvents([]);
      }
    } catch (error) {
      console.error('Error fetching calendar events:', error);
      setCalendarEvents([]);
    }
  };

  // Save calendar schedule
  const handleSaveCalendarSchedule = async () => {
    setIsSavingCalendar(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication token not found');
        return;
      }

      // Validate that a location is selected
      if (!calendarSettings.schedule_location) {
        alert('Please select a location before saving.');
        setIsSavingCalendar(false);
        return;
      }

      console.log('Saving calendar with location:', calendarSettings.schedule_location);
      console.log('Full calendar settings:', calendarSettings);

      const formData = new FormData();
      formData.append('type', 'resource');
      formData.append('refrence_id', String(calendarSettings.refrence_id));
      formData.append('schedule_location', calendarSettings.schedule_location);
      formData.append('min_days', calendarSettings.min_booking.min_days);
      formData.append('min_hours', calendarSettings.min_booking.min_hours);
      formData.append('min_minutes', calendarSettings.min_booking.min_minutes);
      formData.append('max_days', calendarSettings.max_booking.max_days);
      formData.append('max_hours', calendarSettings.max_booking.max_hours);
      formData.append('max_minutes', calendarSettings.max_booking.max_minutes);
      formData.append('availability_type', availabilityType === 'daily' ? '0' : '1');

      if (availabilityType === 'daily') {
        // Add daily schedules
        availabilityRows.forEach((row, index) => {
          if (row.date) {
            formData.append(`daily_start_date[${index}]`, row.date);
            formData.append(`daily_start_time[${index}]`, row.startTime);
            formData.append(`daily_end_time[${index}]`, row.endTime);
            formData.append(`daily_daily_rate[${index}]`, row.dailyRate);
            formData.append(`daily_hourly_rate[${index}]`, row.hourlyRate);
          }
        });
      } else {
        // Add recurring schedules
        recurringRows.forEach((row) => {
          if (row.enabled && row.slots.length > 0) {
            // Add checkbox value for enabled days
            formData.append(`recurring_days[${row.dayIndex}]`, String(row.dayIndex));
            
            // Add all slots for this day
            row.slots.forEach((slot, slotIndex) => {
              formData.append(`recurring_start_time[${row.dayIndex}][${slotIndex}]`, slot.startTime);
              formData.append(`recurring_end_time[${row.dayIndex}][${slotIndex}]`, slot.endTime);
              formData.append(`recurring_daily_rate[${row.dayIndex}][${slotIndex}]`, slot.dailyRate);
              formData.append(`recurring_hourly_rate[${row.dayIndex}][${slotIndex}]`, slot.hourlyRate);
            });
          }
        });
      }

      const response = await fetch(`${API_BASE_URL}/calendar-upsert-sch`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        alert('Calendar schedule saved successfully!');
        // Refresh calendar events
        fetchCalendarEvents();
      } else {
        alert(result.message || 'Failed to save calendar schedule. Please try again.');
      }
    } catch (error) {
      console.error('Error saving calendar schedule:', error);
      alert('An error occurred while saving the calendar schedule.');
    } finally {
      setIsSavingCalendar(false);
    }
  };

  // Send OTP for security verification
  const handleSendSecurityOtp = async () => {
    setIsSendingOtp(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication token not found');
        return;
      }

      const formData = new FormData();
      formData.append('email', userProfile.email);
      formData.append('phone_number', `${userProfile.countryCode}${userProfile.phoneNumber}`);

      const response = await fetch(`${API_BASE_URL}/security-otp-send`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        setOtpSent(true);
        setShowOtpModal(true);
        setResendTimer(24); // 24 seconds as shown in the image
        alert('OTP sent successfully to your email and mobile number');
      } else {
        alert('Failed to send OTP. Please try again.');
      }
    } catch (error) {
      console.error('Error sending OTP:', error);
      alert('Error sending OTP. Please check your connection and try again.');
    } finally {
      setIsSendingOtp(false);
    }
  };

  // Verify OTP
  const handleVerifySecurityOtp = async () => {
    if (!otpData.emailOtp || !otpData.mobileOtp) {
      alert('Please enter both email and mobile OTP codes');
      return;
    }

    setIsVerifyingOtp(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication token not found');
        return;
      }

      const formData = new FormData();
      formData.append('email', userProfile.email);
      formData.append('mobile', `${userProfile.countryCode}${userProfile.phoneNumber}`);
      formData.append('otp_email', otpData.emailOtp);
      formData.append('otp_mobile', otpData.mobileOtp);

      const response = await fetch(`${API_BASE_URL}/verify-security-otp`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        alert('Security verification successful!');
        setShowOtpModal(false);
        setOtpData({ emailOtp: '', mobileOtp: '' });
        setOtpSent(false);
      } else {
        alert('Invalid OTP codes. Please try again.');
      }
    } catch (error) {
      console.error('Error verifying OTP:', error);
      alert('Error verifying OTP. Please try again.');
    } finally {
      setIsVerifyingOtp(false);
    }
  };

  // Handle OTP input change
  const handleOtpChange = (field: 'emailOtp' | 'mobileOtp', value: string) => {
    // Only allow numbers and limit to 6 digits
    const numericValue = value.replace(/\D/g, '').slice(0, 6);
    setOtpData(prev => ({ ...prev, [field]: numericValue }));
  };

  // Handle save security settings - verify with OTP only for changed fields
  const handleSaveSecuritySettings = async () => {
    // Detect which fields changed
    const changes: ('email' | 'phone')[] = [];
    
    if (userProfile.email !== originalProfile.email) {
      changes.push('email');
    }
    
    const currentPhone = `${userProfile.countryCode}${userProfile.phoneNumber}`;
    const originalPhone = `${originalProfile.countryCode}${originalProfile.phoneNumber}`;
    
    if (currentPhone !== originalPhone) {
      changes.push('phone');
    }
    
    if (changes.length === 0) {
      alert('No changes detected');
      return;
    }
    
    setChangedFields(changes);
    await sendSecurityOtpForVerification(changes);
  };

  // Send OTP for security verification (only to changed fields)
  const sendSecurityOtpForVerification = async (fields: ('email' | 'phone')[]) => {
    setIsSendingOtp(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication token not found');
        return;
      }

      const formData = new FormData();
      
      // Always send both fields (API requirement), but OTP only sent to changed fields
      formData.append('email', userProfile.email);
      formData.append('phone_number', `${userProfile.countryCode}${userProfile.phoneNumber}`);
      
      // Send indicator of which fields changed (if API supports it)
      // This helps backend know which fields to send OTP to
      formData.append('changed_fields', fields.join(','));

      const response = await fetch(`${API_BASE_URL}/security-otp-send`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      const result = await response.json();
      
      if (result.status) {
        setOtpSent(true);
        setShowOtpModal(true);
        setResendTimer(30);
        
        // Dynamic message based on what changed
        let message = 'OTP sent successfully to your ';
        if (fields.includes('email') && fields.includes('phone')) {
          message += 'email and mobile number';
        } else if (fields.includes('email')) {
          message += 'email';
        } else {
          message += 'mobile number';
        }
        alert(message);
      } else {
        // Display the actual error message from API
        let errorMessage = 'Failed to send OTP. Please try again.';
        
        if (result.error) {
          // Handle nested error object structure
          if (typeof result.error === 'string') {
            errorMessage = result.error;
          } else if (typeof result.error === 'object') {
            // Extract first error message from nested object
            const errorKey = Object.keys(result.error)[0];
            if (errorKey && Array.isArray(result.error[errorKey])) {
              errorMessage = result.error[errorKey][0];
            } else if (errorKey) {
              errorMessage = result.error[errorKey];
            }
          }
        } else if (result.message) {
          errorMessage = result.message;
        }
        
        alert(errorMessage);
      }
    } catch (error) {
      console.error('Error sending OTP:', error);
      alert('Error sending OTP. Please check your connection and try again.');
    } finally {
      setIsSendingOtp(false);
    }
  };

  // Verify security OTP (updated to match correct API)
  const handleVerifySecurityOtpNew = async () => {
    // Validate based on changed fields
    if (changedFields.includes('email') && !otpData.emailOtp) {
      alert('Please enter email OTP code');
      return;
    }
    
    if (changedFields.includes('phone') && !otpData.mobileOtp) {
      alert('Please enter mobile OTP code');
      return;
    }

    if (changedFields.includes('email') && otpData.emailOtp.length !== 6) {
      alert('Please enter complete 6-digit email OTP');
      return;
    }
    
    if (changedFields.includes('phone') && otpData.mobileOtp.length !== 6) {
      alert('Please enter complete 6-digit mobile OTP');
      return;
    }

    setIsVerifyingOtp(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication token not found');
        return;
      }

      const formData = new FormData();
      formData.append('email', userProfile.email);
      formData.append('mobile', `${userProfile.countryCode}${userProfile.phoneNumber}`);
      formData.append('otp_email', otpData.emailOtp);
      formData.append('otp_mobile', otpData.mobileOtp);

      const response = await fetch(`${API_BASE_URL}/verify-security-otp`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        // Save the changes to localStorage
        saveSecuritySettingsDirectly();
        setShowOtpModal(false);
        setOtpData({ emailOtp: '', mobileOtp: '' });
        setOtpSent(false);
        
        // Update original profile to new values
        setOriginalProfile({
          email: userProfile.email,
          phoneNumber: userProfile.phoneNumber,
          countryCode: userProfile.countryCode
        });
        
        alert('Security settings updated successfully!');
      } else {
        alert('Invalid OTP codes. Please try again.');
      }
    } catch (error) {
      console.error('Error verifying OTP:', error);
      alert('Error verifying OTP. Please try again.');
    } finally {
      setIsVerifyingOtp(false);
    }
  };

  // Save security settings without verification (no changes made)
  const saveSecuritySettingsDirectly = async () => {
    try {
      // Update the stored user data
      const userData = localStorage.getItem('userData');
      if (userData) {
        const parsed = JSON.parse(userData);
        const updatedUserData = {
          ...parsed,
          email: userProfile.email,
          phone_number: userProfile.phoneNumber,
          country_code: userProfile.countryCode,
          mobile: `${userProfile.countryCode}${userProfile.phoneNumber}`
        };
        localStorage.setItem('userData', JSON.stringify(updatedUserData));
        alert('Security settings updated successfully!');
      }
    } catch (error) {
      console.error('Error saving security settings:', error);
      alert('Error saving security settings. Please try again.');
    }
  };



  // Resend security OTP
  const handleResendSecurityOtp = async () => {
    if (resendTimer > 0) return;
    
    setResendTimer(30);
    setOtpData({ emailOtp: '', mobileOtp: '' });
    
    await sendSecurityOtpForVerification(changedFields);
  };

  // Calendar helper functions
  const formatMonthYear = (date: Date) => {
    return date.toLocaleDateString('en-US', { month: 'short', year: 'numeric' }).toUpperCase();
  };

  const navigateMonth = (direction: 'prev' | 'next') => {
    setCurrentDate(prev => {
      const newDate = new Date(prev);
      if (direction === 'prev') {
        newDate.setMonth(prev.getMonth() - 1);
      } else {
        newDate.setMonth(prev.getMonth() + 1);
      }
      
      // Refetch events for the new month
      setTimeout(() => {
        if (calendarSettings.refrence_id && calendarSettings.schedule_location) {
          fetchCalendarEvents(calendarSettings.refrence_id, calendarSettings.schedule_location);
        }
      }, 100);
      
      return newDate;
    });
  };

  const addAvailabilityRow = () => {
    if (availabilityType === 'daily') {
      // Get user rates from the first row if available, or from localStorage
      let dailyRate = '0.00';
      let hourlyRate = '0.00';
      
      if (availabilityRows.length > 0) {
        dailyRate = availabilityRows[0].dailyRate;
        hourlyRate = availabilityRows[0].hourlyRate;
      }
      
      const newId = Math.max(...availabilityRows.map(r => r.id)) + 1;
      setAvailabilityRows([...availabilityRows, {
        id: newId,
        date: '',
        startTime: '09:00 AM',
        endTime: '05:00 PM',
        dailyRate: dailyRate,
        hourlyRate: hourlyRate
      }]);
    }
  };

  const deleteAvailabilityRow = (id: number) => {
    if (availabilityRows.length > 1) {
      setAvailabilityRows(availabilityRows.filter(row => row.id !== id));
    }
  };

  const updateAvailabilityRow = (id: number, field: string, value: string) => {
    setAvailabilityRows(prev => 
      prev.map(row => row.id === id ? { ...row, [field]: value } : row)
    );
  };

  const updateRecurringRow = (id: number, slotIndex: number, field: string, value: string | boolean) => {
    setRecurringRows(prev =>
      prev.map(row => {
        if (row.id === id) {
          if (field === 'enabled') {
            return { ...row, enabled: value as boolean };
          } else {
            const updatedSlots = [...row.slots];
            updatedSlots[slotIndex] = { ...updatedSlots[slotIndex], [field]: value };
            return { ...row, slots: updatedSlots };
          }
        }
        return row;
      })
    );
  };

  const addRecurringRow = () => {
    const newId = Math.max(...recurringRows.map(r => r.id)) + 1;
    const dayNames = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];
    const dayIndex = recurringRows.length % 7;
    setRecurringRows([...recurringRows, {
      id: newId,
      day: dayNames[dayIndex],
      dayIndex: dayIndex,
      enabled: false,
      slots: [{
        startTime: '09:00 AM',
        endTime: '05:00 PM',
        dailyRate: '0.00',
        hourlyRate: '0.00'
      }]
    }]);
  };

  const addSlotToDay = (rowId: number) => {
    setRecurringRows(prev =>
      prev.map(row => {
        if (row.id === rowId) {
          // Use rates from existing slot if available
          const existingSlot = row.slots[0] || { dailyRate: '0.00', hourlyRate: '0.00' };
          return {
            ...row,
            slots: [...row.slots, {
              startTime: '09:00 AM',
              endTime: '05:00 PM',
              dailyRate: existingSlot.dailyRate,
              hourlyRate: existingSlot.hourlyRate
            }]
          };
        }
        return row;
      })
    );
  };

  const removeSlotFromDay = (rowId: number, slotIndex: number) => {
    setRecurringRows(prev =>
      prev.map(row => {
        if (row.id === rowId && row.slots.length > 1) {
          const updatedSlots = row.slots.filter((_, index) => index !== slotIndex);
          return { ...row, slots: updatedSlots };
        }
        return row;
      })
    );
  };

  const deleteRecurringRow = (id: number) => {
    if (recurringRows.length > 1) {
      setRecurringRows(recurringRows.filter(row => row.id !== id));
    }
  };

  const generateCalendarGrid = () => {
    const year = currentDate.getFullYear();
    const month = currentDate.getMonth();
    const firstDay = new Date(year, month, 1);
    const lastDay = new Date(year, month + 1, 0);
    const startDate = new Date(firstDay);
    startDate.setDate(startDate.getDate() - firstDay.getDay());
    
    const days = [];
    const currentDateLoop = new Date(startDate);
    
    for (let i = 0; i < 42; i++) {
      days.push(new Date(currentDateLoop));
      currentDateLoop.setDate(currentDateLoop.getDate() + 1);
    }
    
    return days;
  };

  const generateWeekView = () => {
    // Get the start of the week (Sunday) for current date
    const startOfWeek = new Date(currentDate);
    startOfWeek.setDate(currentDate.getDate() - currentDate.getDay());
    
    // Generate 7 days from start of week
    const days = [];
    for (let i = 0; i < 7; i++) {
      const day = new Date(startOfWeek);
      day.setDate(startOfWeek.getDate() + i);
      days.push(day);
    }
    
    const hours = ['6am', '7am', '8am', '9am', '10am', '11am', '12pm', '1pm', '2pm', '3pm', '4pm', '5pm', '6pm', '7pm'];
    
    return { days, hours };
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-black">
        <AppHeader theme="dark" />
        
        <main className="pt-20">
          {/* Header Section */}
          <div className="bg-black">
            <div className="mx-auto w-full px-4 sm:px-6 py-2" style={{ maxWidth: "1328px" }}>
              <h1 className="text-2xl sm:text-3xl lg:text-4xl font-bold text-white mb-2 uppercase tracking-wide">{t("accountSettings.title")}</h1>
              
              {/* Tabs Navigation */}
              <nav className="flex space-x-4 sm:space-x-8 overflow-x-auto scrollbar-hide -mx-4 sm:mx-0 px-4 sm:px-0">
                {tabs.map((tab) => (
                  <button
                    key={tab.id}
                    onClick={() => setActiveTab(tab.id)}
                    className={`py-4 px-1 text-xs sm:text-sm font-medium transition-colors relative uppercase tracking-wide whitespace-nowrap flex-shrink-0 ${
                      activeTab === tab.id
                        ? "text-white border-b-2 border-white"
                        : "text-white/60 hover:text-white/80 border-b-2 border-transparent"
                    }`}
                  >
                    {tab.label}
                  </button>
                ))}
              </nav>
            </div>
          </div>

          {/* Tab Content */}
          <div className="bg-black">
            <div className="mx-auto w-full px-4 sm:px-6 py-6 sm:py-8" style={{ maxWidth: "1328px" }}>
              
              {/* My Profile Tab */}
             {activeTab === "profile" && (
  <div className="max-w-2xl mx-auto w-full bg-white/10 backdrop-blur-lg rounded-xl sm:rounded-2xl border border-white/20 p-4 sm:p-6 lg:p-8 shadow-2xl">
    {/* Profile Picture Section */}
    <div className="flex flex-col items-center mb-6 sm:mb-8">
      <h3 className="text-white text-xs sm:text-sm font-medium mb-4 sm:mb-6 w-full text-left uppercase tracking-wider">Your Profile Picture</h3>
      
      <div className="relative">
        <label
          htmlFor="profile-upload"
          className={`w-32 h-32 sm:w-40 sm:h-40 rounded-full overflow-hidden bg-white/10 mb-4 flex items-center justify-center cursor-pointer hover:bg-white/15 transition-all duration-200 border-2 border-dashed border-white/20 hover:border-white/40 group ${
            isUploading ? 'opacity-50 cursor-not-allowed' : ''
          }`}
        >
          {profilePictureUrl ? (
            <img 
              src={profilePictureUrl} 
              alt="Profile" 
              className="w-full h-full object-cover"
            />
          ) : (
            <div className="flex flex-col items-center justify-center text-center">
              {isUploading ? (
                <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mb-2"></div>
              ) : (
                <Camera className="w-8 h-8 text-white/40 group-hover:text-white/60 mb-2 transition-colors" />
              )}
              <span className="text-xs text-white/40 group-hover:text-white/60 transition-colors uppercase tracking-wider">
                {isUploading ? 'Uploading...' : 'Upload Photo'}
              </span>
            </div>
          )}
          <input
            id="profile-upload"
            type="file"
            accept="image/*"
            className="hidden"
            onChange={handleFileInputChange}
            disabled={isUploading}
          />
        </label>
        
        {profilePictureUrl && (
          <button
            onClick={handleRemoveProfilePicture}
            disabled={isUploading}
            className="absolute -top-2 -right-2 w-8 h-8 bg-red-500 hover:bg-red-600 rounded-full flex items-center justify-center text-white transition-colors disabled:opacity-50"
          >
            <X className="w-4 h-4" />
          </button>
        )}
      </div>

      <div className="text-center">
        <p className="text-white/60 text-sm">
          <span className="text-orange-400">Click above</span> or drag and drop
        </p>
        <p className="text-white/60 text-xs mt-1">
          PNG, JPG or GIF (max. 5MB)
        </p>
      </div>
    </div>

    {/* Personal Info Section */}
    <div className="mb-6">
      <h3 className="text-white text-sm font-medium mb-4 uppercase tracking-wider">Personal Info</h3>
      <div className="grid grid-cols-2 gap-4">
        <input
          type="text"
          placeholder="First name"
          value={formData.firstName}
          onChange={(e) => handleInputChange('firstName', e.target.value)}
          className="w-full px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 placeholder-white/60 text-white backdrop-blur-sm transition-all"
        />
        <input
          type="text"
          placeholder="Last name"
          value={formData.lastName}
          onChange={(e) => handleInputChange('lastName', e.target.value)}
          className="w-full px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 placeholder-white/60 text-white backdrop-blur-sm transition-all"
        />
      </div>
    </div>

    {/* Bio Section */}
    <div className="mb-6">
      <h3 className="text-white text-sm font-medium mb-4 uppercase tracking-wider">Bio</h3>
      <textarea
        placeholder="Tell us about yourself..."
        rows={5}
        value={formData.bio}
        onChange={(e) => handleInputChange('bio', e.target.value)}
        className="w-full px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 placeholder-white/60 text-white backdrop-blur-sm transition-all resize-none"
      />
    </div>

    {/* Home Location Section */}
    <div className="mb-6">
      <h3 className="text-white text-sm font-medium mb-4 uppercase tracking-wider">Home Location</h3>
      <select
        value={formData.homeCityId}
        onChange={(e) => handleInputChange('homeCityId', e.target.value)}
        className="w-full px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 text-white backdrop-blur-sm transition-all"
        disabled={isLoadingLocations}
      >
        <option value="" className="bg-black">Select your home location</option>
        {locations.map((location) => (
          <option key={location.id} value={location.id} className="bg-black">
            {location.location}
          </option>
        ))}
      </select>
      {isLoadingLocations && (
        <p className="text-xs text-white/60 mt-2">Loading locations...</p>
      )}
    </div>

    {/* Price Section */}
    {/* <div className="mb-8">
      <h3 className="text-white text-sm font-medium mb-4 uppercase tracking-wider">Price (USD)</h3>
      <div className="grid grid-cols-2 gap-4">
        <div>
          <label className="block text-white/60 text-xs mb-2 uppercase tracking-wider">Daily Rate</label>
          <input
            type="number"
            placeholder="0.00"
            value={formData.projectRate}
            onChange={(e) => handleInputChange('projectRate', e.target.value)}
            step="0.01"
            className="w-full px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 placeholder-white/60 text-white backdrop-blur-sm transition-all"
          />
        </div>
        <div>
          <label className="block text-white/60 text-xs mb-2 uppercase tracking-wider">Hourly Rate</label>
          <input
            type="number"
            placeholder="0.00"
            value={formData.hourlyRate}
            onChange={(e) => handleInputChange('hourlyRate', e.target.value)}
            step="0.01"
            className="w-full px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 placeholder-white/60 text-white backdrop-blur-sm transition-all"
          />
        </div>
      </div>
    </div> */}

    {/* Save Button */}
    <div className="flex justify-center">
      <Button
        type="button"
        variant="fill-white"
        size="lg"
        className="w-full"
        arrowRight={true}
        onClick={handleSaveAccountSettings}
        disabled={isSaving}
      >
        {isSaving ? 'Saving...' : 'Save changes'}
      </Button>
    </div>
  </div>
)}

              {/* Membership Tab */}
              {activeTab === "membership" && (
                <div className="max-w-2xl mx-auto w-full space-y-4 sm:space-y-6 lg:space-y-8">
                  <div className="bg-white/10 backdrop-blur-lg rounded-xl sm:rounded-2xl border border-white/20 p-4 sm:p-6 lg:p-8 shadow-2xl">
                    <h3 className="text-white text-xs sm:text-sm font-medium mb-4 sm:mb-6 uppercase tracking-wider">Membership Level</h3>
                    
                    {isLoadingMembership ? (
                      <div className="text-center py-8">
                        <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
                        <p className="text-sm text-white/60">Loading membership data...</p>
                      </div>
                    ) : (
                      <div className="bg-white/5 border border-white/10 rounded-lg sm:rounded-xl p-4 sm:p-6">
                        <div className="mb-3 sm:mb-4">
                          <span className="inline-block bg-white/20 text-white px-2 sm:px-3 py-1 rounded text-xs font-medium uppercase">
                            {membershipData.plan.name}
                          </span>
                        </div>
                        
                        <div className="mb-4 sm:mb-6">
                          <h4 className="text-xl sm:text-2xl font-bold text-white mb-1 sm:mb-2">
                            ${membershipData.plan.price}/month
                          </h4>
                          <p className="text-xs sm:text-sm text-white/60">Current plan</p>
                        </div>

                        <div className="space-y-2 sm:space-y-3 mb-4 sm:mb-6">
                          <div className="flex items-center gap-2 sm:gap-3 text-xs sm:text-sm text-white">
                            <span className="text-green-400 font-bold">✓</span>
                            <span>{membershipData.plan.points} Points included</span>
                          </div>
                          {membershipData.plan.pm_type && membershipData.plan.pm_last_four && (
                            <div className="flex items-center gap-2 sm:gap-3 text-xs sm:text-sm text-white">
                              <span className="text-green-400 font-bold">✓</span>
                              <span className="uppercase">{membershipData.plan.pm_type} •••• {membershipData.plan.pm_last_four}</span>
                            </div>
                          )}
                          {membershipData.plan.name.toLowerCase() === 'guest' && (
                            <>
                              <div className="flex items-center gap-2 sm:gap-3 text-xs sm:text-sm text-white">
                                <span className="text-green-400 font-bold">✓</span>
                                <span>Must be invited by member</span>
                              </div>
                              <div className="flex items-center gap-2 sm:gap-3 text-xs sm:text-sm text-white">
                                <span className="text-green-400 font-bold">✓</span>
                                <span>Zero guest access</span>
                              </div>
                            </>
                          )}
                        </div>

                        <div className="flex flex-col sm:flex-row gap-2 sm:gap-3">
                          <Button
                            type="button"
                            variant="fill-black"
                            size="md"
                            className="flex-1 uppercase tracking-wide text-xs sm:text-sm"
                            onClick={handleCancelPlan}
                          >
                            Cancel Plan
                          </Button>
                          <Button
                            type="button"
                            variant="fill-white"
                            size="md"
                            className="flex-1 uppercase text-xs sm:text-sm"
                            arrowRight={true}
                            onClick={() => navigate("/change-plan")}
                          >
                            Change Plan
                          </Button>
                        </div>
                      </div>
                    )}
                  </div>

                  <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                    <h3 className="text-white text-sm font-medium mb-6 uppercase tracking-wider">Transactions</h3>
                    <div className="bg-white/5 border border-white/10 rounded-xl overflow-hidden">
                      <table className="w-full">
                        <thead className="bg-white/10">
                          <tr>
                            <th className="text-left py-3 px-4 text-xs font-medium text-white uppercase tracking-wide">Date</th>
                            <th className="text-left py-3 px-4 text-xs font-medium text-white uppercase tracking-wide">Status</th>
                            <th className="text-right py-3 px-4 text-xs font-medium text-white uppercase tracking-wide">Points</th>
                          </tr>
                        </thead>
                        <tbody>
                          {isLoadingMembership ? (
                            <tr>
                              <td colSpan={3} className="py-8 sm:py-12 text-center text-xs sm:text-sm text-white/60">
                                <div className="flex items-center justify-center">
                                  <div className="w-4 h-4 sm:w-5 sm:h-5 border-2 border-white/40 border-t-white animate-spin rounded-full mr-2"></div>
                                  Loading transactions...
                                </div>
                              </td>
                            </tr>
                          ) : membershipData.transactions.length > 0 ? (
                            membershipData.transactions.map((transaction: any) => {
                              // Format action text
                              const actionText = transaction.action.replace(/_/g, ' ').toLowerCase()
                                .split(' ')
                                .map((word: string) => word.charAt(0).toUpperCase() + word.slice(1))
                                .join(' ');
                              
                              // Add plan name if available
                              const statusText = transaction.plan 
                                ? `${actionText} (${transaction.plan.name})`
                                : actionText;
                              
                              // Determine points display
                              const points = transaction.credit || transaction.debit;
                              const isCredit = !!transaction.credit;
                              
                              return (
                                <tr key={transaction.id} className="border-t border-white/10 hover:bg-white/5 transition-colors">
                                  <td className="py-2 sm:py-3 px-2 sm:px-4 text-xs sm:text-sm text-white">
                                    {new Date(transaction.created_at).toLocaleDateString('en-US', {
                                      day: '2-digit',
                                      month: 'short',
                                      year: 'numeric'
                                    })}
                                  </td>
                                  <td className="py-2 sm:py-3 px-2 sm:px-4 text-xs sm:text-sm text-white/80">
                                    {statusText}
                                  </td>
                                  <td className="py-2 sm:py-3 px-2 sm:px-4 text-xs sm:text-sm text-white text-right">
                                    <span className={isCredit ? 'text-green-400' : 'text-red-400'}>
                                      {isCredit ? '+' : '-'}{points} pts
                                    </span>
                                  </td>
                                </tr>
                              );
                            })
                          ) : (
                            <tr>
                              <td colSpan={3} className="py-8 sm:py-12 text-center text-xs sm:text-sm text-white/60">
                                No transactions yet
                              </td>
                            </tr>
                          )}
                        </tbody>
                      </table>
                    </div>
                  </div>
                </div>
              )}

              {/* Points Balance Tab */}
              {activeTab === "points" && (
                <div className="max-w-2xl mx-auto w-full space-y-8">
                  <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                    <div className="flex items-center justify-between mb-6">
                      <h3 className="text-white text-sm font-medium uppercase tracking-wider">Booking Points</h3>
                      <button
                        onClick={fetchPointsBalance}
                        disabled={isLoadingPoints}
                        className="text-white/60 hover:text-white transition-colors disabled:opacity-50"
                        title="Refresh points balance"
                      >
                        <svg className={`w-5 h-5 ${isLoadingPoints ? 'animate-spin' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                        </svg>
                      </button>
                    </div>
                    
                    {isLoadingPoints ? (
                      <div className="text-center py-8">
                        <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
                        <p className="text-sm text-white/60">Loading points data...</p>
                      </div>
                    ) : (
                      <div className="bg-white/5 border border-white/10 rounded-xl p-6">
                        <div className="space-y-3 mb-6">
                          <div className="flex justify-between items-center py-2">
                            <span className="text-sm text-white/60">Last month</span>
                            <span className="text-sm font-medium text-white">{pointsBalance.lastMonthBookingCost} pts</span>
                          </div>
                          <div className="flex justify-between items-center py-2">
                            <span className="text-sm text-white/60">Points used</span>
                            <span className="text-sm font-medium text-white">{pointsBalance.currentMonthBookingCost} pts</span>
                          </div>
                          <div className="flex justify-between items-center py-2 border-t border-white/10 pt-3">
                            <span className="text-sm text-white/60">Current balance</span>
                            <span className="text-lg font-bold text-white">{pointsBalance.currentBalance} pts</span>
                          </div>
                        </div>
                        
                        <Button
                          type="button"
                          variant="fill-white"
                          size="md"
                          className="w-full uppercase"
                          arrowRight={true}
                          onClick={() => setShowBuyPointsModal(true)}
                        >
                          Buy Points
                        </Button>
                      </div>
                    )}
                  </div>

                  <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                    <h3 className="text-white text-sm font-medium mb-6 uppercase tracking-wider">Payout Points</h3>
                    
                    {isLoadingPoints ? (
                      <div className="text-center py-8">
                        <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
                        <p className="text-sm text-white/60">Loading payout data...</p>
                      </div>
                    ) : (
                      <div className="bg-white/5 border border-white/10 rounded-xl p-6">
                        <div className="space-y-3 mb-6">
                          <div className="flex justify-between items-center py-2">
                            <span className="text-sm text-white/60">Last month</span>
                            <span className="text-sm font-medium text-white">{pointsBalance.lastMonthBookingCost} pts</span>
                          </div>
                          <div className="flex justify-between items-center py-2">
                            <span className="text-sm text-white/60">Points used</span>
                            <span className="text-sm font-medium text-white">{pointsBalance.currentMonthBookingCost} pts</span>
                          </div>
                          <div className="flex justify-between items-center py-2 border-t border-white/10 pt-3">
                            <span className="text-sm text-white/60">Current balance</span>
                            <span className="text-lg font-bold text-white">{pointsBalance.currentBalance} pts</span>
                          </div>
                        </div>
                        
                        <Button
                          type="button"
                          variant="fill-white"
                          size="md"
                          className="w-full uppercase opacity-50 cursor-not-allowed"
                          arrowRight={true}
                          disabled
                        >
                          Payout Info
                        </Button>
                      </div>
                    )}
                  </div>

                  <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                    <h3 className="text-white text-sm font-medium mb-6 uppercase tracking-wider">Transactions</h3>
                    <div className="bg-white/5 border border-white/10 rounded-xl overflow-hidden">
                      <table className="w-full">
                        <thead className="bg-white/10">
                          <tr>
                            <th className="text-left py-3 px-4 text-xs font-medium text-white uppercase tracking-wide">Date</th>
                            <th className="text-left py-3 px-4 text-xs font-medium text-white uppercase tracking-wide">Status</th>
                            <th className="text-right py-3 px-4 text-xs font-medium text-white uppercase tracking-wide">Points</th>
                          </tr>
                        </thead>
                        <tbody>
                          {isLoadingPoints ? (
                            <tr>
                              <td colSpan={3} className="py-12 text-center text-sm text-white/60">
                                <div className="flex items-center justify-center">
                                  <div className="w-5 h-5 border-2 border-white/40 border-t-white animate-spin rounded-full mr-2"></div>
                                  Loading transactions...
                                </div>
                              </td>
                            </tr>
                          ) : pointsBalance.points && pointsBalance.points.length > 0 ? (
                            pointsBalance.points.map((transaction: any, index: number) => {
                              // Get formatted action text
                              const actionText = transaction.action && pointsBalance.pointsAction[transaction.action]
                                ? pointsBalance.pointsAction[transaction.action]
                                : (transaction.action || 'Unknown').replace(/_/g, ' ').toLowerCase()
                                    .split(' ')
                                    .map((word: string) => word.charAt(0).toUpperCase() + word.slice(1))
                                    .join(' ');
                              
                              // Determine if credit or debit
                              const points = transaction.credit || transaction.debit || transaction.points || 0;
                              const isCredit = !!transaction.credit || (transaction.points && transaction.points > 0);
                              
                              return (
                                <tr key={index} className="border-t border-white/10 hover:bg-white/5 transition-colors">
                                  <td className="py-3 px-4 text-sm text-white">
                                    {transaction.created_at 
                                      ? new Date(transaction.created_at).toLocaleDateString('en-US', {
                                          day: '2-digit',
                                          month: 'short',
                                          year: 'numeric'
                                        })
                                      : (transaction.date ? new Date(transaction.date).toLocaleDateString('en-US', {
                                          day: '2-digit',
                                          month: 'short',
                                          year: 'numeric'
                                        }) : '-')
                                    }
                                  </td>
                                  <td className="py-3 px-4 text-sm text-white/80">
                                    {actionText}
                                  </td>
                                  <td className="py-3 px-4 text-sm text-white text-right">
                                    <span className={isCredit ? 'text-green-400' : 'text-red-400'}>
                                      {isCredit ? '+' : '-'}{Math.abs(points)} pts
                                    </span>
                                  </td>
                                </tr>
                              );
                            })
                          ) : (
                            <tr>
                              <td colSpan={3} className="py-12 text-center text-sm text-white/60">
                                No transactions yet
                              </td>
                            </tr>
                          )}
                        </tbody>
                      </table>
                    </div>
                  </div>
                </div>
              )}

              {/* Cards Tab */}
              {activeTab === "cards" && (
                <div className="max-w-2xl mx-auto w-full">
                  <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                    <h3 className="text-white text-sm font-medium mb-8 uppercase tracking-wider">Payment Methods</h3>
                    
                    <Button
                      type="button"
                      variant="fill-white"
                      size="md"
                      className="w-full mb-6"
                      onClick={() => navigate("/add-payment-method")}
                    >
                      + Add Payment Method
                    </Button>

                    <div className="border-t border-white/20 pt-6">
                      {isLoadingPaymentMethods ? (
                        <div className="text-center py-8">
                          <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
                          <p className="text-sm text-white/60">Loading payment methods...</p>
                        </div>
                      ) : paymentMethods.length > 0 ? (
                        <div className="space-y-4">
                          {paymentMethods.map((method) => (
                            <div key={method.id} className="bg-white/5 border border-white/10 rounded-xl p-6 hover:bg-white/10 transition-colors">
                              <div className="flex items-center justify-between">
                                <div className="flex items-center gap-4">
                                  {/* Card Brand Icon */}
                                  <div className="w-12 h-12 bg-white/20 rounded-lg flex items-center justify-center">
                                    <span className="text-2xl">
                                      {method.card.brand === 'visa' ? '💳' : 
                                       method.card.brand === 'mastercard' ? '💳' : 
                                       method.card.brand === 'amex' ? '💳' : '💳'}
                                    </span>
                                  </div>
                                  
                                  <div>
                                    <div className="flex items-center gap-2 mb-1">
                                      <p className="text-white font-medium uppercase">
                                        {method.card.brand} •••• {method.card.last4}
                                      </p>
                                      {defaultPaymentMethod?.id === method.id && (
                                        <span className="bg-green-500/20 text-green-400 px-2 py-0.5 rounded text-xs font-medium uppercase">
                                          Default
                                        </span>
                                      )}
                                    </div>
                                    <p className="text-white/60 text-sm">
                                      {method.billing_details.name || 'No name'}
                                    </p>
                                    <p className="text-white/40 text-xs mt-1">
                                      Expires {String(method.card.exp_month).padStart(2, '0')}/{method.card.exp_year}
                                    </p>
                                  </div>
                                </div>
                                
                                <button
                                  onClick={() => handleDeletePaymentMethod(method.id)}
                                  className="text-white/40 hover:text-red-400 transition-colors p-2"
                                  title="Delete payment method"
                                >
                                  <Trash2 className="w-5 h-5" />
                                </button>
                              </div>
                            </div>
                          ))}
                        </div>
                      ) : (
                        <div className="text-center py-8">
                          <p className="text-sm text-white/60">No payment methods added yet</p>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              )}

              {/* Security Tab */}
              {activeTab === "security" && (
                <div className="max-w-2xl mx-auto w-full">
                  <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                    <div>
                      <h3 className="text-white text-sm font-medium mb-8 uppercase tracking-wider">Security Settings</h3>
                      

                      
                      {/* Profile Picture Display */}
                      {userProfile.profilePicture && (
                        <div className="flex justify-center mb-6">
                          <div className="w-20 h-20 rounded-full overflow-hidden bg-white/10">
                            <img 
                              src={userProfile.profilePicture} 
                              alt="Profile" 
                              className="w-full h-full object-cover"
                            />
                          </div>
                        </div>
                      )}
                      
                      <div className="space-y-6 mb-8">
                        {isLoadingProfile && (
                          <div className="text-center py-8">
                            <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
                            <p className="text-white/60 text-sm">Loading profile data...</p>
                          </div>
                        )}
                        
                        <div>
                          <label className="block text-sm text-white mb-2 uppercase tracking-wider">Mobile Number</label>
                          <div className="relative">
                            <div className="flex items-center px-4 py-4 bg-white/10 border border-white/20 rounded-lg focus-within:ring-2 focus-within:ring-white/40 focus-within:border-white/40">
                              <button
                                type="button"
                                onClick={() => setIsDropdownOpen(!isDropdownOpen)}
                                className="flex items-center text-sm text-white/60 mr-2 hover:text-white/80 transition-colors"
                              >
                                <span className="mr-1">
                                  {(() => {
                                    const currentCountry = countryCodes.find(c => c.code === userProfile.countryCode);
                                    return currentCountry ? currentCountry.flag : selectedCountry.flag;
                                  })()}
                                </span>
                                <span className="mr-1">{userProfile.countryCode}</span>
                                <ChevronDown className="w-3 h-3" />
                              </button>
                              <input 
                                type="tel" 
                                value={userProfile.phoneNumber}
                                onChange={(e) => {
                                  const value = e.target.value.replace(/\D/g, ''); // Only allow digits
                                  setUserProfile(prev => ({ ...prev, phoneNumber: value }));
                                }}
                                placeholder="Enter your mobile number"
                                className="flex-1 border-0 outline-none text-sm text-white placeholder:text-white/60 bg-transparent"
                              />
                            </div>
                            
                            {isDropdownOpen && (
                              <div className="absolute top-full left-0 right-0 mt-1 bg-black border border-white/20 rounded-lg max-h-60 overflow-y-auto z-50 shadow-xl">
                                {countryCodes.map((country) => (
                                  <button
                                    key={country.code}
                                    type="button"
                                    data-country={country.country}
                                    onClick={() => {
                                      console.log('Selected country:', country);
                                      setSelectedCountry(country);
                                      setUserProfile(prev => {
                                        console.log('Updating userProfile countryCode to:', country.code);
                                        return { ...prev, countryCode: country.code };
                                      });
                                      setIsDropdownOpen(false);
                                    }}
                                    className="w-full flex items-center px-4 py-3 text-sm text-white hover:bg-white/10 transition-colors text-left"
                                  >
                                    <span className="mr-3">{country.flag}</span>
                                    <span className="mr-3 font-mono">{country.code}</span>
                                    <span>{country.country}</span>
                                  </button>
                                ))}
                              </div>
                            )}
                          </div>
                        </div>

                        <div>
                          <label className="block text-sm text-white mb-2 uppercase tracking-wider">Email</label>
                          <input 
                            type="email" 
                            value={userProfile.email}
                            onChange={(e) => setUserProfile(prev => ({ ...prev, email: e.target.value }))}
                            placeholder="your.email@gmail.com"
                            className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-lg text-white placeholder:text-white/60 text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                          />
                        </div>

                        <div>
                          <button 
                            className="text-sm text-white hover:text-white/80 flex items-center gap-2 underline uppercase tracking-wider transition-colors"
                            onClick={() => navigate("/reset-password")}
                          >
                            Forgot Password?
                          </button>
                        </div>
                      </div>
                    </div>

                    <div className="flex justify-center">
                      <Button
                        type="button"
                        variant="fill-white"
                        size="lg"
                        className="w-full"
                        arrowRight={true}
                        onClick={handleSaveSecuritySettings}
                        disabled={isSendingOtp}
                      >
                        {isSendingOtp ? 'Sending...' : 'Save Changes'}
                      </Button>
                    </div>
                  </div>
                </div>
              )}

              {/* Calendar Tab */}
              {activeTab === "calendar" && (
                <div className="w-full">
                  <div className="grid grid-cols-1 lg:grid-cols-3 gap-4 sm:gap-6 lg:gap-8">
                    {/* Calendar Section */}
                    <div className="lg:col-span-2">
                      <div className="bg-white/10 backdrop-blur-lg rounded-xl sm:rounded-2xl border border-white/20 p-3 sm:p-4 lg:p-6 shadow-2xl">
                        {/* Calendar Header */}
                        <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-3 sm:gap-4 mb-4 sm:mb-6">
                          <div className="flex items-center gap-2 sm:gap-4 w-full sm:w-auto">
                            <button 
                              onClick={() => setCurrentDate(new Date())}
                              className="bg-white/20 hover:bg-white/30 text-white px-2 sm:px-3 py-1 sm:py-1.5 rounded text-xs sm:text-sm uppercase tracking-wide transition-colors"
                            >
                              today
                            </button>
                            <div className="flex gap-1">
                              <button 
                                onClick={() => navigateMonth('prev')}
                                className="bg-white/20 hover:bg-white/30 text-white p-1.5 sm:p-2 rounded transition-colors"
                              >
                                <ChevronDown className="w-3 h-3 sm:w-4 sm:h-4 rotate-90" />
                              </button>
                              <button 
                                onClick={() => navigateMonth('next')}
                                className="bg-white/20 hover:bg-white/30 text-white p-1.5 sm:p-2 rounded transition-colors"
                              >
                                <ChevronDown className="w-3 h-3 sm:w-4 sm:h-4 -rotate-90" />
                              </button>
                            </div>
                            <h2 className="text-sm sm:text-lg font-bold text-white uppercase tracking-wide">{formatMonthYear(currentDate)}</h2>
                          </div>
                          <div className="flex gap-1 overflow-x-auto scrollbar-hide w-full sm:w-auto">
                            <button 
                              onClick={() => setCalendarView('schedule')}
                              className={`px-2 sm:px-3 py-1 sm:py-1.5 rounded text-xs sm:text-sm uppercase tracking-wide font-medium transition-colors whitespace-nowrap ${
                                calendarView === 'schedule' ? 'bg-white text-black' : 'bg-white/20 hover:bg-white/30 text-white'
                              }`}
                            >
                              Schedule
                            </button>
                            <button 
                              onClick={() => setCalendarView('month')}
                              className={`px-2 sm:px-3 py-1 sm:py-1.5 rounded text-xs sm:text-sm uppercase tracking-wide font-medium transition-colors whitespace-nowrap ${
                                calendarView === 'month' ? 'bg-white text-black' : 'bg-white/20 hover:bg-white/30 text-white'
                              }`}
                            >
                              month
                            </button>
                            <button 
                              onClick={() => setCalendarView('week')}
                              className={`px-2 sm:px-3 py-1 sm:py-1.5 rounded text-xs sm:text-sm uppercase tracking-wide font-medium transition-colors whitespace-nowrap ${
                                calendarView === 'week' ? 'bg-white text-black' : 'bg-white/20 hover:bg-white/30 text-white'
                              }`}
                            >
                              week
                            </button>
                            <button 
                              onClick={() => setCalendarView('day')}
                              className={`px-2 sm:px-3 py-1 sm:py-1.5 rounded text-xs sm:text-sm uppercase tracking-wide font-medium transition-colors whitespace-nowrap ${
                                calendarView === 'day' ? 'bg-white text-black' : 'bg-white/20 hover:bg-white/30 text-white'
                              }`}
                            >
                              day
                            </button>
                          </div>
                        </div>

                        {/* Calendar Content based on view */}
                        <div className="bg-white/5 border border-white/10 rounded-xl p-4">
                          {calendarView === 'schedule' && (
                            <div className="max-h-[600px] overflow-y-auto">
                              {isLoadingCalendar ? (
                                <div className="text-center py-20">
                                  <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
                                  <p className="text-sm text-white/60">Loading schedules...</p>
                                </div>
                              ) : calendarEvents.length > 0 ? (
                                <div className="space-y-0">
                                  {(() => {
                                    // Sort events by start date first
                                    const sortedEvents = [...calendarEvents].sort((a, b) => 
                                      new Date(a.start).getTime() - new Date(b.start).getTime()
                                    );
                                    
                                    // Group events by date (using UTC to avoid timezone issues)
                                    const groupedEvents: Record<string, any[]> = {};
                                    sortedEvents.forEach(event => {
                                      // Extract date in YYYY-MM-DD format from the ISO string
                                      const eventDateStr = event.start.split('T')[0]; // e.g., "2025-11-12"
                                      const eventDate = new Date(eventDateStr + 'T12:00:00'); // Use noon to avoid timezone issues
                                      
                                      const dateKey = eventDate.toLocaleDateString('en-US', {
                                        year: 'numeric',
                                        month: 'long',
                                        day: 'numeric'
                                      });
                                      
                                      if (!groupedEvents[dateKey]) {
                                        groupedEvents[dateKey] = [];
                                      }
                                      groupedEvents[dateKey].push(event);
                                    });

                                    return Object.entries(groupedEvents).map(([dateKey, events]) => {
                                      const firstEvent = events[0];
                                      const eventDateStr = firstEvent.start.split('T')[0];
                                      const eventDate = new Date(eventDateStr + 'T12:00:00');
                                      const dayName = eventDate.toLocaleDateString('en-US', { weekday: 'long' });
                                      
                                      return (
                                        <div key={dateKey} className="border-b border-white/10 last:border-b-0">
                                          {/* Date Header */}
                                          <div className="flex items-center justify-between py-4 px-2">
                                            <div>
                                              <h3 className="text-white font-medium text-base">{dateKey}</h3>
                                            </div>
                                            <div className="text-white/60 text-sm">{dayName}</div>
                                          </div>
                                          
                                          {/* Events for this date */}
                                          <div className="space-y-2 pb-4">
                                            {events.map((event) => {
                                              // Format time without timezone conversion
                                              const formatTime = (isoString: string) => {
                                                const timePart = isoString.split('T')[1]; // "09:00:00"
                                                if (!timePart) return '';
                                                const [hours, minutes] = timePart.split(':');
                                                const hour = parseInt(hours);
                                                const ampm = hour >= 12 ? 'pm' : 'am';
                                                const displayHour = hour % 12 || 12;
                                                return `${displayHour}:${minutes} ${ampm}`;
                                              };
                                              
                                              return (
                                                <div 
                                                  key={event.id}
                                                  className="flex items-center gap-3 px-2 py-2 hover:bg-white/5 transition-colors rounded"
                                                >
                                                  {/* Time */}
                                                  <div className="flex items-center gap-2 min-w-[140px]">
                                                    <div 
                                                      className="w-2 h-2 rounded-full"
                                                      style={{ backgroundColor: event.color || '#99CCFF' }}
                                                    />
                                                    <span className="text-white text-sm">
                                                      {formatTime(event.start)}
                                                      {' - '}
                                                      {formatTime(event.end)}
                                                    </span>
                                                  </div>
                                                  
                                                  {/* Title */}
                                                  <div className="flex-1">
                                                    <span className="text-white/80 text-sm">{event.title}</span>
                                                  </div>
                                                </div>
                                              );
                                            })}
                                          </div>
                                        </div>
                                      );
                                    });
                                  })()}
                                </div>
                              ) : (
                                <div className="text-center text-white/60 text-sm py-20">
                                  No events to display
                                </div>
                              )}
                            </div>
                          )}

                          {calendarView === 'month' && (
                            <div className="max-h-[600px] overflow-y-auto">
                              {/* Month View Header */}
                              <div className="grid grid-cols-7 gap-1 mb-2 sticky top-0 bg-[#121212] z-10">
                                {['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'].map((day) => (
                                  <div key={day} className="text-center text-white/60 text-sm font-medium py-2">
                                    {day}
                                  </div>
                                ))}
                              </div>
                              {/* Month Grid */}
                              <div className="grid grid-cols-7 gap-1">
                                {generateCalendarGrid().map((date, index) => {
                                  const isCurrentMonth = date.getMonth() === currentDate.getMonth();
                                  const isToday = date.toDateString() === new Date().toDateString();
                                  
                                  // Format date as YYYY-MM-DD WITHOUT timezone conversion
                                  const year = date.getFullYear();
                                  const month = String(date.getMonth() + 1).padStart(2, '0');
                                  const day = String(date.getDate()).padStart(2, '0');
                                  const dateStr = `${year}-${month}-${day}`;
                                  
                                  // Check if this date has any events - use same date extraction as schedule view
                                  const dayEvents = calendarEvents.filter(event => {
                                    const eventDateStr = event.start.split('T')[0];
                                    return eventDateStr === dateStr;
                                  });
                                  const hasEvents = dayEvents.length > 0;
                                  
                                  // Helper to format time from ISO string without timezone conversion
                                  const formatEventTime = (isoString: string) => {
                                    const timePart = isoString.split('T')[1]; // "09:00:00"
                                    if (!timePart) return '';
                                    const [hours, minutes] = timePart.split(':');
                                    const hour = parseInt(hours);
                                    const ampm = hour >= 12 ? 'PM' : 'AM';
                                    const displayHour = hour % 12 || 12;
                                    return `${displayHour}:${minutes} ${ampm}`;
                                  };
                                  
                                  return (
                                    <div 
                                      key={index}
                                      className={`aspect-square flex flex-col items-center justify-center text-sm border border-white/10 relative hover:bg-white/10 transition-colors cursor-pointer ${
                                        isCurrentMonth ? 'text-white' : 'text-white/30'
                                      } ${isToday ? 'bg-white/20' : ''} ${hasEvents ? 'bg-blue-500/10' : ''}`}
                                      title={hasEvents ? `${dayEvents.length} event(s)` : ''}
                                    >
                                      <span className="mb-1">{date.getDate()}</span>
                                      {hasEvents && (
                                        <div className="absolute bottom-2 flex gap-0.5">
                                          {dayEvents.slice(0, 3).map((event, i) => (
                                            <div 
                                              key={i}
                                              className="w-1.5 h-1.5 rounded-full"
                                              style={{ backgroundColor: event.color || '#99CCFF' }}
                                              title={`${event.title || 'Event'} - ${formatEventTime(event.start)}`}
                                            />
                                          ))}
                                          {dayEvents.length > 3 && (
                                            <span className="text-[10px] text-white/60 ml-0.5">+{dayEvents.length - 3}</span>
                                          )}
                                        </div>
                                      )}
                                    </div>
                                  );
                                })}
                              </div>
                            </div>
                          )}

                          {calendarView === 'week' && (
                            <div className="overflow-x-auto">
                              {(() => {
                                const { days, hours } = generateWeekView();
                                
                                // Helper to check if an event falls in a specific day and hour
                                const getEventsForSlot = (day: Date, hourStr: string) => {
                                  const hourMap: Record<string, number> = {
                                    '6am': 6, '7am': 7, '8am': 8, '9am': 9, '10am': 10, '11am': 11,
                                    '12pm': 12, '1pm': 13, '2pm': 14, '3pm': 15, '4pm': 16, '5pm': 17, '6pm': 18, '7pm': 19
                                  };
                                  const slotHour = hourMap[hourStr];
                                  
                                  const dayStr = day.toISOString().split('T')[0];
                                  
                                  return calendarEvents.filter(event => {
                                    const eventStart = new Date(event.start);
                                    const eventStartDate = eventStart.toISOString().split('T')[0];
                                    const eventStartHour = eventStart.getHours();
                                    
                                    // Check if event starts in this day and hour slot
                                    return eventStartDate === dayStr && eventStartHour === slotHour;
                                  });
                                };
                                
                                return (
                                  <div className="min-w-full">
                                    {/* Week Header */}
                                    <div className="grid grid-cols-8 gap-1 mb-2 min-w-full">
                                      <div className="text-center text-white/60 text-sm font-medium py-2"></div>
                                      {days.map((day, index) => {
                                        const isToday = day.toDateString() === new Date().toDateString();
                                        return (
                                          <div 
                                            key={index} 
                                            className={`text-center text-white/60 text-sm font-medium py-2 ${isToday ? 'text-white font-bold' : ''}`}
                                          >
                                            {day.toLocaleDateString('en-US', { weekday: 'short' })} {day.getMonth() + 1}/{day.getDate()}
                                          </div>
                                        );
                                      })}
                                    </div>
                                    {/* Week Grid */}
                                    {hours.map((hour, hourIndex) => (
                                      <div key={hour} className="grid grid-cols-8 gap-1 min-w-full">
                                        <div className="text-white/60 text-sm py-2 pr-2 text-right">
                                          {hour}
                                        </div>
                                        {days.map((day, dayIndex) => {
                                          const slotEvents = getEventsForSlot(day, hour);
                                          const hasEvent = slotEvents.length > 0;
                                          const isToday = day.toDateString() === new Date().toDateString();
                                          
                                          return (
                                            <div 
                                              key={`${dayIndex}-${hour}`}
                                              className={`h-12 border border-white/10 relative ${
                                                isToday ? 'bg-white/5' : ''
                                              }`}
                                            >
                                              {hasEvent && slotEvents.map((event, eventIndex) => (
                                                <div 
                                                  key={eventIndex}
                                                  className="absolute inset-0 overflow-hidden"
                                                  style={{ backgroundColor: event.color || '#99CCFF', opacity: 0.7 }}
                                                  title={event.title || 'Event'}
                                                >
                                                  <div className="text-[10px] text-white px-1 py-0.5 truncate font-medium">
                                                    {event.title || 'Busy'}
                                                  </div>
                                                </div>
                                              ))}
                                            </div>
                                          );
                                        })}
                                      </div>
                                    ))}
                                  </div>
                                );
                              })()}
                            </div>
                          )}

                          {calendarView === 'day' && (
                            <div>
                              {/* Day Header */}
                              <div className="text-center text-white font-medium py-4 text-lg">
                                {currentDate.toLocaleDateString('en-US', { weekday: 'long', month: 'short', day: 'numeric', year: 'numeric' })}
                              </div>
                              {/* Day View */}
                              <div className="overflow-y-auto max-h-96">
                                {(() => {
                                  const hours = ['all-day', '6am', '7am', '8am', '9am', '10am', '11am', '12pm', '1pm', '2pm', '3pm', '4pm', '5pm', '6pm', '7pm'];
                                  const hourMap: Record<string, number> = {
                                    '6am': 6, '7am': 7, '8am': 8, '9am': 9, '10am': 10, '11am': 11,
                                    '12pm': 12, '1pm': 13, '2pm': 14, '3pm': 15, '4pm': 16, '5pm': 17, '6pm': 18, '7pm': 19
                                  };
                                  
                                  const dayStr = currentDate.toISOString().split('T')[0];
                                  
                                  // Get all-day events
                                  const allDayEvents = calendarEvents.filter(event => {
                                    const eventDate = new Date(event.start).toISOString().split('T')[0];
                                    return eventDate === dayStr;
                                  });
                                  
                                  return hours.map((time, index) => {
                                    let slotEvents: any[] = [];
                                    
                                    if (time !== 'all-day') {
                                      const slotHour = hourMap[time];
                                      slotEvents = calendarEvents.filter(event => {
                                        const eventStart = new Date(event.start);
                                        const eventStartDate = eventStart.toISOString().split('T')[0];
                                        const eventStartHour = eventStart.getHours();
                                        
                                        return eventStartDate === dayStr && eventStartHour === slotHour;
                                      });
                                    } else {
                                      slotEvents = allDayEvents;
                                    }
                                    
                                    const hasEvent = slotEvents.length > 0;
                                    
                                    return (
                                      <div key={time} className="flex border-b border-white/10">
                                        <div className="w-16 text-white/60 text-sm py-3 pr-2 text-right">
                                          {time}
                                        </div>
                                        <div className={`flex-1 min-h-12 relative`}>
                                          {hasEvent && slotEvents.map((event, eventIndex) => (
                                            <div 
                                              key={eventIndex}
                                              className="absolute inset-0 m-0.5"
                                              style={{ backgroundColor: event.color || '#99CCFF', opacity: 0.8 }}
                                              title={event.title || 'Event'}
                                            >
                                              <div className="text-xs text-white px-2 py-1 font-medium">
                                                {event.title || 'Busy'}
                                              </div>
                                              <div className="text-[10px] text-white/80 px-2">
                                                {new Date(event.start).toLocaleTimeString('en-US', { 
                                                  hour: 'numeric',
                                                  minute: '2-digit',
                                                  hour12: true
                                                })}
                                                {' - '}
                                                {new Date(event.end).toLocaleTimeString('en-US', { 
                                                  hour: 'numeric',
                                                  minute: '2-digit',
                                                  hour12: true
                                                })}
                                              </div>
                                            </div>
                                          ))}
                                        </div>
                                      </div>
                                    );
                                  });
                                })()}
                              </div>
                            </div>
                          )}
                        </div>
                      </div>
                    </div>

                    {/* Availability & Prices Section */}
                    <div className="space-y-6">
                      <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-6 shadow-2xl">
                        <h3 className="text-white text-sm font-medium mb-6 uppercase tracking-wider">Availability & prices</h3>
                        
                        {isLoadingCalendar ? (
                          <div className="text-center py-8">
                            <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
                            <p className="text-sm text-white/60">Loading calendar settings...</p>
                          </div>
                        ) : (
                          <>
                            {/* Location Selector */}
                            <div className="mb-4">
                              <label className="block text-white text-xs font-medium mb-2 uppercase tracking-wider">Location</label>
                              <select 
                                className="w-full bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                                value={calendarSettings.schedule_location}
                                onChange={(e) => {
                                  setCalendarSettings(prev => ({ ...prev, schedule_location: e.target.value }));
                                  if (e.target.value && calendarSettings.refrence_id) {
                                    fetchCalendarEvents(calendarSettings.refrence_id, e.target.value);
                                  }
                                }}
                              >
                                {Object.keys(calendarSettings.locations).length === 0 ? (
                                  <option value="" className="bg-gray-800">No locations available</option>
                                ) : (
                                  Object.entries(calendarSettings.locations).map(([id, name]) => (
                                    <option key={id} value={id} className="bg-gray-800">
                                      {name}
                                    </option>
                                  ))
                                )}
                              </select>
                            </div>
                        
                            {/* Minimum Booking Time */}
                            <div className="mb-4">
                              <label className="block text-white text-xs font-medium mb-2 uppercase tracking-wider">Minimum Booking Time</label>
                              <div className="grid grid-cols-3 gap-2">
                                <select 
                                  className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                                  value={calendarSettings.min_booking.min_days}
                                  onChange={(e) => setCalendarSettings(prev => ({
                                    ...prev,
                                    min_booking: { ...prev.min_booking, min_days: e.target.value }
                                  }))}
                                >
                                  <option value="0" className="bg-gray-800">0 Days</option>
                                  {Array.from({ length: 30 }, (_, i) => (
                                    <option key={i + 1} value={String(i + 1).padStart(2, '0')} className="bg-gray-800">
                                      {String(i + 1).padStart(2, '0')} Days
                                    </option>
                                  ))}
                                </select>
                                <select 
                                  className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                                  value={calendarSettings.min_booking.min_hours}
                                  onChange={(e) => setCalendarSettings(prev => ({
                                    ...prev,
                                    min_booking: { ...prev.min_booking, min_hours: e.target.value }
                                  }))}
                                >
                                  <option value="00" className="bg-gray-800">00 Hours</option>
                                  {Array.from({ length: 23 }, (_, i) => (
                                    <option key={i + 1} value={String(i + 1).padStart(2, '0')} className="bg-gray-800">
                                      {String(i + 1).padStart(2, '0')} Hours
                                    </option>
                                  ))}
                                </select>
                                <select 
                                  className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                                  value={calendarSettings.min_booking.min_minutes}
                                  onChange={(e) => setCalendarSettings(prev => ({
                                    ...prev,
                                    min_booking: { ...prev.min_booking, min_minutes: e.target.value }
                                  }))}
                                >
                                  <option value="00" className="bg-gray-800">00 Min</option>
                                  <option value="15" className="bg-gray-800">15 Min</option>
                                  <option value="30" className="bg-gray-800">30 Min</option>
                                  <option value="45" className="bg-gray-800">45 Min</option>
                                </select>
                              </div>
                            </div>

                            {/* Maximum Booking Time */}
                            <div className="mb-4">
                              <label className="block text-white text-xs font-medium mb-2 uppercase tracking-wider">Maximum Booking Time</label>
                              <div className="grid grid-cols-3 gap-2">
                                <select 
                                  className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                                  value={calendarSettings.max_booking.max_days}
                                  onChange={(e) => setCalendarSettings(prev => ({
                                    ...prev,
                                    max_booking: { ...prev.max_booking, max_days: e.target.value }
                                  }))}
                                >
                                  <option value="0" className="bg-gray-800">0 Days</option>
                                  {Array.from({ length: 30 }, (_, i) => (
                                    <option key={i + 1} value={String(i + 1).padStart(2, '0')} className="bg-gray-800">
                                      {String(i + 1).padStart(2, '0')} Days
                                    </option>
                                  ))}
                                </select>
                                <select 
                                  className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                                  value={calendarSettings.max_booking.max_hours}
                                  onChange={(e) => setCalendarSettings(prev => ({
                                    ...prev,
                                    max_booking: { ...prev.max_booking, max_hours: e.target.value }
                                  }))}
                                >
                                  <option value="00" className="bg-gray-800">00 Hours</option>
                                  {Array.from({ length: 23 }, (_, i) => (
                                    <option key={i + 1} value={String(i + 1).padStart(2, '0')} className="bg-gray-800">
                                      {String(i + 1).padStart(2, '0')} Hours
                                    </option>
                                  ))}
                                </select>
                                <select 
                                  className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                                  value={calendarSettings.max_booking.max_minutes}
                                  onChange={(e) => setCalendarSettings(prev => ({
                                    ...prev,
                                    max_booking: { ...prev.max_booking, max_minutes: e.target.value }
                                  }))}
                                >
                                  <option value="00" className="bg-gray-800">00 Min</option>
                                  <option value="15" className="bg-gray-800">15 Min</option>
                                  <option value="30" className="bg-gray-800">30 Min</option>
                                  <option value="45" className="bg-gray-800">45 Min</option>
                                </select>
                              </div>
                            </div>

                        {/* Availability Type Dropdown */}
                        <div className="mb-6">
                          <select 
                            className="w-full bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                            value={availabilityType}
                            onChange={(e) => setAvailabilityType(e.target.value as 'daily' | 'recurring')}
                          >
                            <option value="daily" className="bg-gray-800">Daily availability</option>
                            <option value="recurring" className="bg-gray-800">Recurring availability</option>
                          </select>
                        </div>

                        {/* Availability Section */}
                        {availabilityType === 'daily' && (
                          <div className="mb-6">
                            {/* Header with labels */}
                            <div className="flex items-center gap-2 mb-4">
                              <div className="w-4 h-4 border-2 border-white/40 rounded-full flex items-center justify-center">
                                <div className="w-2 h-2 bg-white rounded-full"></div>
                              </div>
                              <span className="text-white text-sm font-medium uppercase tracking-wider">Daily Availability</span>
                            </div>
                            
                            {/* Column Headers */}
                            <div className="overflow-x-auto">
                              <div className="min-w-max">
                                <div className="flex items-center gap-3 mb-2 text-white/60 text-sm">
                                  <div style={{ width: '140px' }}>Date</div> {/* Date column space */}
                                  <div style={{ width: '100px' }}>Start time</div>
                                  <div style={{ width: '100px' }}>End time</div>
                                  <div style={{ width: '80px' }}>Daily (USD)</div>
                                  <div style={{ width: '80px' }}>Hourly (USD)</div>
                                  {/* <div style={{ width: '50px' }}>+</div> */}
                                </div>
                                
                                {/* Availability Rows */}
                                <div className="space-y-2">
                                  {availabilityRows.map((row, index) => (
                                    <div key={row.id} className="flex items-center gap-3">
                                      <input 
                                        type="date" 
                                        value={row.date}
                                        onChange={(e) => updateAvailabilityRow(row.id, 'date', e.target.value)}
                                        className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                        style={{ width: '140px' }}
                                        placeholder="dd/mm/yyyy"
                                      />
                                      <input 
                                        type="text" 
                                        value={row.startTime}
                                        onChange={(e) => updateAvailabilityRow(row.id, 'startTime', e.target.value)}
                                        className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                        style={{ width: '100px' }}
                                        placeholder="09:00 AM"
                                      />
                                      <input 
                                        type="text" 
                                        value={row.endTime}
                                        onChange={(e) => updateAvailabilityRow(row.id, 'endTime', e.target.value)}
                                        className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                        style={{ width: '100px' }}
                                        placeholder="05:00 PM"
                                      />
                                      <input 
                                        type="number" 
                                        step="0.01" 
                                        value={row.dailyRate}
                                        onChange={(e) => updateAvailabilityRow(row.id, 'dailyRate', e.target.value)}
                                        className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                        style={{ width: '80px' }}
                                      />
                                      <input 
                                        type="number" 
                                        step="0.01" 
                                        value={row.hourlyRate}
                                        onChange={(e) => updateAvailabilityRow(row.id, 'hourlyRate', e.target.value)}
                                        className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                        style={{ width: '80px' }}
                                      />
                                      <div className="flex gap-1" style={{ width: '50px' }}>
                                        {index === availabilityRows.length - 1 && (
                                          <button 
                                            onClick={addAvailabilityRow}
                                            className="text-white hover:text-white/70 p-2"
                                          >
                                            <Plus className="w-4 h-4" />
                                          </button>
                                        )}
                                        {availabilityRows.length > 1 && (
                                          <button 
                                            onClick={() => deleteAvailabilityRow(row.id)}
                                            className="text-white hover:text-red-400 p-2"
                                          >
                                            <Trash2 className="w-4 h-4" />
                                          </button>
                                        )}
                                      </div>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                          </div>
                        )}

                        {availabilityType === 'recurring' && (
                          <div className="mb-6">
                            {/* Header with labels */}
                            <div className="flex items-center gap-2 mb-4">
                              <div className="w-4 h-4 border-2 border-white/40 rounded-full flex items-center justify-center">
                                <div className="w-2 h-2 bg-white rounded-full"></div>
                              </div>
                              <span className="text-white text-sm font-medium uppercase tracking-wider">Recurring Availability</span>
                            </div>
                            
                            {/* Column Headers */}
                            <div className="overflow-x-auto">
                              <div className="min-w-max">
                                <div className="flex items-center gap-3 mb-2 text-white/60 text-sm">
                                  <div style={{ width: '80px' }}></div> {/* Day + checkbox column space */}
                                  <div style={{ width: '100px' }}>Start time</div>
                                  <div style={{ width: '100px' }}>End time</div>
                                  <div style={{ width: '80px' }}>Daily (USD)</div>
                                  <div style={{ width: '80px' }}>Hourly (USD)</div>
                                  <div style={{ width: '50px' }}>Actions</div>
                                </div>
                                
                                {/* Recurring Rows */}
                                <div className="space-y-4">
                                  {recurringRows.map((row) => (
                                    <div key={row.id} className="border-l-2 border-white/10 pl-3">
                                      {/* Day checkbox - shown once per day */}
                                      <div className="flex items-center gap-2 mb-2">
                                        <input 
                                          type="checkbox"
                                          checked={row.enabled}
                                          onChange={(e) => updateRecurringRow(row.id, 0, 'enabled', e.target.checked)}
                                          className="w-4 h-4 bg-white/10 border border-white/20 rounded focus:ring-2 focus:ring-white/40"
                                        />
                                        <span className="text-white text-sm font-medium">{row.day}</span>
                                      </div>
                                      
                                      {/* Time slots for this day */}
                                      <div className="space-y-2 ml-6">
                                        {row.slots.map((slot, slotIndex) => (
                                          <div key={slotIndex} className="flex items-center gap-3">
                                            <input 
                                              type="text" 
                                              value={slot.startTime}
                                              onChange={(e) => updateRecurringRow(row.id, slotIndex, 'startTime', e.target.value)}
                                              className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                              style={{ width: '100px' }}
                                              placeholder="09:00 AM"
                                              disabled={!row.enabled}
                                            />
                                            <input 
                                              type="text" 
                                              value={slot.endTime}
                                              onChange={(e) => updateRecurringRow(row.id, slotIndex, 'endTime', e.target.value)}
                                              className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                              style={{ width: '100px' }}
                                              placeholder="05:00 PM"
                                              disabled={!row.enabled}
                                            />
                                            <input 
                                              type="number" 
                                              step="0.01" 
                                              value={slot.dailyRate}
                                              onChange={(e) => updateRecurringRow(row.id, slotIndex, 'dailyRate', e.target.value)}
                                              className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                              style={{ width: '80px' }}
                                              disabled={!row.enabled}
                                            />
                                            <input 
                                              type="number" 
                                              step="0.01" 
                                              value={slot.hourlyRate}
                                              onChange={(e) => updateRecurringRow(row.id, slotIndex, 'hourlyRate', e.target.value)}
                                              className="bg-white/10 border border-white/20 rounded px-3 py-2 text-white text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 flex-shrink-0"
                                              style={{ width: '80px' }}
                                              disabled={!row.enabled}
                                            />
                                            <div className="flex gap-1" style={{ width: '50px' }}>
                                              {slotIndex === row.slots.length - 1 && row.enabled && (
                                                <button 
                                                  onClick={() => addSlotToDay(row.id)}
                                                  className="text-white hover:text-white/70 p-2"
                                                  title="Add time slot"
                                                >
                                                  <Plus className="w-4 h-4" />
                                                </button>
                                              )}
                                              {row.slots.length > 1 && row.enabled && (
                                                <button 
                                                  onClick={() => removeSlotFromDay(row.id, slotIndex)}
                                                  className="text-white hover:text-red-400 p-2"
                                                  title="Remove time slot"
                                                >
                                                  <Trash2 className="w-4 h-4" />
                                                </button>
                                              )}
                                            </div>
                                          </div>
                                        ))}
                                      </div>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                          </div>
                        )}

                        {/* Action Buttons */}
                        <div className="flex gap-3 mt-6">
                          <Button
                            type="button"
                            variant="fill-white"
                            size="sm"
                            className="flex-1 uppercase tracking-wide"
                            onClick={handleSaveCalendarSchedule}
                            disabled={isSavingCalendar}
                          >
                            {isSavingCalendar ? 'Saving...' : 'Save'}
                          </Button>
                          <Button
                            type="button"
                            variant="fill-black"
                            size="sm"
                            className="flex-1 uppercase tracking-wide"
                            onClick={fetchCalendarSettings}
                          >
                            Reset
                          </Button>
                        </div>
                          </>
                        )}
                      </div>
                    </div>
                  </div>
                </div>
              )}

            </div>
          </div>
        </main>
      </div>

      {/* OTP Verification Modal */}
      {showOtpModal && (
        <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-3 sm:p-4">
          <div className="bg-black/90 backdrop-blur-lg border border-white/20 rounded-xl sm:rounded-2xl p-4 sm:p-6 lg:p-8 shadow-2xl max-w-md w-full mx-3 sm:mx-4 max-h-[90vh] overflow-y-auto">
            <div className="text-center mb-6 sm:mb-8">
              <h2 className="text-2xl sm:text-3xl font-bold text-white mb-3 sm:mb-4 uppercase tracking-wide">
                Verify Your Changes
              </h2>
              <p className="text-white/80 text-xs sm:text-sm">
                {changedFields.includes('email') && changedFields.includes('phone')
                  ? `Please input 6-digit codes we just sent to ${userProfile.email} and ${userProfile.countryCode}${userProfile.phoneNumber}`
                  : changedFields.includes('email')
                  ? `Please input 6-digit code we just sent to ${userProfile.email}`
                  : `Please input 6-digit code we just sent to ${userProfile.countryCode}${userProfile.phoneNumber}`}
              </p>
            </div>

            {/* OTP Input Fields */}
            <div className="space-y-6 mb-6">
              {/* Email OTP - Only show if email changed */}
              {changedFields.includes('email') && (
              <div>
                <label className="block text-white/60 text-xs sm:text-sm mb-2 sm:mb-3 uppercase tracking-wider">Email OTP</label>
                <div className="flex gap-1 sm:gap-2 justify-center">
                  {Array.from({ length: 6 }, (_, index) => (
                    <input
                      key={index}
                      type="text"
                      maxLength={1}
                      value={otpData.emailOtp[index] || ''}
                      onChange={(e) => {
                        const newValue = otpData.emailOtp.split('');
                        newValue[index] = e.target.value;
                        handleOtpChange('emailOtp', newValue.join(''));
                        
                        // Auto-focus next input
                        if (e.target.value && index < 5) {
                          const nextInput = e.target.parentElement?.children[index + 1] as HTMLInputElement;
                          nextInput?.focus();
                        }
                      }}
                      className="w-9 h-9 sm:w-12 sm:h-12 bg-white/10 border border-white/20 rounded-lg text-white text-center text-base sm:text-lg font-mono focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                    />
                  ))}
                </div>
              </div>
              )}

              {/* Mobile OTP - Only show if phone changed */}
              {changedFields.includes('phone') && (
              <div>
                <label className="block text-white/60 text-xs sm:text-sm mb-2 sm:mb-3 uppercase tracking-wider">Mobile OTP</label>
                <div className="flex gap-1 sm:gap-2 justify-center">
                  {Array.from({ length: 6 }, (_, index) => (
                    <input
                      key={index}
                      type="text"
                      maxLength={1}
                      value={otpData.mobileOtp[index] || ''}
                      onChange={(e) => {
                        const newValue = otpData.mobileOtp.split('');
                        newValue[index] = e.target.value;
                        handleOtpChange('mobileOtp', newValue.join(''));
                        
                        // Auto-focus next input
                        if (e.target.value && index < 5) {
                          const nextInput = e.target.parentElement?.children[index + 1] as HTMLInputElement;
                          nextInput?.focus();
                        }
                      }}
                      className="w-9 h-9 sm:w-12 sm:h-12 bg-white/10 border border-white/20 rounded-lg text-white text-center text-base sm:text-lg font-mono focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                    />
                  ))}
                </div>
              </div>
              )}
            </div>

            {/* Resend Timer */}
            <div className="text-center mb-6">
              <p className="text-white/60 text-sm">
                Didn't get the code? {resendTimer > 0 ? `Resend in ${resendTimer} seconds` : (
                  <button 
                    onClick={handleResendSecurityOtp}
                    disabled={isSendingOtp}
                    className="text-orange-400 hover:text-orange-300 transition-colors underline"
                  >
                    Resend now
                  </button>
                )}
              </p>
            </div>

            {/* Submit Button */}
            <Button
              type="button"
              variant="fill-white"
              size="lg"
              className="w-full mb-4"
              arrowRight={true}
              onClick={handleVerifySecurityOtpNew}
              disabled={
                isVerifyingOtp || 
                (changedFields.includes('email') && otpData.emailOtp.length !== 6) ||
                (changedFields.includes('phone') && otpData.mobileOtp.length !== 6)
              }
            >
              {isVerifyingOtp ? 'Verifying...' : 'SUBMIT'}
            </Button>

            {/* Wrong Phone Number */}
            <div className="flex items-center justify-between">
              <button
                onClick={() => {
                  setShowOtpModal(false);
                  setOtpData({ emailOtp: '', mobileOtp: '' });
                  setOtpSent(false);
                }}
                className="bg-white/20 hover:bg-white/30 text-white px-4 py-2 rounded text-sm transition-colors uppercase tracking-wider"
              >
               <span className="text-white/60 text-sm">Wrong phone number?</span>
              </button>
              
              <button
                onClick={() => {
                  setShowOtpModal(false);
                  setOtpData({ emailOtp: '', mobileOtp: '' });
                  setOtpSent(false);
                }}
                className="bg-white/20 hover:bg-white/30 text-white px-4 py-2 rounded text-sm transition-colors uppercase tracking-wider"
              >
                EDIT
              </button>
            </div>
          </div>
        </div>
      )}



      {/* Buy Points Modal */}
      {showBuyPointsModal && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-3 sm:p-4">
          <div className="bg-white/10 backdrop-blur-lg border border-white/20 rounded-xl sm:rounded-2xl p-4 sm:p-6 lg:p-8 shadow-2xl max-w-md w-full mx-3 sm:mx-4 max-h-[90vh] overflow-y-auto">
            <div className="flex justify-between items-center mb-4 sm:mb-6">
              <h3 className="text-lg sm:text-xl font-bold text-white uppercase tracking-wide">Buy More Points</h3>
              <button
                onClick={() => setShowBuyPointsModal(false)}
                className="text-white/60 hover:text-white transition-colors"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>

            <div className="mb-4 sm:mb-6">
              <p className="text-white/80 text-xs sm:text-sm mb-3 sm:mb-4">Enter number of points to buy.</p>
              
              {/* Payment Method Dropdown */}
              <div className="mb-3 sm:mb-4">
                <select 
                  className="w-full px-3 sm:px-4 py-3 sm:py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 text-white backdrop-blur-sm transition-all rounded appearance-none text-sm sm:text-base"
                  value={selectedPaymentMethodId}
                  onChange={(e) => setSelectedPaymentMethodId(e.target.value)}
                >
                  <option value="" className="bg-gray-800 text-white">
                    {paymentMethods.length === 0 ? 'No payment method available' : 'Select payment method'}
                  </option>
                  {paymentMethods.map((method) => (
                    <option key={method.id} value={method.id} className="bg-gray-800 text-white">
                      {method.card.brand.toUpperCase()} •••• {method.card.last4}
                      {defaultPaymentMethod?.id === method.id ? ' (Default)' : ''}
                    </option>
                  ))}
                </select>
              </div>

              {/* Add Payment Method */}
              <button 
                className="text-orange-400 hover:text-orange-300 text-sm mb-6 flex items-center gap-2 transition-colors"
                onClick={() => navigate("/add-payment-method")}
              >
                <span>+</span>
                <span>Add a new payment method</span>
              </button>

              {/* Points Input */}
              <div className="flex border border-white/20 rounded overflow-hidden">
                <input
                  type="text"
                  placeholder="Points to buy"
                  value={pointsToBuy}
                  onChange={(e) => {
                    // Only allow numbers
                    const value = e.target.value.replace(/\D/g, '');
                    setPointsToBuy(value);
                  }}
                  className="flex-1 px-3 sm:px-4 py-3 sm:py-4 bg-white/10 focus:outline-none placeholder-white/60 text-white backdrop-blur-sm text-sm sm:text-base"
                />
                <div className="bg-white/10 px-3 sm:px-4 py-3 sm:py-4 text-white font-medium border-l border-white/20 text-sm sm:text-base">
                  ${pointsToBuy ? `${pointsToBuy}.00` : '0.00'}
                </div>
              </div>
            </div>

            {/* Action Buttons */}
            <div className="flex gap-4">
              <Button
                type="button"
                variant="fill-black"
                size="md"
                className="flex-1"
                onClick={() => setShowBuyPointsModal(false)}
                disabled={isBuyingPoints}
              >
                Cancel
              </Button>
              <Button
                type="button"
                variant="fill-white"
                size="md"
                className="flex-1"
                arrowRight={true}
                onClick={handleBuyPoints}
                disabled={isBuyingPoints || !pointsToBuy || !selectedPaymentMethodId}
              >
                {isBuyingPoints ? 'Processing...' : 'Buy Points'}
              </Button>
            </div>
          </div>
        </div>
      )}
    </ProtectedRoute>
  );
}
