import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { Link, useNavigate } from "@remix-run/react";
import { useState, useEffect } from "react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { useLanguage } from "~/hooks/useLanguage";
import { API_BASE_URL } from "~/lib/api";

export const meta: MetaFunction = () => {
  return [
    { title: "Booking Confirmation - Nomadics Studios" },
    {
      name: "description",
      content: "Review your booking details before confirmation.",
    },
  ];
};

interface BookingData {
  location: string;
  location_name: string;
  start_date: string;
  end_date: string;
  start_time: string;
  end_time: string;
}

interface Studio {
  id: number;
  name: string;
  studio_points: number;
  daily_rate: string;
  hourly_rate: string;
  description?: string;
  landing_page?: {
    banner_picture: string;
  };
}

interface Equipment {
  id: number;
  name: string;
  equipment_points: number;
  cost_per_day: string;
  cost_per_hour: string;
  description?: string;
  photo?: string;
}

interface Resource {
  id: number;
  fname: string;
  lname: string;
  resource_points: number;
  daily_rate: string;
  hourly_rate: string;
  description?: string;
  profile_pic?: string;
  languages?: string;
}

export default function BookConfirmation() {
  const navigate = useNavigate();
  const { t } = useLanguage();
  const [bookingData, setBookingData] = useState<BookingData | null>(null);
  const [locations, setLocations] = useState<Record<string, string>>({});
  const [selectedStudios, setSelectedStudios] = useState<Studio[]>([]);
  const [selectedEquipments, setSelectedEquipments] = useState<Equipment[]>([]);
  const [selectedResources, setSelectedResources] = useState<Resource[]>([]);
  const [loading, setLoading] = useState(true);
  const [userPoints, setUserPoints] = useState(0);
  const [locationImage, setLocationImage] = useState<string>("");
  const [showAddGuestForm, setShowAddGuestForm] = useState(false);
  const [guestRows, setGuestRows] = useState<
    Array<{
      firstName: string;
      lastName: string;
      email: string;
      mobile: string;
    }>
  >([{ firstName: "", lastName: "", email: "", mobile: "" }]);
  const [savedGuests, setSavedGuests] = useState<
    Array<{
      firstName: string;
      lastName: string;
      email: string;
      mobile: string;
    }>
  >([]);
  const [studioApiPoints, setStudioApiPoints] = useState<any>(null);
  const [equipmentApiPoints, setEquipmentApiPoints] = useState<any>(null);
  const [resourceApiPoints, setResourceApiPoints] = useState<any>(null);
  const [studioTotal, setStudioTotal] = useState(0);
  const [equipmentTotal, setEquipmentTotal] = useState(0);
  const [resourceTotal, setResourceTotal] = useState(0);
  const [showInsufficientPointsModal, setShowInsufficientPointsModal] =
    useState(false);
  const [allowedGuests, setAllowedGuests] = useState(1); // Default to 1 free guest
  const [additionalGuestPoints, setAdditionalGuestPoints] = useState(0);
  const [guestFee, setGuestFee] = useState(0);

  // Calculate totals from stored sessionStorage points
  const grandTotal = parseFloat(
    (studioTotal + equipmentTotal + resourceTotal + guestFee).toFixed(2)
  );

  // Format date range
  const formatDateRange = () => {
    if (!bookingData) return "";
    const startDate = new Date(
      `${bookingData.start_date}T${bookingData.start_time}`
    );
    const endDate = new Date(`${bookingData.end_date}T${bookingData.end_time}`);
    const options: Intl.DateTimeFormatOptions = {
      day: "numeric",
      month: "short",
      year: "numeric",
    };

    const formattedStart = startDate.toLocaleDateString("en-US", options);
    const formattedEnd = endDate.toLocaleDateString("en-US", options);

    // Calculate duration from API points data
    let totalDays = 0;
    let totalHours = 0;

    // Get days and hours from studio API points
    if (studioApiPoints && studioApiPoints.studio) {
      Object.keys(studioApiPoints.studio).forEach((studioId) => {
        const pointsData = studioApiPoints.studio[studioId];
        if (pointsData) {
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalDays = Math.max(totalDays, pointsData.daily.length);
          }
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            const hours = pointsData.hourly.reduce((sum: number, h: string) => {
              const [hrs] = h.split("|").map((v) => parseFloat(v));
              return sum + hrs;
            }, 0);
            totalHours = Math.max(totalHours, hours);
          }
        }
      });
    }

    // Get days and hours from equipment API points
    if (equipmentApiPoints && equipmentApiPoints.equipment) {
      Object.keys(equipmentApiPoints.equipment).forEach((equipmentId) => {
        const pointsData = equipmentApiPoints.equipment[equipmentId];
        if (pointsData) {
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalDays = Math.max(totalDays, pointsData.daily.length);
          }
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            const hours = pointsData.hourly.reduce((sum: number, h: string) => {
              const [hrs] = h.split("|").map((v) => parseFloat(v));
              return sum + hrs;
            }, 0);
            totalHours = Math.max(totalHours, hours);
          }
        }
      });
    }

    // Get days and hours from resource API points
    if (resourceApiPoints && resourceApiPoints.resource) {
      Object.keys(resourceApiPoints.resource).forEach((resourceId) => {
        const pointsData = resourceApiPoints.resource[resourceId];
        if (pointsData) {
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalDays = Math.max(totalDays, pointsData.daily.length);
          }
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            const hours = pointsData.hourly.reduce((sum: number, h: string) => {
              const [hrs] = h.split("|").map((v) => parseFloat(v));
              return sum + hrs;
            }, 0);
            totalHours = Math.max(totalHours, hours);
          }
        }
      });
    }

    return `${formattedStart} - ${formattedEnd} (${totalDays} day${
      totalDays !== 1 ? "s" : ""
    } ${totalHours} hour${totalHours !== 1 ? "s" : ""})`;
  };

  useEffect(() => {
    const loadBookingDetails = async () => {
      try {
        setLoading(true);

        // Get booking data
        const storedBookingData = sessionStorage.getItem("bookingData");
        if (storedBookingData) {
          const data = JSON.parse(storedBookingData);
          setBookingData(data);

          // Fetch location image
          fetchLocationImage(data.location);
        }

        // Get locations mapping
        const storedLocations = sessionStorage.getItem("locations");
        if (storedLocations) {
          setLocations(JSON.parse(storedLocations));
        }

        // Get user points and allowed guests
        const userData =
          localStorage.getItem("userData") ||
          localStorage.getItem("userFullResponse");
        if (userData) {
          try {
            const parsed = JSON.parse(userData);
            const user = parsed.user || parsed;
            setUserPoints(user.point_balance || user.points || 0);
            setAllowedGuests(user.allowed_guests || 1); // Set allowed guests from login API
          } catch (e) {
            console.error("Error parsing user data:", e);
          }
        }

        // Load API points from sessionStorage
        const storedStudioPoints = sessionStorage.getItem("studioPoints");
        if (storedStudioPoints) {
          setStudioTotal(parseFloat(storedStudioPoints));
        }

        const storedEquipmentPoints = sessionStorage.getItem("equipmentPoints");
        if (storedEquipmentPoints) {
          setEquipmentTotal(parseFloat(storedEquipmentPoints));
        }

        const storedResourcePoints = sessionStorage.getItem("resourcePoints");
        if (storedResourcePoints) {
          setResourceTotal(parseFloat(storedResourcePoints));
        }

        const storedStudioApiPoints = sessionStorage.getItem("studioApiPoints");
        if (storedStudioApiPoints) {
          setStudioApiPoints(JSON.parse(storedStudioApiPoints));
        }

        const storedEquipmentApiPoints =
          sessionStorage.getItem("equipmentApiPoints");
        if (storedEquipmentApiPoints) {
          setEquipmentApiPoints(JSON.parse(storedEquipmentApiPoints));
        }

        const storedResourceApiPoints =
          sessionStorage.getItem("resourceApiPoints");
        if (storedResourceApiPoints) {
          setResourceApiPoints(JSON.parse(storedResourceApiPoints));
        }

        // Fetch selected studios details
        const storedStudioIds = sessionStorage.getItem("selectedStudios");

        // Fetch additional_guest_points from preference API
        const token =
          localStorage.getItem("authToken") ||
          localStorage.getItem("auth_token") ||
          localStorage.getItem("token");
        if (token && storedStudioIds) {
          const studioIds: number[] = JSON.parse(storedStudioIds);
          if (studioIds.length > 0) {
            try {
              const response = await fetch(
                `${API_BASE_URL}/choose-your-preference/studio`,
                {
                  method: "POST",
                  headers: {
                    Authorization: `Bearer ${token}`,
                    "Content-Type": "application/json",
                  },
                  body: JSON.stringify({
                    studio: studioIds,
                    start_date: bookingData?.start_date,
                    end_date: bookingData?.end_date,
                    start_time: bookingData?.start_time,
                    end_time: bookingData?.end_time,
                    location: bookingData?.location,
                  }),
                }
              );
              if (response.ok) {
                const data = await response.json();
                if (data.additional_guest_points !== undefined) {
                  setAdditionalGuestPoints(data.additional_guest_points);
                }
              }
            } catch (error) {
              console.error("Error fetching additional guest points:", error);
            }
          }
        }

        if (storedStudioIds) {
          const studioIds: number[] = JSON.parse(storedStudioIds);
          if (studioIds.length > 0) {
            await fetchStudiosDetails(studioIds);
          }
        }

        // Fetch selected equipments details
        const storedEquipmentIds = sessionStorage.getItem("selectedEquipment");
        if (storedEquipmentIds) {
          const equipmentIds: number[] = JSON.parse(storedEquipmentIds);
          if (equipmentIds.length > 0) {
            await fetchEquipmentsDetails(equipmentIds);
          }
        }

        // Fetch selected resources details
        const storedResourceIds = sessionStorage.getItem("selectedResources");
        if (storedResourceIds) {
          const resourceIds: number[] = JSON.parse(storedResourceIds);
          if (resourceIds.length > 0) {
            await fetchResourcesDetails(resourceIds);
          }
        }
      } catch (error) {
        console.error("Error loading booking details:", error);
      } finally {
        setLoading(false);
      }
    };

    loadBookingDetails();
  }, []);

  // Calculate guest fee when savedGuests changes
  useEffect(() => {
    if (savedGuests.length > allowedGuests && additionalGuestPoints > 0) {
      const extraGuests = savedGuests.length - allowedGuests;
      const calculatedFee = extraGuests * additionalGuestPoints;
      setGuestFee(calculatedFee);
    } else {
      setGuestFee(0);
    }
  }, [savedGuests, allowedGuests, additionalGuestPoints]);

  const fetchLocationImage = async (locationId: string) => {
    try {
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) return;

      const response = await fetch(`${API_BASE_URL}/locations`, {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      });

      if (response.ok) {
        const result = await response.json();
        if (result.status && Array.isArray(result.data)) {
          const location = result.data.find(
            (loc: any) => loc.id.toString() === locationId
          );
          if (location && location.banner_picture) {
            // Extract filename from banner_picture if it's a full URL, otherwise use as-is
            const filename = location.banner_picture.includes("/")
              ? location.banner_picture.split("/").pop()
              : location.banner_picture;
            console.log("Setting locationImage to:", filename);
            setLocationImage(filename);
          }
        }
      }
    } catch (error) {
      console.error("Failed to fetch location image:", error);
    }
  };

  const fetchStudiosDetails = async (studioIds: number[]) => {
    try {
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) return;

      const storedBookingData = sessionStorage.getItem("bookingData");
      if (!storedBookingData) return;
      const data = JSON.parse(storedBookingData);

      const response = await fetch(
        `${API_BASE_URL}/choose-your-preference/studio`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${token}`,
          },
          body: JSON.stringify(data),
        }
      );

      if (response.ok) {
        const result = await response.json();
        if (result.status && result.data && result.data.studios) {
          const studios = result.data.studios.filter((studio: any) =>
            studioIds.includes(studio.id)
          );
          setSelectedStudios(studios);
        }
      }
    } catch (error) {
      console.error("Failed to fetch studios details:", error);
    }
  };

  const fetchEquipmentsDetails = async (equipmentIds: number[]) => {
    try {
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) return;

      const storedBookingData = sessionStorage.getItem("bookingData");
      if (!storedBookingData) return;
      const data = JSON.parse(storedBookingData);

      const response = await fetch(
        `${API_BASE_URL}/choose-your-preference/equipment`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${token}`,
          },
          body: JSON.stringify(data),
        }
      );

      if (response.ok) {
        const result = await response.json();
        if (result.status && result.data && result.data.equipments) {
          const equipments = result.data.equipments.filter((equip: any) =>
            equipmentIds.includes(equip.id)
          );
          setSelectedEquipments(equipments);
        }
      }
    } catch (error) {
      console.error("Failed to fetch equipments details:", error);
    }
  };

  const fetchResourcesDetails = async (resourceIds: number[]) => {
    try {
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) return;

      const storedBookingData = sessionStorage.getItem("bookingData");
      if (!storedBookingData) return;
      const data = JSON.parse(storedBookingData);

      const response = await fetch(
        `${API_BASE_URL}/choose-your-preference/resource`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${token}`,
          },
          body: JSON.stringify(data),
        }
      );

      if (response.ok) {
        const result = await response.json();
        if (result.status && result.data && result.data.resources) {
          const resources = result.data.resources.filter((res: any) =>
            resourceIds.includes(res.id)
          );
          setSelectedResources(resources);
        }
      }
    } catch (error) {
      console.error("Failed to fetch resources details:", error);
    }
  };

  const handleConfirmBooking = async () => {
    // Check if user has sufficient points
    if (userPoints < grandTotal) {
      setShowInsufficientPointsModal(true);
      return;
    }

    try {
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        alert("Authentication required. Please login again.");
        return;
      }

      const storedBookingData = sessionStorage.getItem("bookingData");
      if (!storedBookingData) {
        alert(
          "Booking data not found. Please start the booking process again."
        );
        return;
      }

      const bookingData = JSON.parse(storedBookingData);

      // Prepare booking request payload
      const bookingPayload: any = {
        location: bookingData.location,
        start_date: bookingData.start_date,
        end_date: bookingData.end_date,
        start_time: bookingData.start_time,
        end_time: bookingData.end_time,
      };

      // Add studio data if selected
      if (selectedStudios.length > 0) {
        const studio = selectedStudios[0]; // Assuming single studio selection
        bookingPayload.studio_id = studio.id;

        // Use API points data from sessionStorage
        if (
          studioApiPoints &&
          studioApiPoints.studio &&
          studioApiPoints.studio[studio.id]
        ) {
          const pointsData = studioApiPoints.studio[studio.id];
          let totalPoints = 0;

          // Calculate from daily array
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalPoints += pointsData.daily.reduce(
              (s: number, p: number) => s + p,
              0
            );
          }

          // Calculate from hourly array
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            totalPoints += pointsData.hourly.reduce((s: number, h: string) => {
              const [hours, points] = h.split("|").map((v) => parseFloat(v));
              return s + points;
            }, 0);
          }

          bookingPayload.studio_points = {
            total_points: totalPoints,
            calc: {
              daily: pointsData.daily || [],
              hourly: pointsData.hourly || [],
            },
          };
        }
      }

      // Add equipment data if selected
      if (selectedEquipments.length > 0) {
        const equipment = selectedEquipments[0]; // Assuming single equipment selection
        bookingPayload.equipment_id = equipment.id;

        // Use API points data from sessionStorage
        if (
          equipmentApiPoints &&
          equipmentApiPoints.equipment &&
          equipmentApiPoints.equipment[equipment.id]
        ) {
          const pointsData = equipmentApiPoints.equipment[equipment.id];
          let totalPoints = 0;

          // Calculate from daily array
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalPoints += pointsData.daily.reduce(
              (s: number, p: number) => s + p,
              0
            );
          }

          // Calculate from hourly array
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            totalPoints += pointsData.hourly.reduce((s: number, h: string) => {
              const [hours, points] = h.split("|").map((v) => parseFloat(v));
              return s + points;
            }, 0);
          }

          bookingPayload.equipment_points = {
            total_points: totalPoints,
            calc: {
              daily: pointsData.daily || [],
              hourly: pointsData.hourly || [],
            },
          };
        }
      }

      // Add resource data if selected
      if (selectedResources.length > 0) {
        const resource = selectedResources[0]; // Assuming single resource selection
        bookingPayload.resource_id = resource.id;

        // Use API points data from sessionStorage
        if (
          resourceApiPoints &&
          resourceApiPoints.resource &&
          resourceApiPoints.resource[resource.id]
        ) {
          const pointsData = resourceApiPoints.resource[resource.id];
          let totalPoints = 0;

          // Calculate from daily array
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalPoints += pointsData.daily.reduce(
              (s: number, p: number) => s + p,
              0
            );
          }

          // Calculate from hourly array
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            totalPoints += pointsData.hourly.reduce((s: number, h: string) => {
              const [hours, points] = h.split("|").map((v) => parseFloat(v));
              return s + points;
            }, 0);
          }

          bookingPayload.resource_points = {
            total_points: totalPoints,
            calc: {
              daily: pointsData.daily || [],
              hourly: pointsData.hourly || [],
            },
          };
        }
      }

      // Add guests data from saved guests
      if (savedGuests.length > 0) {
        bookingPayload.guests = savedGuests.map((guest) => ({
          fname: guest.firstName,
          lname: guest.lastName,
          email: guest.email,
          mobile: guest.mobile,
        }));
      }

      const response = await fetch(`${API_BASE_URL}/booking-process`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${token}`,
        },
        body: JSON.stringify(bookingPayload),
      });
      console.log("Booking request payload:", bookingPayload);
      if (response.ok) {
        try {
          // Get response as text first to handle potential debug output
          const responseText = await response.text();
          console.log("Raw response:", responseText);

          // Try to extract JSON from response (in case there's debug output before it)
          let result;
          try {
            result = JSON.parse(responseText);
          } catch (parseError) {
            // If direct parse fails, try to extract JSON from the text
            const jsonMatch = responseText.match(/\{[\s\S]*\}/);
            if (jsonMatch) {
              result = JSON.parse(jsonMatch[0]);
            } else {
              throw new Error("Invalid JSON response");
            }
          }

          console.log("Parsed result:", result);

          if (result.status && result.data?.booking_id) {
            // Store the booking ID for the confirmation page
            sessionStorage.setItem("currentBookingId", result.data.booking_id);

            // Update user points after successful booking
            const newPointBalance =
              result.data.point_balance || userPoints - grandTotal;
            const userData =
              localStorage.getItem("userData") ||
              localStorage.getItem("userFullResponse");
            if (userData) {
              try {
                const parsed = JSON.parse(userData);
                if (parsed.user) {
                  parsed.user.point_balance = newPointBalance;
                  parsed.user.points = newPointBalance;
                } else {
                  parsed.point_balance = newPointBalance;
                  parsed.points = newPointBalance;
                }
                const storageKey = localStorage.getItem("userData")
                  ? "userData"
                  : "userFullResponse";
                localStorage.setItem(storageKey, JSON.stringify(parsed));
                setUserPoints(newPointBalance);

                // Dispatch custom event to update header
                window.dispatchEvent(
                  new CustomEvent("userPointsUpdated", {
                    detail: { points: newPointBalance },
                  })
                );
              } catch (e) {
                console.error("Error updating user points:", e);
              }
            }

            // Navigate to booking confirmed page with the actual booking ID
            console.log(
              "Navigating to:",
              `/booking-confirmed/${result.data.booking_id}`
            );
            navigate(`/booking-confirmed/${result.data.booking_id}`);
          } else {
            alert("Booking confirmation failed. Please try again.");
          }
        } catch (parseError) {
          console.error("Error parsing response:", parseError);
          alert(
            "Booking failed: Invalid response from server. Please contact support."
          );
        }
      } else {
        const errorData = await response
          .json()
          .catch(() => ({ message: "Unknown error" }));
        alert(`Booking failed: ${errorData.error || "Please try again."}`);
      }
    } catch (error) {
      console.error("Booking confirmation error:", error);
      alert("Network error. Please check your connection and try again.");
    }
  };

  const handleAddGuestRow = () => {
    setGuestRows([
      ...guestRows,
      { firstName: "", lastName: "", email: "", mobile: "" },
    ]);
  };

  const handleRemoveGuestRow = (index: number) => {
    const updatedRows = guestRows.filter((_, i) => i !== index);
    setGuestRows(updatedRows);

    // If no rows left, hide the form
    if (updatedRows.length === 0) {
      setShowAddGuestForm(false);
      setGuestRows([{ firstName: "", lastName: "", email: "", mobile: "" }]);
    }
  };

  const handleGuestInputChange = (
    index: number,
    field: string,
    value: string
  ) => {
    const updatedRows = guestRows.map((row, i) =>
      i === index ? { ...row, [field]: value } : row
    );
    setGuestRows(updatedRows);
  };

  const handleSaveGuests = () => {
    const validGuests = guestRows.filter(
      (row) => row.firstName.trim() && row.lastName.trim() && row.email.trim()
    );

    if (validGuests.length > 0) {
      // Add new guests to existing saved guests
      setSavedGuests([...savedGuests, ...validGuests]);
      console.log("Saving guests:", validGuests);
      alert(`${validGuests.length} guest(s) saved successfully!`);
      setShowAddGuestForm(false);
      setGuestRows([{ firstName: "", lastName: "", email: "", mobile: "" }]);
    } else {
      alert("Please fill in at least one complete guest entry.");
    }
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />

        {/* Hero Section */}
        <section className="relative bg-black text-white py-12 md:py-16">
          <div className="absolute inset-0 bg-black/20" />
          <div className="relative z-10 max-w-7xl mx-auto px-4">
            <h1 className="text-5xl md:text-6xl lg:text-7xl font-light mb-4">
              {t("booking.hero.confirmTitle")}
            </h1>
          </div>
        </section>

        {loading ? (
          <div className="max-w-7xl mx-auto px-4 py-20 text-center">
            <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-black"></div>
            <p className="mt-4 text-gray-600">Loading booking details...</p>
          </div>
        ) : (
          <main className="max-w-7xl mx-auto px-4 md:px-6 lg:px-8 py-8 md:py-12">
            <h2 className="text-4xl md:text-5xl font-light text-black mb-8">
              {t("booking.confirmation.details")}
            </h2>

            <div className="space-y-8 mb-12">
              {/* Location Section */}
              {bookingData && (
                <div className="border-b border-gray-200 pb-8">
                  <h3 className="text-2xl md:text-3xl font-medium text-black mb-4">
                    {t("booking.confirmation.location")}
                  </h3>
                  <div className="flex items-center gap-6">
                    <img
                      src={
                        locationImage
                          ? `https://dashboard.nomadicstudios.net/storage/location/${locationImage}`
                          : `https://dashboard.nomadicstudios.net/storage/location/default-location.jpg`
                      }
                      alt={bookingData.location_name}
                      className="w-60 h-40 object-cover"
                      loading="lazy"
                      onError={(e) => {
                        const target = e.currentTarget as HTMLImageElement;
                        if (!target.dataset.errorHandled) {
                          target.dataset.errorHandled = "true";
                          console.log(
                            "Location image failed to load, using fallback"
                          );
                          target.src =
                            "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQwIiBoZWlnaHQ9IjE2MCIgdmlld0JveD0iMCAwIDI0MDE2MCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHJlY3Qgd2lkdGg9IjI0MCIgaGVpZ2h0PSIxNjAiIGZpbGw9IiNGM0Y0RjYiLz4KPHR5eHQgeD0iMTIwIiB5PSI4MCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOUNBM0FGIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBkb21pbmFudC1iYXNlbGluZT0ibWlkZGxlIj5Mb2NhdGlvbjwvdGV4dD4KPC9zdmc+Cg==";
                        }
                      }}
                      onLoad={() => {
                        console.log("Location image loaded successfully");
                      }}
                    />
                    <div>
                      <p className="text-3xl md:text-4xl font-light text-black">
                        {bookingData.location_name ||
                          locations[bookingData.location] ||
                          bookingData.location}
                      </p>
                    </div>
                  </div>
                </div>
              )}

              {/* Date Section */}
              {bookingData && (
                <div className="border-b border-gray-200 pb-8">
                  <h3 className="text-2xl md:text-3xl font-medium text-black mb-4">
                    {t("booking.confirmation.date")}
                  </h3>
                  <p className="text-2xl md:text-3xl text-gray-700">
                    {formatDateRange()}
                  </p>
                  <p className="text-base md:text-lg text-gray-500 mt-2">
                    {t("booking.confirmation.time")}:
                    {bookingData.start_time.substring(0, 5)} -{" "}
                    {bookingData.end_time.substring(0, 5)}
                  </p>
                </div>
              )}

              {/* Type of Studio Section */}
              {selectedStudios.length > 0 && (
                <div className="border-b border-gray-200 pb-8">
                  <h3 className="text-lg font-medium text-black mb-4">
                    Type of studio
                  </h3>
                  <div className="space-y-6">
                    {selectedStudios.map((studio) => (
                      <div key={studio.id} className="flex items-start gap-6">
                        <img
                          src={
                            studio.landing_page?.banner_picture
                              ? `https://dashboard.nomadicstudios.net/storage/location/${studio.landing_page.banner_picture}`
                              : `https://dashboard.nomadicstudios.net/storage/location/default-studio.jpg`
                          }
                          alt={studio.name}
                          className="w-32 h-32 object-cover"
                          loading="lazy"
                          onError={(e) => {
                            console.log(
                              "Studio image failed to load:",
                              e.currentTarget.src
                            );
                            e.currentTarget.src =
                              "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTI4IiBoZWlnaHQ9IjEyOCIgdmlld0JveD0iMCAwIDEyOCAxMjgiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSIxMjgiIGhlaWdodD0iMTI4IiBmaWxsPSIjRjNGNEY2Ii8+Cjx0ZXh0IHg9IjY0IiB5PSI2NCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOUNBM0FGIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBkb21pbmFudC1iYXNlbGluZT0ibWlkZGxlIj5TdHVkaW88L3RleHQ+Cjwvc3ZnPgo=";
                          }}
                          onLoad={() => {
                            console.log(
                              "Studio image loaded successfully:",
                              studio.landing_page?.banner_picture
                            );
                          }}
                        />
                        <div className="flex-1">
                          <div className="flex justify-between items-start mb-2">
                            <h4 className="text-xl font-medium text-black">
                              {studio.name}
                            </h4>
                            <span className="text-lg font-medium text-black">
                              ={" "}
                              {(() => {
                                if (
                                  !studioApiPoints ||
                                  !studioApiPoints.studio ||
                                  !studioApiPoints.studio[studio.id]
                                ) {
                                  return studio.studio_points;
                                }

                                const pointsData =
                                  studioApiPoints.studio[studio.id];
                                let total = 0;

                                // Calculate from daily array
                                if (
                                  pointsData.daily &&
                                  Array.isArray(pointsData.daily)
                                ) {
                                  total += pointsData.daily.reduce(
                                    (s: number, p: number) => s + p,
                                    0
                                  );
                                }

                                // Calculate from hourly array
                                if (
                                  pointsData.hourly &&
                                  Array.isArray(pointsData.hourly)
                                ) {
                                  total += pointsData.hourly.reduce(
                                    (s: number, h: string) => {
                                      const [hours, points] = h
                                        .split("|")
                                        .map((v) => parseFloat(v));
                                      return s + points;
                                    },
                                    0
                                  );
                                }

                                return total.toFixed(2);
                              })()}{" "}
                              pts
                            </span>
                          </div>
                          <p className="text-sm text-gray-600 mb-2">
                            {(() => {
                              if (
                                !studioApiPoints ||
                                !studioApiPoints.studio ||
                                !studioApiPoints.studio[studio.id]
                              ) {
                                return `${studio.studio_points}pts`;
                              }

                              const pointsData =
                                studioApiPoints.studio[studio.id];
                              const parts: string[] = [];

                              // Process daily points
                              if (
                                pointsData.daily &&
                                Array.isArray(pointsData.daily)
                              ) {
                                pointsData.daily.forEach((pts: number) => {
                                  parts.push(`${pts.toFixed(2)}pts x 1day`);
                                });
                              }

                              // Process hourly points
                              if (
                                pointsData.hourly &&
                                Array.isArray(pointsData.hourly)
                              ) {
                                pointsData.hourly.forEach((h: string) => {
                                  const [hours, points] = h
                                    .split("|")
                                    .map((v) => parseFloat(v));
                                  parts.push(
                                    `${points.toFixed(2)}pts x ${hours}hour`
                                  );
                                });
                              }

                              return (
                                parts.join(" + ") ||
                                `${studio.studio_points}pts`
                              );
                            })()}
                          </p>
                          {studio.description && (
                            <p className="text-sm text-gray-600 mt-3">
                              {studio.description}
                            </p>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* Equipment Section */}
              {selectedEquipments.length > 0 && (
                <div className="border-b border-gray-200 pb-8">
                  <h3 className="text-lg font-medium text-black mb-4">
                    Equipment
                  </h3>
                  <div className="space-y-6">
                    {selectedEquipments.map((equipment) => (
                      <div
                        key={equipment.id}
                        className="flex items-start gap-6"
                      >
                        <img
                          src={
                            equipment.photo
                              ? `https://dashboard.nomadicstudios.net/storage/rental/${equipment.photo}`
                              : `https://dashboard.nomadicstudios.net/storage/rental/default-equipment.jpg`
                          }
                          alt={equipment.name}
                          className="w-32 h-24 object-cover"
                          loading="lazy"
                          onError={(e) => {
                            console.log(
                              "Equipment image failed to load:",
                              e.currentTarget.src
                            );
                            e.currentTarget.src =
                              "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTI4IiBoZWlnaHQ9Ijk2IiB2aWV3Qm94PSIwIDAgMTI4IDk2IiBmaWxsPSJub25lIiB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciPgo8cmVjdCB3aWR0aD0iMTI4IiBoZWlnaHQ9Ijk2IiBmaWxsPSIjRjNGNEY2Ii8+Cjx0ZXh0IHg9IjY0IiB5PSI0OCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjEyIiBmaWxsPSIjOUNBM0FGIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBkb21pbmFudC1iYXNlbGluZT0ibWlkZGxlIj5FcXVpcG1lbnQ8L3RleHQ+Cjwvc3ZnPgo=";
                          }}
                          onLoad={() => {
                            console.log(
                              "Equipment image loaded successfully:",
                              equipment.photo
                            );
                          }}
                        />
                        <div className="flex-1">
                          <div className="flex justify-between items-start mb-2">
                            <h4 className="text-xl font-medium text-black">
                              {equipment.name}
                            </h4>
                            <span className="text-lg font-medium text-black">
                              ={" "}
                              {(() => {
                                if (
                                  !equipmentApiPoints ||
                                  !equipmentApiPoints.equipment ||
                                  !equipmentApiPoints.equipment[equipment.id]
                                ) {
                                  return equipment.equipment_points;
                                }

                                const pointsData =
                                  equipmentApiPoints.equipment[equipment.id];
                                let total = 0;

                                // Calculate from daily array
                                if (
                                  pointsData.daily &&
                                  Array.isArray(pointsData.daily)
                                ) {
                                  total += pointsData.daily.reduce(
                                    (s: number, p: number) => s + p,
                                    0
                                  );
                                }

                                // Calculate from hourly array
                                if (
                                  pointsData.hourly &&
                                  Array.isArray(pointsData.hourly)
                                ) {
                                  total += pointsData.hourly.reduce(
                                    (s: number, h: string) => {
                                      const [hours, points] = h
                                        .split("|")
                                        .map((v) => parseFloat(v));
                                      return s + points;
                                    },
                                    0
                                  );
                                }

                                return total.toFixed(2);
                              })()}{" "}
                              pts
                            </span>
                          </div>
                          <p className="text-sm text-gray-600 mb-2">
                            {(() => {
                              if (
                                !equipmentApiPoints ||
                                !equipmentApiPoints.equipment ||
                                !equipmentApiPoints.equipment[equipment.id]
                              ) {
                                return `${equipment.equipment_points}pts`;
                              }

                              const pointsData =
                                equipmentApiPoints.equipment[equipment.id];
                              const parts: string[] = [];

                              // Process daily points
                              if (
                                pointsData.daily &&
                                Array.isArray(pointsData.daily)
                              ) {
                                pointsData.daily.forEach((pts: number) => {
                                  parts.push(`${pts.toFixed(2)}pts x 1day`);
                                });
                              }

                              // Process hourly points
                              if (
                                pointsData.hourly &&
                                Array.isArray(pointsData.hourly)
                              ) {
                                pointsData.hourly.forEach((h: string) => {
                                  const [hours, points] = h
                                    .split("|")
                                    .map((v) => parseFloat(v));
                                  parts.push(
                                    `${points.toFixed(2)}pts x ${hours}hour`
                                  );
                                });
                              }

                              return (
                                parts.join(" + ") ||
                                `${equipment.equipment_points}pts`
                              );
                            })()}
                          </p>
                          {equipment.description && (
                            <p className="text-sm text-gray-600 mt-3">
                              {equipment.description}
                            </p>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* Additional Support Section */}
              {selectedResources.length > 0 && (
                <div className="border-b border-gray-200 pb-8">
                  <h3 className="text-lg font-medium text-black mb-4">
                    Additional Support
                  </h3>
                  <div className="space-y-6">
                    {selectedResources.map((resource) => (
                      <div key={resource.id} className="flex items-start gap-6">
                        <img
                          src={
                            resource.profile_pic
                              ? `https://dashboard.nomadicstudios.net/storage/profile-pic/${resource.profile_pic}`
                              : `https://dashboard.nomadicstudios.net/storage/profile-pic/default-avatar.jpg`
                          }
                          alt={`${resource.fname} ${resource.lname}`}
                          className="w-32 h-32 object-cover rounded"
                          loading="lazy"
                          onError={(e) => {
                            console.log(
                              "Resource image failed to load:",
                              e.currentTarget.src
                            );
                            const initials =
                              resource.fname.charAt(0) +
                              (resource.lname ? resource.lname.charAt(0) : "");
                            e.currentTarget.src = `data:image/svg+xml;base64,${btoa(
                              `<svg width="128" height="128" viewBox="0 0 128 128" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="128" height="128" fill="#E5E7EB"/><text x="64" y="64" font-family="Arial" font-size="32" fill="#6B7280" text-anchor="middle" dominant-baseline="middle">${initials}</text></svg>`
                            )}`;
                          }}
                          onLoad={() => {
                            console.log(
                              "Resource image loaded successfully:",
                              resource.profile_pic
                            );
                          }}
                        />
                        <div className="flex-1">
                          <div className="flex justify-between items-start mb-2">
                            <h4 className="text-xl font-medium text-black">
                              {resource.fname} {resource.lname}
                            </h4>
                            <span className="text-lg font-medium text-black">
                              ={" "}
                              {(() => {
                                if (
                                  !resourceApiPoints ||
                                  !resourceApiPoints.resource ||
                                  !resourceApiPoints.resource[resource.id]
                                ) {
                                  return resource.resource_points;
                                }

                                const pointsData =
                                  resourceApiPoints.resource[resource.id];
                                let total = 0;

                                // Calculate from daily array
                                if (
                                  pointsData.daily &&
                                  Array.isArray(pointsData.daily)
                                ) {
                                  total += pointsData.daily.reduce(
                                    (s: number, p: number) => s + p,
                                    0
                                  );
                                }

                                // Calculate from hourly array
                                if (
                                  pointsData.hourly &&
                                  Array.isArray(pointsData.hourly)
                                ) {
                                  total += pointsData.hourly.reduce(
                                    (s: number, h: string) => {
                                      const [hours, points] = h
                                        .split("|")
                                        .map((v) => parseFloat(v));
                                      return s + points;
                                    },
                                    0
                                  );
                                }

                                return total.toFixed(2);
                              })()}{" "}
                              pts
                            </span>
                          </div>
                          <p className="text-sm text-gray-600 mb-2">
                            {(() => {
                              if (
                                !resourceApiPoints ||
                                !resourceApiPoints.resource ||
                                !resourceApiPoints.resource[resource.id]
                              ) {
                                return `${resource.resource_points}pts`;
                              }

                              const pointsData =
                                resourceApiPoints.resource[resource.id];
                              const parts: string[] = [];

                              // Process daily points
                              if (
                                pointsData.daily &&
                                Array.isArray(pointsData.daily)
                              ) {
                                pointsData.daily.forEach((pts: number) => {
                                  parts.push(`${pts.toFixed(2)}pts x 1day`);
                                });
                              }

                              // Process hourly points
                              if (
                                pointsData.hourly &&
                                Array.isArray(pointsData.hourly)
                              ) {
                                pointsData.hourly.forEach((h: string) => {
                                  const [hours, points] = h
                                    .split("|")
                                    .map((v) => parseFloat(v));
                                  parts.push(
                                    `${points.toFixed(2)}pts x ${hours}hour`
                                  );
                                });
                              }

                              return (
                                parts.join(" + ") ||
                                `${resource.resource_points}pts`
                              );
                            })()}
                          </p>
                          {resource.languages && (
                            <p className="text-sm text-blue-600 mb-2">
                              Languages: {resource.languages}
                            </p>
                          )}
                          {resource.description && (
                            <p className="text-sm text-gray-600 mt-3">
                              {resource.description}
                            </p>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>

            {/* Add Guest Section */}
            <div className="mb-12">
              <h3 className="text-lg font-medium text-black mb-6">Add Guest</h3>

              {/* Display Saved Guests */}
              {savedGuests.length > 0 && (
                <div className="mb-6">
                  <h4 className="text-md font-medium text-black mb-4">
                    Additional Guest
                  </h4>
                  {savedGuests.map((guest, index) => {
                    // Calculate if this guest incurs a fee
                    const guestNumber = index + 1;
                    const guestFeeForThis =
                      guestNumber > allowedGuests ? additionalGuestPoints : 0;

                    return (
                      <div
                        key={index}
                        className="flex items-center justify-between bg-gray-50 p-4 rounded mb-2"
                      >
                        <span className="text-black">
                          {guest.firstName} {guest.lastName} ({guest.email})
                          {guest.mobile && ` - ${guest.mobile}`}
                        </span>
                        <div className="flex items-center gap-2">
                          <span className="text-sm text-gray-600">
                            Guest Fee = {guestFeeForThis.toFixed(2)} pts
                          </span>
                          <Button
                            onClick={() => {
                              const updatedGuests = savedGuests.filter(
                                (_, i) => i !== index
                              );
                              setSavedGuests(updatedGuests);
                            }}
                            variant="outline-black"
                            size="sm"
                            className="text-red-600 border-red-600 hover:bg-red-50"
                          >
                            Remove
                          </Button>
                        </div>
                      </div>
                    );
                  })}
                </div>
              )}

              {/* Add Guest Form */}
              {showAddGuestForm ? (
                <div className="border-2 border-black p-6 bg-white">
                  <div className="space-y-4">
                    {guestRows.map((row, index) => (
                      <div
                        key={index}
                        className="grid grid-cols-1 md:grid-cols-5 gap-4 items-end"
                      >
                        <div>
                          <input
                            type="text"
                            placeholder="First Name"
                            value={row.firstName}
                            onChange={(e) =>
                              handleGuestInputChange(
                                index,
                                "firstName",
                                e.target.value
                              )
                            }
                            className="w-full px-4 py-3 border-2 border-black bg-white text-black placeholder-gray-500 focus:border-black focus:outline-none transition-colors"
                          />
                        </div>
                        <div>
                          <input
                            type="text"
                            placeholder="Last Name"
                            value={row.lastName}
                            onChange={(e) =>
                              handleGuestInputChange(
                                index,
                                "lastName",
                                e.target.value
                              )
                            }
                            className="w-full px-4 py-3 border-2 border-black bg-white text-black placeholder-gray-500 focus:border-black focus:outline-none transition-colors"
                          />
                        </div>
                        <div>
                          <input
                            type="email"
                            placeholder="Email"
                            value={row.email}
                            onChange={(e) =>
                              handleGuestInputChange(
                                index,
                                "email",
                                e.target.value
                              )
                            }
                            className="w-full px-4 py-3 border-2 border-black bg-white text-black placeholder-gray-500 focus:border-black focus:outline-none transition-colors"
                          />
                        </div>
                        <div>
                          <input
                            type="tel"
                            placeholder="Mobile"
                            value={row.mobile}
                            onChange={(e) =>
                              handleGuestInputChange(
                                index,
                                "mobile",
                                e.target.value
                              )
                            }
                            className="w-full px-4 py-3 border-2 border-black bg-white text-black placeholder-gray-500 focus:border-black focus:outline-none transition-colors"
                          />
                        </div>
                        <div>
                          <Button
                            onClick={() => handleRemoveGuestRow(index)}
                            variant="outline-black"
                            size="md"
                            className="w-full bg-red-600 text-white border-red-600 hover:bg-red-700"
                          >
                            Remove
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>

                  <div className="flex gap-3 mt-6">
                    <Button
                      onClick={handleAddGuestRow}
                      variant="outline-black"
                      size="md"
                      className="bg-white text-black border-black hover:bg-gray-100 uppercase tracking-wider font-medium"
                    >
                      Add Guest
                    </Button>
                    <Button
                      onClick={handleSaveGuests}
                      variant="fill-black"
                      size="md"
                      className="bg-black text-white hover:bg-gray-800 uppercase tracking-wider font-medium"
                    >
                      Save Guests
                    </Button>
                  </div>
                </div>
              ) : (
                <Button
                  variant="outline-black"
                  size="md"
                  className="uppercase tracking-wider font-medium"
                  onClick={() => setShowAddGuestForm(true)}
                >
                  Add Guest
                </Button>
              )}
            </div>

            {/* Booking Cost Summary */}
            <div className="border-t-2 border-black pt-8 mb-8">
              <div className="space-y-2 mb-4">
                {studioTotal > 0 && (
                  <div className="flex justify-between items-center">
                    <span className="text-lg text-gray-600">Studio</span>
                    <span className="text-lg text-gray-600">
                      {studioTotal.toFixed(2)} pts
                    </span>
                  </div>
                )}
                {equipmentTotal > 0 && (
                  <div className="flex justify-between items-center">
                    <span className="text-lg text-gray-600">Equipment</span>
                    <span className="text-lg text-gray-600">
                      {equipmentTotal.toFixed(2)} pts
                    </span>
                  </div>
                )}
                {resourceTotal > 0 && (
                  <div className="flex justify-between items-center">
                    <span className="text-lg text-gray-600">Resource</span>
                    <span className="text-lg text-gray-600">
                      {resourceTotal.toFixed(2)} pts
                    </span>
                  </div>
                )}
                {guestFee > 0 && (
                  <div className="flex justify-between items-center">
                    <span className="text-lg text-gray-600">
                      Guest Fee ({savedGuests.length - allowedGuests} extra
                      guest{savedGuests.length - allowedGuests > 1 ? "s" : ""})
                    </span>
                    <span className="text-lg text-gray-600">
                      {guestFee.toFixed(2)} pts
                    </span>
                  </div>
                )}
              </div>
              <div className="flex justify-between items-center mb-4 border-t pt-4">
                <span className="text-2xl font-medium text-black">
                  Booking cost
                </span>
                <span className="text-2xl font-medium text-black">
                  {grandTotal.toFixed(2)} pts
                </span>
              </div>
              <div className="flex justify-between items-center text-gray-600">
                <span className="text-lg">Points left</span>
                <span className="text-lg">
                  {(userPoints - grandTotal).toFixed(2)} pts
                </span>
              </div>
            </div>

            {/* Action Button */}
            <div className="flex justify-center">
              <Button
                onClick={handleConfirmBooking}
                variant="fill-black"
                size="lg"
                className="px-12 uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-lg"
              >
                Confirm Booking
              </Button>
            </div>
          </main>
        )}

        {/* Insufficient Points Modal */}
        {showInsufficientPointsModal && (
          <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
            <div className="bg-white rounded-lg p-8 max-w-md w-full">
              <h3 className="text-2xl font-medium text-black mb-4">
                Insufficient Points
              </h3>
              <p className="text-gray-600 mb-2">
                You don't have enough points to complete this booking.
              </p>
              <div className="bg-gray-50 p-4 rounded mb-6">
                <div className="flex justify-between mb-2">
                  <span className="text-gray-700">Required:</span>
                  <span className="font-medium">
                    {grandTotal.toFixed(2)} pts
                  </span>
                </div>
                <div className="flex justify-between mb-2">
                  <span className="text-gray-700">Your balance:</span>
                  <span className="font-medium">
                    {userPoints.toFixed(2)} pts
                  </span>
                </div>
                <div className="flex justify-between border-t pt-2">
                  <span className="text-gray-700">Need:</span>
                  <span className="font-medium text-red-600">
                    {(grandTotal - userPoints).toFixed(2)} pts
                  </span>
                </div>
              </div>
              <div className="flex gap-3">
                <Button
                  onClick={() => setShowInsufficientPointsModal(false)}
                  variant="outline-black"
                  size="md"
                  className="flex-1"
                >
                  Cancel
                </Button>
                <Button
                  onClick={() => {
                    setShowInsufficientPointsModal(false);
                    navigate("/account-settings?tab=points");
                  }}
                  variant="fill-black"
                  size="md"
                  className="flex-1"
                >
                  Add Points
                </Button>
              </div>
            </div>
          </div>
        )}

        {/* <AppFooter /> */}
      </div>
    </ProtectedRoute>
  );
}
