import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useState, useEffect } from "react";
import { useNavigate } from "@remix-run/react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { useLanguage } from "~/hooks/useLanguage";
import { API_BASE_URL } from "~/lib/api";

export const meta: MetaFunction = () => {
  return [
    { title: "Select Equipment - Nomadics Studios" },
    {
      name: "description",
      content: "Select equipment you want to rent for your studio session.",
    },
  ];
};

interface BookingData {
  location: string;
  location_name: string;
  start_date: string;
  end_date: string;
  start_time: string;
  end_time: string;
}

interface ResourceType {
  id: number;
  name: string;
}

interface Equipment {
  id: number;
  studios_id: number;
  location_id: number;
  photo: string;
  name: string;
  category: number;
  description: string;
  serial_number: string;
  price: string;
  cost_per_day: string;
  cost_per_hour: string;
  min_booking_time: string;
  min_booking_minutes: number;
  max_booking_time: string;
  max_booking_minutes: number;
  available: boolean;
  booking_days: number;
  booking_hours: number;
  equipment_points: number;
  resource_types: ResourceType;
  schedules: any[];
  bookings: any[];
}

interface PreferenceResponse {
  status: boolean;
  data: {
    type: string;
    equipments: Equipment[];
    advance_date_flag: string;
    points: any;
  };
}

export default function BookEquipment() {
  const navigate = useNavigate();
  const { t } = useLanguage();
  const [bookingData, setBookingData] = useState<BookingData | null>(null);
  const [equipments, setEquipments] = useState<Equipment[]>([]);
  const [locations, setLocations] = useState<Record<string, string>>({});
  const [selectedEquipment, setSelectedEquipment] = useState<number[]>([]);
  const [loading, setLoading] = useState(true);
  const [totalEquipmentPoints, setTotalEquipmentPoints] = useState(0);
  const [studioPoints, setStudioPoints] = useState(0);
  const [userPoints, setUserPoints] = useState(0);
  const [apiPoints, setApiPoints] = useState<any>(null);

  // Calculate booking duration from API points data
  const calculateDuration = () => {
    if (!apiPoints || !apiPoints.equipment) return "0D 0H";

    let totalDays = 0;
    let totalHours = 0;

    // Iterate through all selected equipment to get max duration
    equipments
      .filter((equipment) => selectedEquipment.includes(equipment.id))
      .forEach((equipment) => {
        const pointsData = apiPoints.equipment[equipment.id];
        if (pointsData) {
          // Count days from daily array length
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalDays = Math.max(totalDays, pointsData.daily.length);
          }

          // Extract hours from hourly array
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            const hours = pointsData.hourly.reduce((sum: number, h: string) => {
              const [hrs] = h.split("|").map((v) => parseFloat(v));
              return sum + hrs;
            }, 0);
            totalHours = Math.max(totalHours, hours);
          }
        }
      });

    return `${totalDays}D ${totalHours}H`;
  };

  // Load booking data and fetch equipment
  useEffect(() => {
    const loadBookingData = () => {
      const stored = sessionStorage.getItem("bookingData");
      if (stored) {
        const data = JSON.parse(stored);
        setBookingData(data);
        fetchEquipments(data);
      } else {
        // If no booking data, redirect to home
        navigate("/home");
      }
    };

    // Get studio points from previous step
    const storedStudioPoints = sessionStorage.getItem("studioPoints");
    if (storedStudioPoints) {
      setStudioPoints(parseInt(storedStudioPoints));
    }

    // Get locations from previous step if available
    const storedLocations = sessionStorage.getItem("locations");
    if (storedLocations) {
      setLocations(JSON.parse(storedLocations));
    }

    // Get user points
    const userData =
      localStorage.getItem("userData") ||
      localStorage.getItem("userFullResponse");
    if (userData) {
      try {
        const parsed = JSON.parse(userData);
        const user = parsed.user || parsed;
        setUserPoints(user.point_balance || user.points || 0);
      } catch (e) {
        console.error("Error parsing user data:", e);
      }
    }

    loadBookingData();
  }, []);

  const fetchEquipments = async (data: BookingData) => {
    try {
      setLoading(true);
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        console.error("No auth token found");
        return;
      }

      const response = await fetch(
        `${API_BASE_URL}/choose-your-preference/equipment`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${token}`,
          },
          body: JSON.stringify(data),
        }
      );

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result: PreferenceResponse = await response.json();

      if (result.status && result.data) {
        setEquipments(result.data.equipments || []);
        setApiPoints(result.data.points || null);
      }
    } catch (error) {
      console.error("Failed to fetch equipments:", error);
    } finally {
      setLoading(false);
    }
  };

  const handleEquipmentSelect = (equipmentId: number) => {
    setSelectedEquipment((prev) => {
      if (prev.includes(equipmentId)) {
        return prev.filter((id) => id !== equipmentId);
      } else {
        return [...prev, equipmentId];
      }
    });
  };

  // Calculate total equipment points using API points data
  useEffect(() => {
    if (!apiPoints || !apiPoints.equipment) {
      setTotalEquipmentPoints(0);
      return;
    }

    const total = equipments
      .filter((equipment) => selectedEquipment.includes(equipment.id))
      .reduce((sum, equipment) => {
        const equipmentPointsData = apiPoints.equipment[equipment.id];
        if (!equipmentPointsData) return sum;

        let equipmentTotal = 0;

        // Calculate from daily array
        if (
          equipmentPointsData.daily &&
          Array.isArray(equipmentPointsData.daily)
        ) {
          equipmentTotal += equipmentPointsData.daily.reduce(
            (s: number, p: number) => s + p,
            0
          );
        }

        // Calculate from hourly array (format: "hours|points")
        if (
          equipmentPointsData.hourly &&
          Array.isArray(equipmentPointsData.hourly)
        ) {
          equipmentTotal += equipmentPointsData.hourly.reduce(
            (s: number, h: string) => {
              const [hours, points] = h.split("|").map((v) => parseFloat(v));
              return s + points;
            },
            0
          );
        }

        return sum + equipmentTotal;
      }, 0);
    setTotalEquipmentPoints(total);
  }, [selectedEquipment, equipments, apiPoints]);

  const handleSkip = () => {
    // Store selected equipment (if any) and navigate to support
    sessionStorage.setItem(
      "selectedEquipment",
      JSON.stringify(selectedEquipment)
    );
    sessionStorage.setItem("equipmentPoints", totalEquipmentPoints.toString());
    sessionStorage.setItem("equipmentApiPoints", JSON.stringify(apiPoints));
    navigate("/book-support");
  };

  const handleContinue = () => {
    // Store selected equipment and navigate to support
    sessionStorage.setItem(
      "selectedEquipment",
      JSON.stringify(selectedEquipment)
    );
    sessionStorage.setItem("equipmentPoints", totalEquipmentPoints.toString());
    sessionStorage.setItem("equipmentApiPoints", JSON.stringify(apiPoints));
    navigate("/book-support");
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />

        {/* Hero Section */}
        <section className="relative bg-black text-white py-12 md:py-16">
          <div className="absolute inset-0 bg-black/20" />
          <div className="relative z-10 max-w-7xl mx-auto px-4">
            <h1 className="text-5xl md:text-6xl lg:text-7xl font-light mb-4">
              {t("booking.hero.title")}
            </h1>
          </div>
        </section>

        <main className="max-w-7xl mx-auto px-4 md:px-6 lg:px-8 py-8 md:py-12">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 md:gap-8 lg:gap-12">
            {/* Left Column - Equipment Selection */}
            <div className="lg:col-span-2">
              <div className="mb-8">
                <h2 className="text-3xl md:text-4xl font-light text-black mb-8">
                  {t("booking.equipment.title")}
                </h2>
              </div>

              {loading ? (
                <div className="text-center py-20">
                  <div className="w-8 h-8 border-2 border-black/40 border-t-black animate-spin rounded-full mx-auto mb-4"></div>
                  <p className="text-sm text-gray-600">Loading equipment...</p>
                </div>
              ) : equipments.length > 0 ? (
                <div className="space-y-6">
                  {equipments.map((item) => (
                    <div
                      key={item.id}
                      className={`border-2 transition-all ${
                        !item.available
                          ? "border-gray-300 bg-gray-100 cursor-not-allowed opacity-60"
                          : selectedEquipment.includes(item.id)
                          ? "border-black bg-gray-50 cursor-pointer"
                          : "border-gray-200 hover:border-gray-300 cursor-pointer"
                      }`}
                      onClick={() =>
                        item.available && handleEquipmentSelect(item.id)
                      }
                    >
                      <div className="p-6">
                        <div className="flex items-start gap-6">
                          {/* Checkbox */}
                          <div className="flex-shrink-0 mt-1">
                            <div
                              className={`w-6 h-6 border-2 transition-colors ${
                                !item.available
                                  ? "border-gray-300 bg-gray-200 cursor-not-allowed"
                                  : selectedEquipment.includes(item.id)
                                  ? "border-black bg-black"
                                  : "border-gray-400"
                              }`}
                            >
                              {selectedEquipment.includes(item.id) &&
                                item.available && (
                                  <svg
                                    className="w-full h-full text-white p-1"
                                    fill="currentColor"
                                    viewBox="0 0 20 20"
                                  >
                                    <path
                                      fillRule="evenodd"
                                      d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                      clipRule="evenodd"
                                    />
                                  </svg>
                                )}
                            </div>
                          </div>

                          {/* Content */}
                          <div className="flex-1">
                            <div className="flex items-start justify-between mb-4">
                              <div>
                                <h3 className="text-2xl md:text-3xl font-medium text-black mb-2">
                                  {item.name}
                                </h3>
                                {!item.available && (
                                  <p className="text-base md:text-lg text-red-600 mb-2">
                                    {t("booking.equipment.unavailable")}
                                  </p>
                                )}
                                {item.resource_types && (
                                  <p className="text-sm text-red-600 mb-2">
                                    Require {item.resource_types.name}
                                  </p>
                                )}
                                {item.description && (
                                  <p className="text-sm text-gray-600">
                                    {item.description}
                                  </p>
                                )}
                              </div>
                              <span className="text-2xl md:text-3xl font-medium text-black">
                                {item.equipment_points}pts
                              </span>
                            </div>
                          </div>

                          {/* Equipment Image */}
                          <div className="flex-shrink-0">
                            <img
                              src={
                                item.photo
                                  ? `https://dashboard.nomadicstudios.net/storage/rental/${item.photo}`
                                  : `https://dashboard.nomadicstudios.net/storage/rental/default-equipment.jpg`
                              }
                              alt={item.name}
                              className="w-32 h-24 object-cover"
                              loading="lazy"
                              onError={(e) => {
                                console.log(
                                  "Equipment image failed to load:",
                                  e.currentTarget.src
                                );
                                e.currentTarget.src =
                                  "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTI4IiBoZWlnaHQ9Ijk2IiB2aWV3Qm94PSIwIDAgMTI4IDk2IiBmaWxsPSJub25lIiB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciPgo8cmVjdCB3aWR0aD0iMTI4IiBoZWlnaHQ9Ijk2IiBmaWxsPSIjRjNGNEY2Ii8+Cjx0ZXh0IHg9IjY0IiB5PSI0OCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjEyIiBmaWxsPSIjOUNBM0FGIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBkb21pbmFudC1iYXNlbGluZT0ibWlkZGxlIj5FcXVpcG1lbnQ8L3RleHQ+Cjwvc3ZnPgo=";
                              }}
                              onLoad={() => {
                                console.log(
                                  "Equipment image loaded successfully:",
                                  item.photo
                                );
                              }}
                            />
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-20">
                  <p className="text-gray-600">
                    No equipment available for the selected time.
                  </p>
                </div>
              )}
            </div>

            {/* Right Column - Booking Summary */}
            <div className="lg:col-span-1">
              <div className="bg-gray-50 p-6 sticky top-8">
                <div className="text-center mb-8">
                  <div className="text-4xl font-light text-black mb-2">
                    {calculateDuration()}
                  </div>
                </div>

                {bookingData && (
                  <div className="space-y-4 mb-8">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                        Where
                      </label>
                      <input
                        type="text"
                        value={
                          bookingData.location_name ||
                          locations[bookingData.location] ||
                          bookingData.location
                        }
                        disabled
                        className="w-full p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      />
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                        Start
                      </label>
                      <div className="grid grid-cols-2 gap-2">
                        <input
                          type="text"
                          value={bookingData.start_date}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                        <input
                          type="text"
                          value={bookingData.start_time.substring(0, 5)}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                      </div>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                        End
                      </label>
                      <div className="grid grid-cols-2 gap-2">
                        <input
                          type="text"
                          value={bookingData.end_date}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                        <input
                          type="text"
                          value={bookingData.end_time.substring(0, 5)}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:outline-none transition-colors"
                        />
                      </div>
                    </div>
                  </div>
                )}

                {/* Price Breakdown */}
                <div className="space-y-3 mb-6">
                  {studioPoints > 0 && (
                    <div className="flex justify-between text-sm">
                      <span className="font-semibold">Studio</span>
                      <span>{studioPoints} pts</span>
                    </div>
                  )}
                  {selectedEquipment.length > 0 ? (
                    equipments
                      .filter((equipment) =>
                        selectedEquipment.includes(equipment.id)
                      )
                      .map((equipment) => {
                        if (
                          !apiPoints ||
                          !apiPoints.equipment ||
                          !apiPoints.equipment[equipment.id]
                        ) {
                          return (
                            <div key={equipment.id} className="text-sm">
                              <div className="font-semibold mb-1">
                                Equipment
                              </div>
                              <div className="text-gray-700">
                                {equipment.equipment_points} pts
                              </div>
                            </div>
                          );
                        }
                        const pointsData = apiPoints.equipment[equipment.id];
                        const parts: string[] = [];
                        let totalPoints = 0;

                        // Process daily points
                        if (
                          pointsData.daily &&
                          Array.isArray(pointsData.daily)
                        ) {
                          pointsData.daily.forEach(
                            (pts: number, idx: number) => {
                              parts.push(`${pts.toFixed(2)}pts x 1day`);
                              totalPoints += pts;
                            }
                          );
                        }

                        // Process hourly points
                        if (
                          pointsData.hourly &&
                          Array.isArray(pointsData.hourly)
                        ) {
                          pointsData.hourly.forEach((h: string) => {
                            const [hours, points] = h
                              .split("|")
                              .map((v) => parseFloat(v));
                            parts.push(
                              `${points.toFixed(2)}pts x ${hours}hour`
                            );
                            totalPoints += points;
                          });
                        }

                        const breakdown =
                          parts.length > 0
                            ? `${parts.join(" + ")} = ${totalPoints.toFixed(
                                2
                              )} pts`
                            : `${totalPoints.toFixed(2)} pts`;
                        return (
                          <div key={equipment.id} className="text-sm">
                            <div className="font-semibold mb-1">Equipment</div>
                            <div className="text-gray-700">{breakdown}</div>
                          </div>
                        );
                      })
                  ) : (
                    <div className="text-sm text-gray-500">
                      No equipment selected
                    </div>
                  )}
                  <hr className="border-gray-300" />
                  <div className="flex justify-between font-medium">
                    <span>Total</span>
                    <span>
                      {(studioPoints + totalEquipmentPoints).toFixed(2)} pts
                    </span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Points left</span>
                    <span>
                      {(
                        userPoints -
                        studioPoints -
                        totalEquipmentPoints
                      ).toFixed(2)}{" "}
                      pts
                    </span>
                  </div>
                </div>

                <div className="space-y-3">
                  <Button
                    onClick={handleSkip}
                    variant="outline-black"
                    size="md"
                    className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-md"
                  >
                    Skip
                  </Button>

                  <Button
                    onClick={handleContinue}
                    variant="fill-black"
                    size="md"
                    className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-lg"
                  >
                    Continue
                  </Button>
                </div>
              </div>
            </div>
          </div>
        </main>

        {/* <AppFooter /> */}
      </div>
    </ProtectedRoute>
  );
}
