import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useState, useEffect } from "react";
import { useNavigate } from "@remix-run/react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { useLanguage } from "~/hooks/useLanguage";
import { API_BASE_URL } from "~/lib/api";

export const meta: MetaFunction = () => {
  return [
    { title: "Book Studio - Nomadics Studios" },
    {
      name: "description",
      content: "Select the type of studio that best fits your creative needs.",
    },
  ];
};

interface BookingData {
  location: string;
  location_name: string;
  start_date: string;
  end_date: string;
  start_time: string;
  end_time: string;
}

interface Studio {
  id: number;
  location_id: number;
  name: string;
  studio_slug: string;
  studio_type: number;
  daily_rate: string;
  hourly_rate: string;
  min_booking_time: string;
  max_booking_time: string;
  available: boolean;
  booking_days: number;
  booking_hours: number;
  studio_points: number;
  landing_page: {
    id: number;
    banner_picture: string;
    studios: string;
  };
  schedules: any[];
  bookings: any[];
}

interface PreferenceResponse {
  status: boolean;
  data: {
    type: string;
    locations: Record<string, string>;
    studios: Studio[];
    advance_date_flag: string;
    points: any;
  };
}

export default function BookStudio() {
  const navigate = useNavigate();
  const { t } = useLanguage();
  const [bookingData, setBookingData] = useState<BookingData | null>(null);
  const [studios, setStudios] = useState<Studio[]>([]);
  const [locations, setLocations] = useState<Record<string, string>>({});
  const [selectedStudios, setSelectedStudios] = useState<number[]>([]);
  const [loading, setLoading] = useState(true);
  const [totalPoints, setTotalPoints] = useState(0);
  const [userPoints, setUserPoints] = useState(0);
  const [apiPoints, setApiPoints] = useState<any>(null);

  // Calculate booking duration from API points data
  const calculateDuration = () => {
    if (!apiPoints || !apiPoints.studio) return "0D 0H";

    let totalDays = 0;
    let totalHours = 0;

    // Iterate through all selected studios to get max duration
    studios
      .filter((studio) => selectedStudios.includes(studio.id))
      .forEach((studio) => {
        const pointsData = apiPoints.studio[studio.id];
        if (pointsData) {
          // Count days from daily array length
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalDays = Math.max(totalDays, pointsData.daily.length);
          }

          // Extract hours from hourly array
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            const hours = pointsData.hourly.reduce((sum: number, h: string) => {
              const [hrs] = h.split("|").map((v) => parseFloat(v));
              return sum + hrs;
            }, 0);
            totalHours = Math.max(totalHours, hours);
          }
        }
      });

    return `${totalDays}D ${totalHours}H`;
  };

  // Load booking data and fetch studios
  useEffect(() => {
    const loadBookingData = () => {
      const stored = sessionStorage.getItem("bookingData");
      if (stored) {
        const data = JSON.parse(stored);
        setBookingData(data);
        fetchStudios(data);
      } else {
        // If no booking data, redirect to home
        navigate("/home");
      }
    };

    // Get user points
    const userData =
      localStorage.getItem("userData") ||
      localStorage.getItem("userFullResponse");
    if (userData) {
      try {
        const parsed = JSON.parse(userData);
        const user = parsed.user || parsed;
        setUserPoints(user.point_balance || user.points || 0);
      } catch (e) {
        console.error("Error parsing user data:", e);
      }
    }

    loadBookingData();
  }, []);

  const fetchStudios = async (data: BookingData) => {
    try {
      setLoading(true);
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        console.error("No auth token found");
        return;
      }

      const response = await fetch(
        `${API_BASE_URL}/choose-your-preference/studio`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${token}`,
          },
          body: JSON.stringify(data),
        }
      );

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result: PreferenceResponse = await response.json();

      if (result.status && result.data) {
        setStudios(result.data.studios || []);
        setLocations(result.data.locations || {});
        setApiPoints(result.data.points || null);
        // Store locations for next pages
        sessionStorage.setItem(
          "locations",
          JSON.stringify(result.data.locations || {})
        );
      }
    } catch (error) {
      console.error("Failed to fetch studios:", error);
    } finally {
      setLoading(false);
    }
  };

  const handleStudioToggle = (studioId: number) => {
    setSelectedStudios((prev) => {
      if (prev.includes(studioId)) {
        return prev.filter((id) => id !== studioId);
      } else {
        return [...prev, studioId];
      }
    });
  };

  // Calculate total points using API points data
  useEffect(() => {
    if (!apiPoints || !apiPoints.studio) {
      setTotalPoints(0);
      return;
    }

    const total = studios
      .filter((studio) => selectedStudios.includes(studio.id))
      .reduce((sum, studio) => {
        const studioPointsData = apiPoints.studio[studio.id];
        if (!studioPointsData) return sum;

        let studioTotal = 0;

        // Calculate from daily array
        if (studioPointsData.daily && Array.isArray(studioPointsData.daily)) {
          studioTotal += studioPointsData.daily.reduce(
            (s: number, p: number) => s + p,
            0
          );
        }

        // Calculate from hourly array (format: "hours|points")
        if (studioPointsData.hourly && Array.isArray(studioPointsData.hourly)) {
          studioTotal += studioPointsData.hourly.reduce(
            (s: number, h: string) => {
              const [hours, points] = h.split("|").map((v) => parseFloat(v));
              return s + points;
            },
            0
          );
        }

        return sum + studioTotal;
      }, 0);
    setTotalPoints(total);
  }, [selectedStudios, studios, apiPoints]);

  const handleSkip = () => {
    // Store selected studios (if any) and navigate to equipment
    sessionStorage.setItem("selectedStudios", JSON.stringify(selectedStudios));
    sessionStorage.setItem("studioPoints", totalPoints.toString());
    sessionStorage.setItem("studioApiPoints", JSON.stringify(apiPoints));
    navigate("/book-equipment");
  };

  const handleContinue = () => {
    if (selectedStudios.length === 0) {
      alert("Please select at least one studio");
      return;
    }

    // Store selected studios and navigate to equipment
    sessionStorage.setItem("selectedStudios", JSON.stringify(selectedStudios));
    sessionStorage.setItem("studioPoints", totalPoints.toString());
    sessionStorage.setItem("studioApiPoints", JSON.stringify(apiPoints));
    navigate("/book-equipment");
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />

        {/* Hero Section */}
        <section className="relative bg-black text-white py-16">
          <div className="absolute inset-0 bg-black/20" />
          <div className="relative z-10 max-w-7xl mx-auto px-4">
            <h1 className="text-5xl md:text-6xl lg:text-7xl font-light mb-4">
              {t("booking.hero.title")}
            </h1>
          </div>
        </section>

        <main className="max-w-7xl mx-auto px-4 py-12">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
            {/* Left Column - Studio Selection */}
            <div className="lg:col-span-2">
              <div className="mb-8">
                <h2 className="text-3xl md:text-4xl font-light text-black mb-8">
                  {t("booking.studio.title")}
                </h2>
              </div>

              {loading ? (
                <div className="text-center py-20">
                  <div className="w-8 h-8 border-2 border-black/40 border-t-black animate-spin rounded-full mx-auto mb-4"></div>
                  <p className="text-sm text-gray-600">Loading studios...</p>
                </div>
              ) : studios.length > 0 ? (
                <div className="space-y-6">
                  {studios.map((studio) => (
                    <div
                      key={studio.id}
                      className={`border-2 transition-all ${
                        !studio.available
                          ? "border-gray-300 bg-gray-100 cursor-not-allowed opacity-60"
                          : selectedStudios.includes(studio.id)
                          ? "border-black bg-gray-50 cursor-pointer"
                          : "border-gray-200 hover:border-gray-300 cursor-pointer"
                      }`}
                      onClick={() =>
                        studio.available && handleStudioToggle(studio.id)
                      }
                    >
                      <div className="p-6">
                        <div className="flex items-start gap-6">
                          {/* Checkbox */}
                          <div className="flex-shrink-0 mt-1">
                            <div
                              className={`w-6 h-6 border-2 transition-colors ${
                                !studio.available
                                  ? "border-gray-300 bg-gray-200 cursor-not-allowed"
                                  : selectedStudios.includes(studio.id)
                                  ? "border-black bg-black"
                                  : "border-gray-400"
                              }`}
                            >
                              {selectedStudios.includes(studio.id) &&
                                studio.available && (
                                  <svg
                                    className="w-full h-full text-white p-1"
                                    fill="currentColor"
                                    viewBox="0 0 20 20"
                                  >
                                    <path
                                      fillRule="evenodd"
                                      d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                      clipRule="evenodd"
                                    />
                                  </svg>
                                )}
                            </div>
                          </div>

                          {/* Content */}
                          <div className="flex-1">
                            <div className="flex items-start justify-between mb-4">
                              <div>
                                <h3 className="text-2xl md:text-3xl font-medium text-black mb-2">
                                  {studio.name}
                                </h3>
                                {!studio.available && (
                                  <p className="text-base md:text-lg text-red-600">
                                    {t("booking.studio.unavailable")}
                                  </p>
                                )}
                              </div>
                              <span className="text-2xl md:text-3xl font-medium text-black">
                                {studio.studio_points}pts
                              </span>
                            </div>
                          </div>

                          {/* Image */}
                          <div className="flex-shrink-0">
                            <img
                              src={
                                studio.landing_page?.banner_picture
                                  ? `https://dashboard.nomadicstudios.net/storage/location/${studio.landing_page.banner_picture}`
                                  : `https://dashboard.nomadicstudios.net/storage/location/default-studio.jpg`
                              }
                              alt={studio.name}
                              className="w-64 h-40 object-cover"
                              onError={(e) => {
                                e.currentTarget.src =
                                  "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjU2IiBoZWlnaHQ9IjE2MCIgdmlld0JveD0iMCAwIDI1NiAxNjAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSIyNTYiIGhlaWdodD0iMTYwIiBmaWxsPSIjRjNGNEY2Ii8+Cjx0ZXh0IHg9IjEyOCIgeT0iODAiIGZvbnQtZmFtaWx5PSJBcmlhbCIgZm9udC1zaXplPSIxNCIgZmlsbD0iIzlDQTNBRiIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZG9taW5hbnQtYmFzZWxpbmU9Im1pZGRsZSI+U3R1ZGlvPC90ZXh0Pgo8L3N2Zz4K";
                              }}
                            />
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-20">
                  <p className="text-gray-600">
                    No studios available for the selected time.
                  </p>
                </div>
              )}
            </div>{" "}
            {/* Right Column - Booking Summary */}
            <div className="lg:col-span-1">
              <div className="bg-gray-50 p-6 sticky top-8">
                <div className="text-center mb-8">
                  <div className="text-5xl md:text-6xl font-light text-black mb-2">
                    {calculateDuration()}
                  </div>
                </div>

                {bookingData && (
                  <div className="space-y-4 mb-8">
                    <div>
                      <label className="block text-base md:text-lg font-medium text-black mb-2 uppercase tracking-wider">
                        {t("booking.form.where")}
                      </label>
                      <input
                        type="text"
                        value={
                          bookingData.location_name ||
                          locations[bookingData.location] ||
                          "Location"
                        }
                        disabled
                        className="w-full p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      />
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                        Start
                      </label>
                      <div className="grid grid-cols-2 gap-2">
                        <input
                          type="text"
                          value={bookingData.start_date}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                        <input
                          type="text"
                          value={bookingData.start_time.substring(0, 5)}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                      </div>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                        End
                      </label>
                      <div className="grid grid-cols-2 gap-2">
                        <input
                          type="text"
                          value={bookingData.end_date}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                        <input
                          type="text"
                          value={bookingData.end_time.substring(0, 5)}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                      </div>
                    </div>
                  </div>
                )}

                {/* Price Breakdown */}
                <div className="space-y-3 mb-6">
                  {selectedStudios.length > 0 ? (
                    <>
                      {studios
                        .filter((studio) => selectedStudios.includes(studio.id))
                        .map((studio) => {
                          if (
                            !apiPoints ||
                            !apiPoints.studio ||
                            !apiPoints.studio[studio.id]
                          ) {
                            return (
                              <div key={studio.id} className="text-sm">
                                <div className="font-semibold mb-1">Studio</div>
                                <div className="text-gray-700">
                                  {studio.studio_points} pts
                                </div>
                              </div>
                            );
                          }

                          const pointsData = apiPoints.studio[studio.id];
                          const parts: string[] = [];
                          let totalPoints = 0;

                          // Process daily points
                          if (
                            pointsData.daily &&
                            Array.isArray(pointsData.daily)
                          ) {
                            pointsData.daily.forEach(
                              (pts: number, idx: number) => {
                                parts.push(`${pts.toFixed(2)}pts x 1day`);
                                totalPoints += pts;
                              }
                            );
                          }

                          // Process hourly points
                          if (
                            pointsData.hourly &&
                            Array.isArray(pointsData.hourly)
                          ) {
                            pointsData.hourly.forEach((h: string) => {
                              const [hours, points] = h
                                .split("|")
                                .map((v) => parseFloat(v));
                              parts.push(
                                `${points.toFixed(2)}pts x ${hours}hour`
                              );
                              totalPoints += points;
                            });
                          }

                          const breakdown =
                            parts.length > 0
                              ? `${parts.join(" + ")} = ${totalPoints.toFixed(
                                  2
                                )} pts`
                              : `${totalPoints.toFixed(2)} pts`;

                          return (
                            <div key={studio.id} className="text-sm">
                              <div className="font-semibold mb-1">Studio</div>
                              <div className="text-gray-700">{breakdown}</div>
                            </div>
                          );
                        })}
                    </>
                  ) : (
                    <div className="text-sm text-gray-500">
                      No studio selected
                    </div>
                  )}
                  <hr className="border-gray-300" />
                  <div className="flex justify-between font-medium">
                    <span>Total</span>
                    <span>{totalPoints.toFixed(2)} pts</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Points left</span>
                    <span>{(userPoints - totalPoints).toFixed(2)} pts</span>
                  </div>
                </div>

                <div className="space-y-3">
                  <Button
                    onClick={handleSkip}
                    variant="outline-black"
                    size="md"
                    className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-md"
                  >
                    Skip
                  </Button>

                  <Button
                    onClick={handleContinue}
                    disabled={selectedStudios.length === 0}
                    variant="fill-black"
                    size="md"
                    className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-lg disabled:opacity-50 disabled:cursor-not-allowed"
                  >
                    Continue
                  </Button>
                </div>
              </div>
            </div>
          </div>
        </main>

        {/* <AppFooter /> */}
      </div>
    </ProtectedRoute>
  );
}
