import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useState, useEffect } from "react";
import { useNavigate } from "@remix-run/react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { useLanguage } from "~/hooks/useLanguage";

export const meta: MetaFunction = () => {
  return [
    { title: "Select Additional Support - Nomadics Studios" },
    {
      name: "description",
      content: "Select additional support for your studio session.",
    },
  ];
};

interface BookingData {
  location: string;
  location_name: string;
  start_date: string;
  end_date: string;
  start_time: string;
  end_time: string;
}

interface Resource {
  id: number;
  uuid: string;
  fname: string;
  lname: string;
  email: string;
  profile_pic: string;
  description: string;
  languages: string;
  daily_rate: string;
  hourly_rate: string;
  available: boolean;
  booking_days: number;
  booking_hours: number;
  resource_points: number;
  resource_name: string | null;
  schedules: any[];
  bookings: any[];
}

interface PreferenceResponse {
  status: boolean;
  data: {
    type: string;
    resources: Resource[];
    advance_date_flag: string;
    points: any;
  };
}

export default function BookSupport() {
  const navigate = useNavigate();
  const { t } = useLanguage();
  const [bookingData, setBookingData] = useState<BookingData | null>(null);
  const [resources, setResources] = useState<Resource[]>([]);
  const [locations, setLocations] = useState<Record<string, string>>({});
  const [selectedSupport, setSelectedSupport] = useState<number[]>([]);
  const [selectedLanguage, setSelectedLanguage] = useState("English");
  const [loading, setLoading] = useState(true);
  const [totalResourcePoints, setTotalResourcePoints] = useState(0);
  const [studioPoints, setStudioPoints] = useState(0);
  const [equipmentPoints, setEquipmentPoints] = useState(0);
  const [userPoints, setUserPoints] = useState(0);
  const [apiPoints, setApiPoints] = useState<any>(null);
  const [validationError, setValidationError] = useState<string | null>(null);
  const [isProfileBooking, setIsProfileBooking] = useState(false);

  const languages = ["English", "Spanish", "French", "German", "Portuguese"];

  // Calculate booking duration from API points data
  const calculateDuration = () => {
    if (!apiPoints || !apiPoints.resource) return "0D 0H";

    let totalDays = 0;
    let totalHours = 0;

    // Iterate through all selected resources to get max duration
    resources
      .filter((resource) => selectedSupport.includes(resource.id))
      .forEach((resource) => {
        const pointsData = apiPoints.resource[resource.id];
        if (pointsData) {
          // Count days from daily array length
          if (pointsData.daily && Array.isArray(pointsData.daily)) {
            totalDays = Math.max(totalDays, pointsData.daily.length);
          }

          // Extract hours from hourly array
          if (pointsData.hourly && Array.isArray(pointsData.hourly)) {
            const hours = pointsData.hourly.reduce((sum: number, h: string) => {
              const [hrs] = h.split("|").map((v) => parseFloat(v));
              return sum + hrs;
            }, 0);
            totalHours = Math.max(totalHours, hours);
          }
        }
      });

    return `${totalDays}D ${totalHours}H`;
  };

  // Load booking data and fetch resources
  useEffect(() => {
    const loadBookingData = () => {
      // First check for booking data from profile page
      const currentBooking = sessionStorage.getItem("currentBooking");
      
      if (currentBooking) {
        const profileBookingData = JSON.parse(currentBooking);
        
        // Mark this as a profile booking (requires resource selection)
        setIsProfileBooking(true);
        
        // Transform profile booking data to match the API format
        const transformedData: BookingData = {
          location: "bali", // Default location, you may need to adjust this based on your needs
          location_name: "Bali",
          start_date: profileBookingData.start.split('T')[0],
          end_date: profileBookingData.end.split('T')[0],
          start_time: profileBookingData.start.split('T')[1] || "09:00",
          end_time: profileBookingData.end.split('T')[1] || "17:00",
        };
        
        // Store in bookingData format
        sessionStorage.setItem("bookingData", JSON.stringify(transformedData));
        setBookingData(transformedData);
        fetchResources(transformedData);
        
        // Clear the current booking after processing
        sessionStorage.removeItem("currentBooking");
      } else {
        // Fall back to existing bookingData
        const stored = sessionStorage.getItem("bookingData");
        if (stored) {
          const data = JSON.parse(stored);
          setBookingData(data);
          fetchResources(data);
        } else {
          // If no booking data, redirect to home
          navigate("/home");
        }
      }
    };

    // Get studio and equipment points from previous steps
    const storedStudioPoints = sessionStorage.getItem("studioPoints");
    if (storedStudioPoints) {
      setStudioPoints(parseInt(storedStudioPoints));
    }

    const storedEquipmentPoints = sessionStorage.getItem("equipmentPoints");
    if (storedEquipmentPoints) {
      setEquipmentPoints(parseInt(storedEquipmentPoints));
    }

    // Get locations from previous step if available
    const storedLocations = sessionStorage.getItem("locations");
    if (storedLocations) {
      setLocations(JSON.parse(storedLocations));
    }

    // Get user points
    const userData =
      localStorage.getItem("userData") ||
      localStorage.getItem("userFullResponse");
    if (userData) {
      try {
        const parsed = JSON.parse(userData);
        const user = parsed.user || parsed;
        setUserPoints(user.point_balance || user.points || 0);
      } catch (e) {
        console.error("Error parsing user data:", e);
      }
    }

    loadBookingData();
  }, []);

  const fetchResources = async (data: BookingData) => {
    try {
      setLoading(true);
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        console.error("No auth token found");
        return;
      }

      const response = await fetch(
        "https://dashboard.nomadicstudios.net/api/choose-your-preference/resource",
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${token}`,
          },
          body: JSON.stringify(data),
        }
      );

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result: PreferenceResponse = await response.json();

      if (result.status && result.data) {
        setResources(result.data.resources || []);
        setApiPoints(result.data.points || null);
      }
    } catch (error) {
      console.error("Failed to fetch resources:", error);
    } finally {
      setLoading(false);
    }
  };

  const handleSupportSelect = (resourceId: number) => {
    setSelectedSupport((prev) => {
      if (prev.includes(resourceId)) {
        return prev.filter((id) => id !== resourceId);
      } else {
        return [...prev, resourceId];
      }
    });
  };

  // Calculate total resource points using API points data
  useEffect(() => {
    if (!apiPoints || !apiPoints.resource) {
      setTotalResourcePoints(0);
      return;
    }

    const total = resources
      .filter((resource) => selectedSupport.includes(resource.id))
      .reduce((sum, resource) => {
        const resourcePointsData = apiPoints.resource[resource.id];
        if (!resourcePointsData) return sum;

        let resourceTotal = 0;

        // Calculate from daily array
        if (
          resourcePointsData.daily &&
          Array.isArray(resourcePointsData.daily)
        ) {
          resourceTotal += resourcePointsData.daily.reduce(
            (s: number, p: number) => s + p,
            0
          );
        }

        // Calculate from hourly array (format: "hours|points")
        if (
          resourcePointsData.hourly &&
          Array.isArray(resourcePointsData.hourly)
        ) {
          resourceTotal += resourcePointsData.hourly.reduce(
            (s: number, h: string) => {
              const [hours, points] = h.split("|").map((v) => parseFloat(v));
              return s + points;
            },
            0
          );
        }

        return sum + resourceTotal;
      }, 0);
    setTotalResourcePoints(total);
  }, [selectedSupport, resources, apiPoints]);

  const handleSkip = () => {
    // Store selected resources (if any) and navigate to confirmation
    sessionStorage.setItem(
      "selectedResources",
      JSON.stringify(selectedSupport)
    );
    sessionStorage.setItem("resourcePoints", totalResourcePoints.toString());
    sessionStorage.setItem("resourceApiPoints", JSON.stringify(apiPoints));
    navigate("/book-confirmation");
  };

  const handleContinue = () => {
    // Clear any previous validation errors
    setValidationError(null);

    // Calculate total cost
    const totalCost = studioPoints + equipmentPoints + totalResourcePoints;
    
    // Validation 0: Check if user has 0 points balance
    if (userPoints === 0) {
      setValidationError("Insufficient points. Please add points to your account to continue.");
      return;
    }

    // Validation 1: For profile bookings, resource selection is MANDATORY
    if (isProfileBooking && selectedSupport.length === 0) {
      setValidationError("Please select at least one resource to continue.");
      return;
    }

    // Validation 2: Check if total cost is valid (not 0 when resources are selected)
    if (selectedSupport.length > 0 && totalCost === 0) {
      setValidationError("Unable to calculate booking cost. Please try again.");
      return;
    }

    // Validation 3: Check if user has sufficient points
    const pointsLeft = userPoints - totalCost;
    if (pointsLeft < 0) {
      setValidationError(`Insufficient points. You need ${Math.abs(pointsLeft).toFixed(2)} more points. Please add points to your account.`);
      return;
    }

    // Store selected resources and navigate to confirmation
    sessionStorage.setItem(
      "selectedResources",
      JSON.stringify(selectedSupport)
    );
    sessionStorage.setItem("resourcePoints", totalResourcePoints.toString());
    sessionStorage.setItem("resourceApiPoints", JSON.stringify(apiPoints));
    navigate("/book-confirmation");
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />

        {/* Hero Section */}
        <section className="relative bg-black text-white py-12 md:py-16">
          <div className="absolute inset-0 bg-black/20" />
          <div className="relative z-10 max-w-7xl mx-auto px-4">
            <h1 className="text-5xl md:text-6xl lg:text-7xl font-light mb-4">
              {t("booking.hero.title")}
            </h1>
          </div>
        </section>

        <main className="max-w-7xl mx-auto px-4 md:px-6 lg:px-8 py-8 md:py-12">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 md:gap-8 lg:gap-12">
            {/* Left Column - Support Selection */}
            <div className="lg:col-span-2">
              <div className="mb-8 flex items-center justify-between">
                <h2 className="text-2xl font-light text-black">
                  3. Select additional support
                </h2>

                {/* Language Selector */}
                {/* <div className="flex items-center gap-2">
                <select 
                  value={selectedLanguage}
                  onChange={(e) => setSelectedLanguage(e.target.value)}
                  className="p-2 border border-gray-300 bg-white text-black focus:border-black focus:outline-none"
                >
                  {languages.map(lang => (
                    <option key={lang} value={lang}>{lang}</option>
                  ))}
                </select>
              </div> */}
              </div>

              {loading ? (
                <div className="text-center py-20">
                  <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-black"></div>
                  <p className="mt-4 text-gray-600">Loading resources...</p>
                </div>
              ) : resources.length > 0 ? (
                <div className="space-y-6">
                  {resources
                    .filter(
                      (resource) =>
                        resource.languages &&
                        resource.languages
                          .toLowerCase()
                          .includes(selectedLanguage.toLowerCase())
                    )
                    .map((resource) => (
                      <div
                        key={resource.id}
                        className={`border-2 transition-all ${
                          !resource.available
                            ? "border-gray-300 bg-gray-100 cursor-not-allowed opacity-60"
                            : selectedSupport.includes(resource.id)
                            ? "border-black bg-gray-50 cursor-pointer"
                            : "border-gray-200 hover:border-gray-300 cursor-pointer"
                        }`}
                        onClick={() =>
                          resource.available && handleSupportSelect(resource.id)
                        }
                      >
                        <div className="p-6">
                          <div className="flex items-start gap-6">
                            {/* Checkbox */}
                            <div className="flex-shrink-0 mt-1">
                              <div
                                className={`w-6 h-6 border-2 transition-colors ${
                                  !resource.available
                                    ? "border-gray-300 bg-gray-200 cursor-not-allowed"
                                    : selectedSupport.includes(resource.id)
                                    ? "border-black bg-black"
                                    : "border-gray-400"
                                }`}
                              >
                                {selectedSupport.includes(resource.id) &&
                                  resource.available && (
                                    <svg
                                      className="w-full h-full text-white p-1"
                                      fill="currentColor"
                                      viewBox="0 0 20 20"
                                    >
                                      <path
                                        fillRule="evenodd"
                                        d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                        clipRule="evenodd"
                                      />
                                    </svg>
                                  )}
                              </div>
                            </div>

                            {/* Content */}
                            <div className="flex-1">
                              <div className="flex items-start justify-between mb-4">
                                <div>
                                  <h3 className="text-2xl md:text-3xl font-medium text-black mb-1">
                                    {resource.fname} {resource.lname}
                                  </h3>
                                  {!resource.available && (
                                    <p className="text-base md:text-lg text-red-600 mb-2">
                                      {t("booking.support.unavailable")}
                                    </p>
                                  )}
                                  <p className="text-sm text-gray-600 mb-2">
                                    {resource.resource_name || "Resource"}
                                  </p>
                                  {resource.languages && (
                                    <p className="text-sm text-blue-600 mb-3">
                                      Languages: {resource.languages}
                                    </p>
                                  )}

                                  {resource.description && (
                                    <div className="mb-4">
                                      <h4 className="font-medium text-black mb-2">
                                        Description
                                      </h4>
                                      <p className="text-sm text-gray-600">
                                        {resource.description}
                                      </p>
                                    </div>
                                  )}

                                  <div className="text-sm text-gray-600">
                                    <p>Daily rate: ${resource.daily_rate}</p>
                                    <p>Hourly rate: ${resource.hourly_rate}</p>
                                  </div>
                                </div>
                                <span className="text-lg font-medium text-black">
                                  {resource.resource_points}pts
                                </span>
                              </div>
                            </div>

                            {/* Person Image */}
                            <div className="flex-shrink-0">
                              <img
                                src={
                                  resource.profile_pic
                                    ? `https://dashboard.nomadicstudios.net/storage/profile-pic/${resource.profile_pic}`
                                    : `https://dashboard.nomadicstudios.net/storage/profile-pic/default-avatar.jpg`
                                }
                                alt={`${resource.fname} ${resource.lname}`}
                                className="w-32 h-32 object-cover rounded"
                                loading="lazy"
                                onError={(e) => {
                                  console.log(
                                    "Resource image failed to load:",
                                    e.currentTarget.src
                                  );
                                  const initials =
                                    resource.fname.charAt(0) +
                                    (resource.lname
                                      ? resource.lname.charAt(0)
                                      : "");
                                  e.currentTarget.src = `data:image/svg+xml;base64,${btoa(
                                    `<svg width="128" height="128" viewBox="0 0 128 128" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="128" height="128" fill="#E5E7EB"/><text x="64" y="64" font-family="Arial" font-size="32" fill="#6B7280" text-anchor="middle" dominant-baseline="middle">${initials}</text></svg>`
                                  )}`;
                                }}
                                onLoad={() => {
                                  console.log(
                                    "Resource image loaded successfully:",
                                    resource.profile_pic
                                  );
                                }}
                              />
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                </div>
              ) : (
                <div className="text-center py-20">
                  <p className="text-gray-600">
                    No resources available for the selected criteria.
                  </p>
                </div>
              )}
            </div>

            {/* Right Column - Booking Summary */}
            <div className="lg:col-span-1">
              <div className="bg-gray-50 p-6 sticky top-8">
                <div className="text-center mb-8">
                  <div className="text-4xl font-light text-black mb-2">
                    {calculateDuration()}
                  </div>
                </div>

                {bookingData && (
                  <div className="space-y-4 mb-8">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                        Where
                      </label>
                      <input
                        type="text"
                        value={
                          bookingData.location_name ||
                          locations[bookingData.location] ||
                          bookingData.location
                        }
                        disabled
                        className="w-full p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      />
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                        Start
                      </label>
                      <div className="grid grid-cols-2 gap-2">
                        <input
                          type="text"
                          value={bookingData.start_date}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                        <input
                          type="text"
                          value={bookingData.start_time.substring(0, 5)}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                      </div>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                        End
                      </label>
                      <div className="grid grid-cols-2 gap-2">
                        <input
                          type="text"
                          value={bookingData.end_date}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                        />
                        <input
                          type="text"
                          value={bookingData.end_time.substring(0, 5)}
                          disabled
                          className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:outline-none transition-colors"
                        />
                      </div>
                    </div>
                  </div>
                )}

                {/* Price Breakdown */}
                <div className="space-y-3 mb-6">
                  {studioPoints > 0 && (
                    <div className="flex justify-between text-sm">
                      <span className="font-semibold">Studio</span>
                      <span>{studioPoints} pts</span>
                    </div>
                  )}
                  {equipmentPoints > 0 && (
                    <div className="flex justify-between text-sm">
                      <span className="font-semibold">Equipment</span>
                      <span>{equipmentPoints} pts</span>
                    </div>
                  )}
                  {selectedSupport.length > 0 ? (
                    resources
                      .filter((resource) =>
                        selectedSupport.includes(resource.id)
                      )
                      .map((resource) => {
                        if (
                          !apiPoints ||
                          !apiPoints.resource ||
                          !apiPoints.resource[resource.id]
                        ) {
                          return (
                            <div key={resource.id} className="text-sm">
                              <div className="font-semibold mb-1">Resource</div>
                              <div className="text-gray-700">
                                {resource.resource_points} pts
                              </div>
                            </div>
                          );
                        }
                        const pointsData = apiPoints.resource[resource.id];
                        const parts: string[] = [];
                        let totalPoints = 0;

                        // Process daily points
                        if (
                          pointsData.daily &&
                          Array.isArray(pointsData.daily)
                        ) {
                          pointsData.daily.forEach(
                            (pts: number, idx: number) => {
                              parts.push(`${pts.toFixed(2)}pts x 1day`);
                              totalPoints += pts;
                            }
                          );
                        }

                        // Process hourly points
                        if (
                          pointsData.hourly &&
                          Array.isArray(pointsData.hourly)
                        ) {
                          pointsData.hourly.forEach((h: string) => {
                            const [hours, points] = h
                              .split("|")
                              .map((v) => parseFloat(v));
                            parts.push(
                              `${points.toFixed(2)}pts x ${hours}hour`
                            );
                            totalPoints += points;
                          });
                        }

                        const breakdown =
                          parts.length > 0
                            ? `${parts.join(" + ")} = ${totalPoints.toFixed(
                                2
                              )} pts`
                            : `${totalPoints.toFixed(2)} pts`;
                        return (
                          <div key={resource.id} className="text-sm">
                            <div className="font-semibold mb-1">Resource</div>
                            <div className="text-gray-700">{breakdown}</div>
                          </div>
                        );
                      })
                  ) : (
                    <div className="text-sm text-gray-500">
                      No resources selected
                    </div>
                  )}
                  <hr className="border-gray-300" />
                  <div className="flex justify-between font-medium">
                    <span>Total</span>
                    <span>
                      {(
                        studioPoints +
                        equipmentPoints +
                        totalResourcePoints
                      ).toFixed(2)}{" "}
                      pts
                    </span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Points left</span>
                    <span>
                      {(
                        userPoints -
                        studioPoints -
                        equipmentPoints -
                        totalResourcePoints
                      ).toFixed(2)}{" "}
                      pts
                    </span>
                  </div>
                </div>

                <div className="space-y-3">
                  {/* Validation Error Message */}
                  {validationError && (
                    <div className="p-3 bg-red-50 border border-red-200 rounded-lg text-red-700 text-sm">
                      {validationError}
                    </div>
                  )}

                  {/* Warning if points will be insufficient */}
                  {selectedSupport.length > 0 && 
                   (userPoints - studioPoints - equipmentPoints - totalResourcePoints) < 0 && (
                    <div className="p-3 bg-yellow-50 border border-yellow-200 rounded-lg text-yellow-700 text-sm">
                      ⚠️ Insufficient points for this booking
                    </div>
                  )}

                  <Button
                    onClick={handleContinue}
                    variant="fill-black"
                    size="md"
                    className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-lg">
                    Review Booking
                  </Button>
                </div>
              </div>
            </div>
          </div>
        </main>

        {/* <AppFooter /> */}
      </div>
    </ProtectedRoute>
  );
}