import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useParams, Link } from "@remix-run/react";
import { useState, useEffect } from "react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { useLanguage } from "~/hooks/useLanguage";
import { API_BASE_URL } from "~/lib/api";

export const meta: MetaFunction = () => {
  return [
    { title: "Booking Confirmed - Nomadic Studios" },
    {
      name: "description",
      content: "Your booking has been confirmed successfully.",
    },
  ];
};

interface BookingConfirmationData {
  bookingId: string;
  booking: {
    status: number;
    start_date: string;
    end_date: string;
    start_time: string;
    end_time: string;
    total_points: number;
    location: string;
    banner_picture: string;
  };
  bookings: Array<{
    studio_id?: number;
    equipment_id?: number;
    resource_id?: number;
    type: string;
    points: number;
    data: string;
    status: number;
  }>;
  studio?: {
    id: number;
    name: string;
    landing_page: {
      id: number;
      banner_picture: string;
      studios: string;
      description: string;
    };
  };
  equipment?: {
    id: number;
    name: string;
    photo: string;
    description: string;
  };
  resource?: {
    id: number;
    fname: string;
    lname: string;
    profile_pic: string;
    description: string;
    resource_name: {
      resourceId: number;
      id: number;
      name: string;
    };
  };
  isCheckedIn: boolean;
}

export default function BookingConfirmed() {
  const { bookingId } = useParams();
  const { t } = useLanguage();
  const [confirmationData, setConfirmationData] =
    useState<BookingConfirmationData | null>(null);
  const [loading, setLoading] = useState(true);
  const [showCheckoutForm, setShowCheckoutForm] = useState(false);
  const [checkoutData, setCheckoutData] = useState<any>(null);
  const [showCheckoutModal, setShowCheckoutModal] = useState(false);
  const [userPoints, setUserPoints] = useState(0);
  const [checkoutForm, setCheckoutForm] = useState({
    description: "",
    files: [] as File[],
  });
  const [cancelLoading, setCancelLoading] = useState(false);
  const [completeLoading, setCompleteLoading] = useState(false);

  // Handle Cancel Booking
  const handleCancelBooking = async () => {
    if (!bookingId) {
      alert("Booking ID not found.");
      return;
    }

    if (!confirm("Are you sure you want to cancel this booking?")) {
      return;
    }

    try {
      setCancelLoading(true);
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        alert("Authentication required.");
        return;
      }

      const formData = new FormData();
      formData.append("booking", bookingId);

      const response = await fetch(`${API_BASE_URL}/booking-cancelled`, {
        method: "POST",
        headers: {
          Authorization: `Bearer ${token}`,
        },
        body: formData,
      });

      if (response.ok) {
        const result = await response.json();
        alert("Booking cancelled successfully!");
        // Optionally redirect to bookings page or home
        window.location.href = "/account-settings";
      } else {
        alert("Failed to cancel booking. Please try again.");
      }
    } catch (error) {
      console.error("Cancel booking error:", error);
      alert("Failed to cancel booking. Please try again.");
    } finally {
      setCancelLoading(false);
    }
  };

  // Handle Complete Booking
  const handleCompleteBooking = async () => {
    if (!bookingId) {
      alert("Booking ID not found.");
      return;
    }

    if (!confirm("Mark this booking as complete?")) {
      return;
    }

    try {
      setCompleteLoading(true);
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        alert("Authentication required.");
        return;
      }

      const formData = new FormData();
      formData.append("booking", bookingId);

      const response = await fetch(`${API_BASE_URL}/booking-completed`, {
        method: "POST",
        headers: {
          Authorization: `Bearer ${token}`,
        },
        body: formData,
      });

      if (response.ok) {
        const result = await response.json();
        alert("Booking completed successfully!");
        // Refresh the page to show updated status
        window.location.reload();
      } else {
        alert("Failed to complete booking. Please try again.");
      }
    } catch (error) {
      console.error("Complete booking error:", error);
      alert("Failed to complete booking. Please try again.");
    } finally {
      setCompleteLoading(false);
    }
  };

  // Format date range with hours
  const formatDateRange = (
    startDate: string,
    endDate: string,
    startTime: string,
    endTime: string
  ) => {
    const start = new Date(`${startDate}T${startTime}`);
    const end = new Date(`${endDate}T${endTime}`);
    const options: Intl.DateTimeFormatOptions = {
      day: "numeric",
      month: "short",
      year: "numeric",
    };

    const formattedStart = start.toLocaleDateString("en-US", options);
    const formattedEnd = end.toLocaleDateString("en-US", options);

    // Calculate duration from bookings data
    let totalDays = 0;
    let totalHours = 0;

    if (confirmationData?.bookings) {
      confirmationData.bookings.forEach((booking) => {
        if (booking.data) {
          try {
            const bookingData = JSON.parse(booking.data);

            // Count days from daily array
            if (bookingData.daily && Array.isArray(bookingData.daily)) {
              totalDays = Math.max(totalDays, bookingData.daily.length);
            }

            // Extract hours from hourly array
            if (bookingData.hourly && Array.isArray(bookingData.hourly)) {
              const hours = bookingData.hourly.reduce(
                (sum: number, h: string) => {
                  const [hrs] = h.split("|").map((v: string) => parseFloat(v));
                  return sum + hrs;
                },
                0
              );
              totalHours = Math.max(totalHours, hours);
            }
          } catch (e) {
            console.error("Error parsing booking data:", e);
          }
        }
      });
    }

    return `${formattedStart} - ${formattedEnd} (${totalDays} day${
      totalDays !== 1 ? "s" : ""
    } ${totalHours} hour${totalHours !== 1 ? "s" : ""})`;
  };

  // Fetch booking confirmation data from API
  const fetchBookingConfirmation = async (id: string) => {
    try {
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) return;

      // Get booking data for POST request - must use actual user selection
      const storedBookingData = sessionStorage.getItem("bookingData");
      if (!storedBookingData) {
        console.error("No booking data found in sessionStorage");
        return;
      }

      const requestBody = JSON.parse(storedBookingData);

      const response = await fetch(`${API_BASE_URL}/booking-confirmed/${id}`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${token}`,
        },
        body: JSON.stringify(requestBody),
      });

      if (response.ok) {
        const result = await response.json();
        if (result.status && result.data) {
          setConfirmationData(result.data);
        }
      }
    } catch (error) {
      console.error("Failed to fetch booking confirmation:", error);
    }
  };

  // Handle checkout button click
  const handleCheckout = async () => {
    if (!confirmationData?.equipment?.id || !bookingId) {
      alert("Equipment information not available for checkout.");
      return;
    }

    try {
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        alert("Authentication required.");
        return;
      }

      const response = await fetch(
        `${API_BASE_URL}/equipment-checkout/${bookingId}/${confirmationData.equipment.id}`,
        {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        }
      );

      if (response.ok) {
        const result = await response.json();
        if (result.data && result.data.CheckedOutData) {
          // If checkout data exists, show it in modal
          setCheckoutData(result.data.CheckedOutData);
          setShowCheckoutModal(true);
        } else {
          // If no checkout data (CheckedOutData is null), redirect to checkout page
          window.location.href = `/equipment-checkout/${bookingId}/${confirmationData.equipment.id}`;
        }
      }
    } catch (error) {
      console.error("Checkout error:", error);
      alert("Failed to process checkout. Please try again.");
    }
  };

  // Handle checkout form submission
  const handleSubmitCheckout = async () => {
    if (!confirmationData?.equipment?.id || !bookingId) {
      alert("Equipment information not available.");
      return;
    }

    if (!checkoutForm.description.trim()) {
      alert("Please enter a description.");
      return;
    }

    try {
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        alert("Authentication required.");
        return;
      }

      const formData = new FormData();
      formData.append("booking_id", bookingId);
      formData.append("equipment_id", confirmationData.equipment.id.toString());
      formData.append("description", checkoutForm.description);

      // Append files
      checkoutForm.files.forEach((file) => {
        formData.append("files", file);
      });

      const response = await fetch(`${API_BASE_URL}/equipment-checkout-save`, {
        method: "POST",
        headers: {
          Authorization: `Bearer ${token}`,
        },
        body: formData,
      });

      if (response.ok) {
        alert("Checkout submitted successfully!");
        setShowCheckoutForm(false);
        setCheckoutForm({ description: "", files: [] });
        // Refresh checkout data
        handleCheckout();
      } else {
        alert("Failed to submit checkout. Please try again.");
      }
    } catch (error) {
      console.error("Submit checkout error:", error);
      alert("Failed to submit checkout. Please try again.");
    }
  };

  // Handle file selection
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(e.target.files || []);
    setCheckoutForm((prev) => ({
      ...prev,
      files: [...prev.files, ...files],
    }));
  };

  // Remove file from selection
  const removeFile = (index: number) => {
    setCheckoutForm((prev) => ({
      ...prev,
      files: prev.files.filter((_, i) => i !== index),
    }));
  };

  useEffect(() => {
    const loadConfirmationData = async () => {
      try {
        setLoading(true);

        // Get user points
        const userData =
          localStorage.getItem("userData") ||
          localStorage.getItem("userFullResponse");
        if (userData) {
          try {
            const parsed = JSON.parse(userData);
            const user = parsed.user || parsed;
            setUserPoints(user.point_balance || user.points || 0);
          } catch (e) {
            console.error("Error parsing user data:", e);
          }
        }

        // First try to get data from sessionStorage (for immediate navigation after booking)
        const storedData = sessionStorage.getItem("bookingConfirmationData");
        if (storedData) {
          const data = JSON.parse(storedData);
          setConfirmationData(data);
          setLoading(false);
          return;
        }

        // If no stored data and we have a booking ID, fetch from API
        if (bookingId) {
          await fetchBookingConfirmation(bookingId);
        }
      } catch (error) {
        console.error("Error loading confirmation data:", error);
      } finally {
        setLoading(false);
      }
    };

    loadConfirmationData();
  }, [bookingId]);

  if (loading) {
    return (
      <ProtectedRoute>
        <div className="min-h-screen bg-white text-black">
          <AppHeader theme="dark" />
          <div className="max-w-7xl mx-auto px-4 py-20 text-center">
            <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-black"></div>
            <p className="mt-4 text-gray-600">
              Loading confirmation details...
            </p>
          </div>
        </div>
      </ProtectedRoute>
    );
  }

  if (!confirmationData) {
    return (
      <ProtectedRoute>
        <div className="min-h-screen bg-white text-black">
          <AppHeader theme="dark" />
          <div className="max-w-7xl mx-auto px-4 py-20 text-center">
            <h1 className="text-2xl font-medium text-black mb-4">
              Booking Not Found
            </h1>
            <p className="text-gray-600 mb-8">
              The booking confirmation data could not be found.
            </p>
            <Link to="/">
              <Button variant="fill-black" size="lg">
                Return to Home
              </Button>
            </Link>
          </div>
        </div>
      </ProtectedRoute>
    );
  }

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />

        {/* Hero Section */}
        <section className="relative bg-black text-white py-16">
          <div className="absolute inset-0 bg-black/20" />
          <div className="relative z-10 max-w-7xl mx-auto px-4">
            <h1 className="text-5xl md:text-6xl lg:text-7xl font-light mb-4">
              Create something with us!
            </h1>
          </div>
        </section>

        {/* Booking Image */}
        <section className="w-full">
          <img
            src={
              confirmationData.booking.banner_picture
                ? `https://dashboard.nomadicstudios.net/images/${confirmationData.booking.banner_picture}`
                : "https://dashboard.nomadicstudios.net/images/booking-image.png"
            }
            alt="Booking confirmation"
            className="w-full h-80 object-cover"
            loading="lazy"
            onError={(e) => {
              const target = e.currentTarget as HTMLImageElement;
              if (!target.dataset.errorHandled) {
                target.dataset.errorHandled = "true";
                console.log(
                  "Booking banner image failed to load, using fallback"
                );
                target.src =
                  "https://dashboard.nomadicstudios.net/images/booking-image.png";
              }
            }}
          />
        </section>

        {/* Confirmation Message */}
        <section className="py-12 text-center bg-gray-50">
          <h2 className="text-3xl md:text-4xl font-medium text-black mb-2">
            {t("booking.confirmation.confirmed")}
          </h2>
          <p className="text-xl md:text-2xl text-gray-600">
            {t("booking.confirmation.confirmedMessage")}
          </p>
        </section>

        {/* Details Section */}
        <main className="max-w-7xl mx-auto px-4 md:px-6 lg:px-8 py-8 md:py-12">
          <div className="grid grid-cols-1 gap-12">
            {/* Details Section */}
            <div className="space-y-8">
              <h3 className="text-4xl md:text-5xl font-light text-black">
                {t("booking.confirmation.details")}
              </h3>

              {/* Location Section */}
              <div className="pb-8">
                <h4 className="text-2xl md:text-3xl font-medium text-black mb-4">
                  {t("booking.confirmation.location")}
                </h4>
                <div className="flex items-center gap-6">
                  <img
                    src={
                      confirmationData.booking.banner_picture
                        ? `https://dashboard.nomadicstudios.net/storage/location/${confirmationData.booking.banner_picture}`
                        : `https://dashboard.nomadicstudios.net/storage/location/default-location.jpg`
                    }
                    alt={confirmationData.booking.location}
                    className="w-60 h-40 object-cover"
                    loading="lazy"
                    onError={(e) => {
                      const target = e.currentTarget as HTMLImageElement;
                      if (!target.dataset.errorHandled) {
                        target.dataset.errorHandled = "true";
                        console.log(
                          "Location image failed to load, using fallback"
                        );
                        target.src =
                          "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQwIiBoZWlnaHQ9IjE2MCIgdmlld0JveD0iMCAwIDI0MDE2MCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHJlY3Qgd2lkdGg9IjI0MCIgaGVpZ2h0PSIxNjAiIGZpbGw9IiNGM0Y0RjYiLz4KPHR5eHQgeD0iMTIwIiB5PSI4MCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOUNBM0FGIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBkb21pbmFudC1iYXNlbGluZT0ibWlkZGxlIj5Mb2NhdGlvbjwvdGV4dD4KPC9zdmc+Cg==";
                      }
                    }}
                  />
                  <div>
                    <p className="text-3xl md:text-4xl font-light text-black">
                      {confirmationData.booking.location}
                    </p>
                  </div>
                </div>
              </div>

              {/* Date Section */}
              <div className="pb-8">
                <h4 className="text-2xl md:text-3xl font-medium text-black mb-4">
                  {t("booking.confirmation.date")}
                </h4>
                <p className="text-2xl md:text-3xl text-gray-700">
                  {formatDateRange(
                    confirmationData.booking.start_date,
                    confirmationData.booking.end_date,
                    confirmationData.booking.start_time,
                    confirmationData.booking.end_time
                  )}
                </p>
                <p className="text-base md:text-lg text-gray-500 mt-2">
                  {confirmationData.booking.start_time.substring(0, 5)} -{" "}
                  {confirmationData.booking.end_time.substring(0, 5)}
                </p>
              </div>

              {/* Type of Studio Section */}
              {confirmationData.studio && (
                <div className="pb-8">
                  <h4 className="text-lg font-medium text-black mb-4">
                    Type of studio
                  </h4>
                  <div className="flex items-start gap-6">
                    <img
                      src={
                        confirmationData.studio.landing_page?.banner_picture
                          ? `https://dashboard.nomadicstudios.net/storage/location/${confirmationData.studio.landing_page.banner_picture}`
                          : `https://dashboard.nomadicstudios.net/storage/location/default-studio.jpg`
                      }
                      alt={confirmationData.studio.name}
                      className="w-32 h-32 object-cover"
                      loading="lazy"
                      onError={(e) => {
                        const target = e.currentTarget as HTMLImageElement;
                        console.log("Studio image failed to load:", target.src);
                        target.src =
                          "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTI4IiBoZWlnaHQ9IjEyOCIgdmlld0JveD0iMCAwIDEyOCAxMjgiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSIxMjgiIGhlaWdodD0iMTI4IiBmaWxsPSIjRjNGNEY2Ii8+Cjx0ZXh0IHg9IjY0IiB5PSI2NCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOUNBM0FGIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBkb21pbmFudC1iYXNlbGluZT0ibWlkZGxlIj5TdHVkaW88L3RleHQ+Cjwvc3ZnPgo=";
                      }}
                    />
                    <div className="flex-1">
                      <div className="flex justify-between items-start mb-2">
                        <h5 className="text-2xl md:text-3xl font-medium text-black">
                          {confirmationData.studio.name}
                        </h5>
                        <span className="text-2xl md:text-3xl font-medium text-black">
                          ={" "}
                          {confirmationData.bookings.find(
                            (b) => b.type === "studio"
                          )?.points || 0}{" "}
                          pts
                        </span>
                      </div>
                      {(() => {
                        const studioBooking = confirmationData.bookings.find(
                          (b) => b.type === "studio"
                        );
                        if (studioBooking?.data) {
                          try {
                            const dailyData = JSON.parse(studioBooking.data);
                            if (
                              dailyData.daily &&
                              Array.isArray(dailyData.daily)
                            ) {
                              return dailyData.daily.map(
                                (points: number, index: number) => (
                                  <p
                                    key={index}
                                    className="text-sm text-gray-600"
                                  >
                                    {points}pts x 1day
                                  </p>
                                )
                              );
                            }
                          } catch (e) {
                            console.error("Error parsing studio data:", e);
                          }
                        }
                        return (
                          <p className="text-sm text-gray-600 mb-2">
                            {confirmationData.bookings.find(
                              (b) => b.type === "studio"
                            )?.points || 0}
                            pts x 1day
                          </p>
                        );
                      })()}
                      <div
                        className="text-sm text-gray-600"
                        dangerouslySetInnerHTML={{
                          __html:
                            confirmationData.studio.landing_page?.description ||
                            "",
                        }}
                      />
                    </div>
                  </div>
                </div>
              )}

              {/* Equipment Section */}
              {confirmationData.equipment && (
                <div className="pb-8">
                  <h4 className="text-lg font-medium text-black mb-4">
                    Equipment
                  </h4>
                  <div className="grid grid-cols-1 lg:grid-cols-4 gap-6 items-start">
                    {/* Equipment Image */}
                    <div>
                      <img
                        src={
                          confirmationData.equipment.photo
                            ? `https://dashboard.nomadicstudios.net/storage/rental/${confirmationData.equipment.photo}`
                            : `https://dashboard.nomadicstudios.net/storage/rental/default-equipment.jpg`
                        }
                        alt={confirmationData.equipment.name}
                        className="w-32 h-24 object-cover"
                        loading="lazy"
                        onError={(e) => {
                          const target = e.currentTarget as HTMLImageElement;
                          console.log(
                            "Equipment image failed to load:",
                            target.src
                          );
                          target.src =
                            "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTI4IiBoZWlnaHQ9Ijk2IiB2aWV3Qm94PSIwIDAgMTI4IDk2IiBmaWxsPSJub25lIiB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciPgo8cmVjdCB3aWR0aD0iMTI4IiBoZWlnaHQ9Ijk2IiBmaWxsPSIjRjNGNEY2Ii8+Cjx0ZXh0IHg9IjY0IiB5PSI0OCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjEyIiBmaWxsPSIjOUNBM0FGIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBkb21pbmFudC1iYXNlbGluZT0ibWlkZGxlIj5FcXVpcG1lbnQ8L3RleHQ+Cjwvc3ZnPgo=";
                        }}
                      />
                    </div>

                    {/* Equipment Details */}
                    <div className="lg:col-span-2">
                      <div className="flex justify-between items-start mb-2">
                        <h5 className="text-2xl md:text-3xl font-medium text-black">
                          {confirmationData.equipment.name}
                        </h5>
                        <span className="text-2xl md:text-3xl font-medium text-black">
                          ={" "}
                          {confirmationData.bookings.find(
                            (b) => b.type === "rental"
                          )?.points || 0}{" "}
                          pts
                        </span>
                      </div>
                      {(() => {
                        const equipmentBooking = confirmationData.bookings.find(
                          (b) => b.type === "rental"
                        );
                        if (equipmentBooking?.data) {
                          try {
                            const dailyData = JSON.parse(equipmentBooking.data);
                            if (
                              dailyData.daily &&
                              Array.isArray(dailyData.daily)
                            ) {
                              return dailyData.daily.map(
                                (points: number, index: number) => (
                                  <p
                                    key={index}
                                    className="text-sm text-gray-600"
                                  >
                                    {points}pts x 1day
                                  </p>
                                )
                              );
                            }
                          } catch (e) {
                            console.error("Error parsing equipment data:", e);
                          }
                        }
                        return (
                          <p className="text-sm text-gray-600 mb-2">
                            {confirmationData.bookings.find(
                              (b) => b.type === "rental"
                            )?.points || 0}
                            pts x 1day
                          </p>
                        );
                      })()}
                      <p className="text-sm text-gray-600">
                        {confirmationData.equipment.description}
                      </p>
                    </div>

                    {/* Checkout Section */}
                    <div className="flex flex-col items-center gap-4">
                      <Button
                        onClick={handleCheckout}
                        variant="fill-black"
                        size="md"
                        className="bg-black text-white hover:bg-gray-800 uppercase tracking-wider font-medium"
                      >
                        Checkout
                      </Button>

                      {/* QR Code */}
                      <Link
                        to={`/equipment-checkout/${bookingId}/${confirmationData.equipment.id}`}
                      >
                        <div className="w-32 h-32 bg-white border-2 border-black flex items-center justify-center cursor-pointer hover:bg-gray-50">
                          <svg
                            width="120"
                            height="120"
                            viewBox="0 0 120 120"
                            className="w-full h-full p-2"
                          >
                            <defs>
                              <pattern
                                id="qr-equipment-pattern"
                                x="0"
                                y="0"
                                width="8"
                                height="8"
                                patternUnits="userSpaceOnUse"
                              >
                                <rect
                                  x="0"
                                  y="0"
                                  width="4"
                                  height="4"
                                  fill="#000"
                                />
                                <rect
                                  x="4"
                                  y="4"
                                  width="4"
                                  height="4"
                                  fill="#000"
                                />
                              </pattern>
                            </defs>
                            <rect
                              width="120"
                              height="120"
                              fill="url(#qr-equipment-pattern)"
                            />
                            {/* QR Code finder patterns */}
                            <rect
                              x="8"
                              y="8"
                              width="20"
                              height="20"
                              fill="#000"
                            />
                            <rect
                              x="12"
                              y="12"
                              width="12"
                              height="12"
                              fill="#fff"
                            />
                            <rect
                              x="92"
                              y="8"
                              width="20"
                              height="20"
                              fill="#000"
                            />
                            <rect
                              x="96"
                              y="12"
                              width="12"
                              height="12"
                              fill="#fff"
                            />
                            <rect
                              x="8"
                              y="92"
                              width="20"
                              height="20"
                              fill="#000"
                            />
                            <rect
                              x="12"
                              y="96"
                              width="12"
                              height="12"
                              fill="#fff"
                            />
                            {/* Additional QR pattern elements */}
                            <rect
                              x="40"
                              y="40"
                              width="8"
                              height="8"
                              fill="#000"
                            />
                            <rect
                              x="60"
                              y="20"
                              width="4"
                              height="4"
                              fill="#000"
                            />
                            <rect
                              x="20"
                              y="60"
                              width="4"
                              height="4"
                              fill="#000"
                            />
                            <rect
                              x="80"
                              y="60"
                              width="8"
                              height="8"
                              fill="#000"
                            />
                            <rect
                              x="60"
                              y="80"
                              width="4"
                              height="4"
                              fill="#000"
                            />
                          </svg>
                        </div>
                      </Link>
                    </div>
                  </div>
                </div>
              )}

              {/* Additional Support Section */}
              {confirmationData.resource && (
                <div className="pb-8">
                  <h4 className="text-lg font-medium text-black mb-4">
                    Additional support
                  </h4>
                  <div className="flex items-start gap-6">
                    <img
                      src={
                        confirmationData.resource.profile_pic
                          ? `https://dashboard.nomadicstudios.net/storage/profile-pic/${confirmationData.resource.profile_pic}`
                          : `https://dashboard.nomadicstudios.net/storage/profile-pic/default-avatar.jpg`
                      }
                      alt={`${confirmationData.resource.fname} ${confirmationData.resource.lname}`}
                      className="w-32 h-32 object-cover rounded"
                      loading="lazy"
                      onError={(e) => {
                        const target = e.currentTarget as HTMLImageElement;
                        console.log(
                          "Resource image failed to load:",
                          target.src
                        );
                        const initials =
                          confirmationData.resource!.fname.charAt(0) +
                          (confirmationData.resource!.lname
                            ? confirmationData.resource!.lname.charAt(0)
                            : "");
                        target.src = `data:image/svg+xml;base64,${btoa(
                          `<svg width="128" height="128" viewBox="0 0 128 128" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="128" height="128" fill="#E5E7EB"/><text x="64" y="64" font-family="Arial" font-size="32" fill="#6B7280" text-anchor="middle" dominant-baseline="middle">${initials}</text></svg>`
                        )}`;
                      }}
                    />
                    <div className="flex-1">
                      <div className="flex justify-between items-start mb-2">
                        <h5 className="text-2xl md:text-3xl font-medium text-black">
                          {confirmationData.resource.fname}{" "}
                          {confirmationData.resource.lname}
                        </h5>
                        <span className="text-2xl md:text-3xl font-medium text-black">
                          ={" "}
                          {confirmationData.bookings.find(
                            (b) => b.type === "resource"
                          )?.points || 0}{" "}
                          pts
                        </span>
                      </div>
                      {(() => {
                        const resourceBooking = confirmationData.bookings.find(
                          (b) => b.type === "resource"
                        );
                        if (resourceBooking?.data) {
                          try {
                            const dailyData = JSON.parse(resourceBooking.data);
                            if (
                              dailyData.daily &&
                              Array.isArray(dailyData.daily)
                            ) {
                              return dailyData.daily.map(
                                (points: number, index: number) => (
                                  <p
                                    key={index}
                                    className="text-sm text-gray-600"
                                  >
                                    {points}pts x 1day
                                  </p>
                                )
                              );
                            }
                          } catch (e) {
                            console.error("Error parsing resource data:", e);
                          }
                        }
                        return (
                          <p className="text-sm text-gray-600 mb-2">
                            {confirmationData.bookings.find(
                              (b) => b.type === "resource"
                            )?.points || 0}
                            pts x 1day
                          </p>
                        );
                      })()}
                      <p className="text-sm text-blue-600 mb-2">
                        {confirmationData.resource.resource_name?.name}
                      </p>
                      <p className="text-sm text-gray-600">
                        {confirmationData.resource.description}
                      </p>
                    </div>
                  </div>
                </div>
              )}

              {/* Booking Cost Summary */}
              <div className="border-t-2 border-gray-200 pt-8 mt-12">
                <div className="flex justify-between items-center mb-4">
                  <span className="text-2xl md:text-3xl font-medium text-black">
                    Booking cost
                  </span>
                  <span className="text-2xl md:text-3xl font-medium text-black">
                    {confirmationData.booking.total_points} pts
                  </span>
                </div>
                <div className="flex justify-between items-center mb-4">
                  <span className="text-xl md:text-2xl text-gray-600">
                    Points left
                  </span>
                  <span className="text-xl md:text-2xl text-gray-600">
                    {userPoints - (confirmationData.booking.total_points || 0)}{" "}
                    pts
                  </span>
                </div>
                <div className="text-center">
                  <p className="text-sm text-gray-600">
                    Not marked as complete
                  </p>
                </div>
              </div>

              {/* Action Buttons */}
              <div className="border-t-2 border-gray-200 pt-8 mt-8">
                <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
                  <Button
                    onClick={handleCancelBooking}
                    variant="outline-black"
                    size="lg"
                    disabled={
                      cancelLoading || confirmationData.booking.status !== 0
                    }
                    className="w-full sm:w-auto min-w-[200px] uppercase tracking-wider font-medium border-2 border-black"
                  >
                    {cancelLoading ? "Cancelling..." : "Cancel Booking"}
                  </Button>
                  <Button
                    onClick={handleCompleteBooking}
                    variant="fill-black"
                    size="lg"
                    disabled={
                      completeLoading || confirmationData.booking.status !== 0
                    }
                    className="w-full sm:w-auto min-w-[200px] uppercase tracking-wider font-medium"
                  >
                    {completeLoading ? "Completing..." : "Complete Booking"}
                  </Button>
                </div>
                {confirmationData.booking.status !== 0 && (
                  <p className="text-center text-sm text-gray-500 mt-4">
                    {confirmationData.booking.status === 1
                      ? "Booking is already completed"
                      : confirmationData.booking.status === 2
                      ? "Booking is cancelled"
                      : "Booking status updated"}
                  </p>
                )}
              </div>
            </div>

            {/* Right Column - Empty for now as per reference image */}
            <div className="lg:col-span-1">
              {/* This column is intentionally empty to match the reference image layout */}
            </div>
          </div>
        </main>

        {/* Checkout Data Modal */}
        {showCheckoutModal && checkoutData && (
          <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
            <div className="bg-white rounded-2xl p-8 shadow-2xl max-w-2xl w-full mx-4 max-h-[90vh] overflow-y-auto">
              <div className="flex justify-between items-center mb-6">
                <h3 className="text-2xl font-medium text-black">
                  Equipment Checkout Details
                </h3>
                <button
                  onClick={() => setShowCheckoutModal(false)}
                  className="text-gray-400 hover:text-black transition-colors"
                >
                  <svg
                    className="w-6 h-6"
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M6 18L18 6M6 6l12 12"
                    />
                  </svg>
                </button>
              </div>

              <div className="space-y-6">
                {/* Checkout Date */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Checkout Date
                  </label>
                  <p className="text-lg text-black">
                    {new Date(checkoutData.checkout_date).toLocaleString(
                      "en-US",
                      {
                        year: "numeric",
                        month: "long",
                        day: "numeric",
                        hour: "2-digit",
                        minute: "2-digit",
                      }
                    )}
                  </p>
                </div>

                {/* Description */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Description
                  </label>
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <p className="text-gray-800">
                      {checkoutData.description || "No description provided"}
                    </p>
                  </div>
                </div>

                {/* Photos */}
                {checkoutData.photos && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Checkout Photos
                    </label>
                    <div className="bg-gray-50 p-4 rounded-lg">
                      <p className="text-gray-600 text-sm">Photos available</p>
                    </div>
                  </div>
                )}

                {/* Check-in Photos */}
                {checkoutData.checkin_photos && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Check-in Photos
                    </label>
                    <div className="bg-gray-50 p-4 rounded-lg">
                      <p className="text-gray-600 text-sm">
                        Check-in photos available
                      </p>
                    </div>
                  </div>
                )}

                {/* Check-in Comments */}
                {checkoutData.checkin_comments && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Check-in Comments
                    </label>
                    <div className="bg-gray-50 p-4 rounded-lg">
                      <p className="text-gray-800">
                        {checkoutData.checkin_comments}
                      </p>
                    </div>
                  </div>
                )}

                {/* Status */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Status
                  </label>
                  <span
                    className={`inline-block px-3 py-1 rounded-full text-sm font-medium ${
                      checkoutData.status === 0
                        ? "bg-yellow-100 text-yellow-800"
                        : checkoutData.status === 1
                        ? "bg-green-100 text-green-800"
                        : "bg-gray-100 text-gray-800"
                    }`}
                  >
                    {checkoutData.status === 0
                      ? "Pending"
                      : checkoutData.status === 1
                      ? "Completed"
                      : "Unknown"}
                  </span>
                </div>

                {/* Created At */}
                {checkoutData.created_at && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Created At
                    </label>
                    <p className="text-gray-600">
                      {new Date(checkoutData.created_at).toLocaleString(
                        "en-US",
                        {
                          year: "numeric",
                          month: "long",
                          day: "numeric",
                          hour: "2-digit",
                          minute: "2-digit",
                        }
                      )}
                    </p>
                  </div>
                )}
              </div>

              <div className="mt-8 flex justify-end">
                <Button
                  onClick={() => setShowCheckoutModal(false)}
                  variant="fill-black"
                  size="md"
                  className="uppercase tracking-wider"
                >
                  Close
                </Button>
              </div>
            </div>
          </div>
        )}
      </div>
    </ProtectedRoute>
  );
}
