import type { MetaFunction } from "@remix-run/node";
import React, { useState, useEffect } from "react";
import { useNavigate, useLocation } from "@remix-run/react";
import { ProtectedRoute } from "~/components/RouteProtection";
import { AppHeader } from "~/components/AppHeader";
import { Button } from "~/components/ui/button";
import { Trash2 } from "lucide-react";
import { useLanguage } from "~/hooks/useLanguage";
import { API_BASE_URL } from "~/lib/api";

// Stripe types
declare global {
  interface Window {
    Stripe: any;
  }
}

const STRIPE_KEY = "pk_test_51NpyhWBfA6yrUxU0IRBFtBsOIk1GXsmQlD1Z8RouELcyISQFBOSfCG5cDohyKBIZPkrHWbSFYcOWdDU2cc9fcjih00CpUxQIAN";

export const meta: MetaFunction = () => {
  return [
    { title: "Complete Plan Change - Nomadics Studios" },
    {
      name: "description",
      content: "Complete your plan change by adding a payment method.",
    },
  ];
};

interface PaymentMethod {
  id: string;
  brand: string;
  last4: string;
  exp_month: number;
  exp_year: number;
}

export default function ChangePlanPayment() {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();

  const [paymentMethods, setPaymentMethods] = useState<PaymentMethod[]>([]);
  const [selectedPaymentMethod, setSelectedPaymentMethod] = useState<string | null>(null);
  const [isAddingNew, setIsAddingNew] = useState(false);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [successMessage, setSuccessMessage] = useState<string>("");
  const [cardholderName, setCardholderName] = useState("");
  const [isSavingCard, setIsSavingCard] = useState(false);
  
  const [stripe, setStripe] = useState<any>(null);
  const [elements, setElements] = useState<any>(null);
  const [cardElement, setCardElement] = useState<any>(null);

  const selectedPlan = location.state?.selectedPlan;

  // Load Stripe and initialize elements
  useEffect(() => {
    let mounted = true;
    
    const loadStripe = async () => {
      try {
        // Load Stripe script if not already loaded
        if (!window.Stripe) {
          const script = document.createElement('script');
          script.src = 'https://js.stripe.com/v3/';
          script.async = true;
          document.head.appendChild(script);
          
          await new Promise<void>((resolve, reject) => {
            script.onload = () => resolve();
            script.onerror = () => reject(new Error('Failed to load Stripe'));
          });
        }

        if (!mounted) return;

        const stripeInstance = window.Stripe(STRIPE_KEY);
        const elementsInstance = stripeInstance.elements();
        
        const cardElementInstance = elementsInstance.create('card', {
          style: {
            base: {
              fontSize: '16px',
              color: '#ffffff',
              fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
              fontWeight: '400',
              '::placeholder': {
                color: 'rgba(255, 255, 255, 0.6)',
              },
              iconColor: '#ffffff',
            },
            invalid: {
              color: '#fa755a',
              iconColor: '#fa755a'
            },
            complete: {
              color: '#ffffff',
            }
          },
          hidePostalCode: true
        });

        if (!mounted) return;

        setStripe(stripeInstance);
        setElements(elementsInstance);
        setCardElement(cardElementInstance);

        // Mount card element after DOM is ready
        setTimeout(() => {
          if (!mounted) return;
          
          const cardContainer = document.getElementById('card-element');
          if (cardContainer) {
            console.log('Mounting Stripe Card Element to:', cardContainer);
            cardElementInstance.mount('#card-element');
            
            // Add event listeners for better UX
            cardElementInstance.on('ready', () => {
              console.log('Stripe Card Element is ready and mounted');
            });
            
            cardElementInstance.on('change', (event: any) => {
              console.log('Card element change event:', event);
              if (event.error) {
                setError(event.error.message);
              } else {
                setError(null);
              }
            });
            
            cardElementInstance.on('focus', () => {
              console.log('Card element focused');
            });

            cardElementInstance.on('blur', () => {
              console.log('Card element blurred');
            });
          } else {
            console.error('Card element container not found');
          }
        }, 1500);
      } catch (error) {
        console.error('Error loading Stripe:', error);
        setError('Failed to load payment system. Please refresh the page.');
      }
    };

    fetchPaymentMethods();
    
    // Only load Stripe when adding new card
    if (isAddingNew) {
      loadStripe();
    }

    return () => {
      mounted = false;
      if (cardElement) {
        cardElement.unmount();
      }
    };
  }, [isAddingNew]);

  const fetchPaymentMethods = async () => {
    try {
      const token = localStorage.getItem("authToken") || 
                    localStorage.getItem("auth_token") || 
                    localStorage.getItem("token");

      if (!token) {
        navigate("/login");
        return;
      }

      const response = await fetch(`${API_BASE_URL}/payment-methods`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Accept': 'application/json',
        },
      });

      if (response.ok) {
        const data = await response.json();
        console.log('Payment methods response:', data);
        
        // Handle both possible response structures
        let methods = [];
        if (data.status && data.data?.payment_methods) {
          methods = data.data.payment_methods;
        } else if (data.status && data.payment_methods) {
          methods = data.payment_methods;
        } else if (Array.isArray(data.payment_methods)) {
          methods = data.payment_methods;
        }
        
        // Map to consistent structure with card property
        const formattedMethods = methods.map((pm: any) => ({
          id: pm.id,
          brand: pm.card?.brand || pm.brand || 'card',
          last4: pm.card?.last4 || pm.last4 || '0000',
          exp_month: pm.card?.exp_month || pm.exp_month || 12,
          exp_year: pm.card?.exp_year || pm.exp_year || 2099,
        }));
        
        setPaymentMethods(formattedMethods);
        
        // Auto-select first payment method if exists and not adding new
        if (formattedMethods.length > 0 && !isAddingNew) {
          setSelectedPaymentMethod(formattedMethods[0].id);
        }
      }
    } catch (error) {
      console.error("Error fetching payment methods:", error);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!selectedPlan) {
      setError("No plan selected. Please go back and select a plan.");
      return;
    }

    setLoading(true);
    setError(null);

    try {
      const token = localStorage.getItem("authToken") || 
                    localStorage.getItem("auth_token") || 
                    localStorage.getItem("token");

      if (!token) {
        navigate("/login");
        return;
      }

      let pmToken = selectedPaymentMethod;
      let pmType = "existing";

      // If adding a new card, create a payment method token
      if (isAddingNew && stripe && cardElement) {
        const { error: stripeError, paymentMethod } = await stripe.createPaymentMethod({
          type: "card",
          card: cardElement,
        });

        if (stripeError) {
          throw new Error(stripeError.message);
        }

        pmToken = paymentMethod.id;
        pmType = "new";
      }

      if (!pmToken) {
        throw new Error("Please select or add a payment method");
      }

      // Call change-plan API
      const formData = new FormData();
      formData.append("plan", selectedPlan.slug);
      formData.append("pm_token", pmToken);
      formData.append("pm_type", pmType);

      const response = await fetch(`${API_BASE_URL}/change-plan`, {
        method: "POST",
        headers: {
          Authorization: `Bearer ${token}`,
        },
        body: formData,
      });

      const data = await response.json();

      if (response.ok && data.status) {
        setSuccessMessage(data.message || "Your plan has been successfully updated.");
        setShowSuccessModal(true);
        // Update localStorage with new plan
        const userDataKeys = ['userData', 'userFullResponse'];
        for (const key of userDataKeys) {
          const stored = localStorage.getItem(key);
          if (stored) {
            const parsed = JSON.parse(stored);
            if (key === 'userFullResponse') {
              parsed.user.plan = selectedPlan;
            } else {
              parsed.plan = selectedPlan;
            }
            localStorage.setItem(key, JSON.stringify(parsed));
          }
        }
      } else {
        throw new Error(data.message || "Failed to change plan");
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : "Failed to change plan");
    } finally {
      setLoading(false);
    }
  };

  const handleSuccessModalClose = () => {
    setShowSuccessModal(false);
    navigate("/account-settings");
  };

  const handleAddCard = async () => {
    if (!stripe || !cardElement) {
      setError("Stripe is not loaded properly. Please refresh the page.");
      return;
    }

    if (!cardholderName.trim()) {
      setError("Name on card is required.");
      return;
    }

    setIsSavingCard(true);
    setError(null);

    try {
      const token = localStorage.getItem("authToken") || 
                    localStorage.getItem("auth_token") || 
                    localStorage.getItem("token");

      if (!token) {
        navigate("/login");
        return;
      }

      // Create payment method with Stripe
      const { error: stripeError, paymentMethod } = await stripe.createPaymentMethod({
        type: "card",
        card: cardElement,
        billing_details: {
          name: cardholderName
        }
      });

      if (stripeError) {
        throw new Error(stripeError.message);
      }

      // Call add-payment-method API
      const formData = new FormData();
      formData.append("name", cardholderName);
      formData.append("token", paymentMethod.id);
      formData.append("type", "change-plan");

      const response = await fetch(`${API_BASE_URL}/add-payment-method`, {
        method: "POST",
        headers: {
          Authorization: `Bearer ${token}`,
        },
        body: formData,
      });

      const data = await response.json();

      if (response.ok && data.status) {
        // Clear form
        setCardholderName("");
        if (cardElement) {
          cardElement.clear();
        }
        
        // Refresh payment methods
        await fetchPaymentMethods();
        
        // Close the add card form
        setIsAddingNew(false);
        
        // Show success message
        setError(null);
        alert(data.message || "Payment method added successfully!");
      } else {
        throw new Error(data.message || "Failed to add payment method");
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : "Failed to add payment method");
    } finally {
      setIsSavingCard(false);
    }
  };

  const handleDeletePaymentMethod = async (pmid: string) => {
    if (!confirm('Are you sure you want to delete this payment method?')) {
      return;
    }

    try {
      const token = localStorage.getItem("authToken") || 
                    localStorage.getItem("auth_token") || 
                    localStorage.getItem("token");

      if (!token) {
        alert('Authentication token not found');
        return;
      }

      const formData = new FormData();
      formData.append('pmid', pmid);

      const response = await fetch(`${API_BASE_URL}/payment-method-delete`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      const result = await response.json();

      if (response.ok && result.status) {
        // Refresh payment methods list
        await fetchPaymentMethods();
        alert('Payment method deleted successfully');
      } else {
        alert(result.message || 'Failed to delete payment method. Please try again.');
      }
    } catch (error) {
      console.error('Delete payment method error:', error);
      alert('An error occurred while deleting the payment method.');
    }
  };

  if (!selectedPlan) {
    return (
      <ProtectedRoute>
        <div className="min-h-screen bg-black flex items-center justify-center px-4">
          <div className="text-white text-center">
            <h2 className="text-xl sm:text-2xl mb-4 sm:mb-6 font-semibold">No plan selected</h2>
            <button
              onClick={() => navigate("/change-plan")}
              className="px-6 sm:px-8 py-3 sm:py-4 bg-white text-black uppercase tracking-wider font-semibold hover:bg-white/90 transition-all rounded-lg text-sm sm:text-base"
            >
              Go Back to Plans
            </button>
          </div>
        </div>
      </ProtectedRoute>
    );
  }

  return (
    <ProtectedRoute>
      <div
        className="min-h-screen bg-black"
        style={{ fontFamily: "Clash Grotesk Variable, sans-serif" }}
      >
        <AppHeader theme="dark" />

        <main className="pt-20 pb-10">
          <style>{`
            /* Custom styles for Stripe Elements */
            .StripeElement {
              height: 40px;
              padding: 10px 12px;
              background-color: transparent !important;
              color: white !important;
              border: none;
            }
            
            .StripeElement--focus {
              box-shadow: none;
            }
            
            .StripeElement--invalid {
              border-color: #fa755a;
            }
            
            .StripeElement--webkit-autofill {
              background-color: transparent !important;
            }
            
            #card-element .StripeElement {
              width: 100%;
              height: auto;
              padding: 12px 0;
            }
          `}</style>
          <div className="min-h-screen bg-black py-8 sm:py-12 md:py-16 px-4">
            <div className="max-w-3xl mx-auto">
              {/* Header */}
              <div className="text-center mb-8 sm:mb-12">
                <h1 className="text-white text-3xl sm:text-4xl md:text-5xl font-bold uppercase tracking-tight mb-4">
                  Complete Plan Change
                </h1>
                <p className="text-white/60 text-base sm:text-lg">
                  Selected Plan: <span className="text-white font-semibold">{selectedPlan.name}</span>
                </p>
                <p className="text-white/60 text-sm sm:text-base">
                  ${selectedPlan.price}/{selectedPlan.frequency}
                </p>
              </div>

              {/* Payment Methods Container */}
              <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                <h3 className="text-white text-sm font-medium mb-8 uppercase tracking-wider">
                  Payment Methods
                </h3>

                {error && (
                  <div className="mb-6 p-3 sm:p-4 bg-red-500/10 border border-red-500/30 rounded-lg text-red-400 text-sm">
                    {error}
                  </div>
                )}

                <form onSubmit={handleSubmit}>
                  {/* Add Payment Method Button - Show when not adding new */}
                  {!isAddingNew && (
                    <Button
                      type="button"
                      variant="fill-white"
                      size="md"
                      className="w-full mb-6"
                      onClick={() => setIsAddingNew(true)}
                    >
                      + Add Payment Method
                    </Button>
                  )}

                  {/* Add New Payment Method Form */}
                  {isAddingNew && (
                    <div className="mb-6 bg-white/5 border border-white/10 rounded-xl p-6">
                      <div className="mb-6">
                        <label className="block text-white text-sm mb-2 uppercase tracking-wider">
                          Name on the card
                        </label>
                        <input
                          type="text"
                          placeholder="Cardholder name"
                          value={cardholderName}
                          onChange={(e) => setCardholderName(e.target.value)}
                          className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-lg text-white placeholder:text-white/60 text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                        />
                      </div>

                      <div className="mb-6">
                        <label className="block text-white/60 text-sm mb-2">
                          Card Information
                        </label>
                        <div 
                          id="card-element"
                          className="w-full px-4 py-4 bg-white/10 border border-white/20 focus-within:ring-2 focus-within:ring-white/40 focus-within:border-white/40 backdrop-blur-sm transition-all"
                          style={{
                            minHeight: '56px',
                          }}
                        />
                        <p className="text-white/40 text-xs mt-2">
                          Enter your card number, expiry date, and CVC. All fields are in one input.
                        </p>
                      </div>

                      <Button
                        type="button"
                        variant="fill-white"
                        size="md"
                        className="w-full mb-4"
                        arrowRight={!isSavingCard}
                        onClick={handleAddCard}
                        disabled={isSavingCard}
                      >
                        {isSavingCard ? "Adding Card..." : "Add Your Card"}
                      </Button>

                      <button
                        type="button"
                        onClick={() => setIsAddingNew(false)}
                        className="text-white/60 hover:text-white text-sm transition-colors"
                      >
                        ← Cancel
                      </button>
                    </div>
                  )}

                  {/* Existing Payment Methods - Show when not adding new */}
                  {paymentMethods.length > 0 && !isAddingNew && (
                    <div className="border-t border-white/20 pt-6">
                      <div className="space-y-4">
                        {paymentMethods.map((pm) => (
                          <label
                            key={pm.id}
                            htmlFor={`payment-${pm.id}`}
                            className={`bg-white/5 rounded-xl p-6 cursor-pointer transition-colors border block ${
                              selectedPaymentMethod === pm.id
                                ? "border-white/40 bg-white/10"
                                : "border-white/10 hover:bg-white/10"
                            }`}
                          >
                            <div className="flex items-center justify-between">
                              <div className="flex items-center gap-4">
                                {/* Radio Button */}
                                <input
                                  type="radio"
                                  id={`payment-${pm.id}`}
                                  name="paymentMethod"
                                  value={pm.id}
                                  checked={selectedPaymentMethod === pm.id}
                                  onChange={() => setSelectedPaymentMethod(pm.id)}
                                  className="w-5 h-5 text-white bg-white/10 border-white/20 focus:ring-white/40 focus:ring-2"
                                />
                                
                                {/* Card Icon */}
                                <div className="w-12 h-12 bg-white/20 rounded-lg flex items-center justify-center">
                                  <span className="text-2xl">💳</span>
                                </div>
                                
                                <div>
                                  <div className="flex items-center gap-2 mb-1">
                                    <p className="text-white font-medium uppercase">
                                      {pm.brand} •••• {pm.last4}
                                    </p>
                                    {selectedPaymentMethod === pm.id && (
                                      <span className="bg-green-500/20 text-green-400 px-2 py-0.5 rounded text-xs font-medium uppercase">
                                        Selected
                                      </span>
                                    )}
                                  </div>
                                  <p className="text-white/60 text-sm">
                                    sudhir
                                  </p>
                                  <p className="text-white/40 text-xs mt-1">
                                    Expires {String(pm.exp_month).padStart(2, '0')}/{pm.exp_year}
                                  </p>
                                </div>
                              </div>

                              {/* Delete Icon */}
                              <button
                                type="button"
                                onClick={(e) => {
                                  e.stopPropagation();
                                  e.preventDefault();
                                  handleDeletePaymentMethod(pm.id);
                                }}
                                className="text-white/40 hover:text-red-400 transition-colors p-2"
                                title="Delete payment method"
                              >
                                <Trash2 className="w-5 h-5" />
                              </button>
                            </div>
                          </label>
                        ))}
                      </div>
                    </div>
                  )}

                  {/* Action Buttons */}
                  {!isAddingNew && (
                    <div className="flex flex-col sm:flex-row gap-4 mt-6">
                      <Button
                        type="button"
                        variant="fill-black"
                        size="lg"
                        className="flex-1"
                        onClick={() => navigate("/change-plan")}
                        disabled={loading}
                      >
                        Cancel
                      </Button>
                      <Button
                        type="submit"
                        variant="fill-white"
                        size="lg"
                        className="flex-1"
                        arrowRight={true}
                        disabled={loading || (!selectedPaymentMethod && !isAddingNew)}
                      >
                        {loading ? "Processing..." : "Update Plan"}
                      </Button>
                    </div>
                  )}
                </form>
              </div>
            </div>
          </div>
        </main>

        {/* Success Modal */}
        {showSuccessModal && (
          <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4">
            <div className="bg-white/10 backdrop-blur-lg border border-white/20 rounded-2xl p-8 max-w-md w-full shadow-2xl">
              <div className="text-center">
                <div className="w-16 h-16 bg-green-500/20 rounded-full flex items-center justify-center mx-auto mb-6">
                  <svg
                    className="w-8 h-8 text-green-400"
                    fill="none"
                    viewBox="0 0 24 24"
                    stroke="currentColor"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M5 13l4 4L19 7"
                    />
                  </svg>
                </div>
                <h3 className="text-white text-2xl font-bold mb-3 uppercase tracking-wide">
                  Plan Updated!
                </h3>
                <p className="text-white/60 mb-8 text-sm">
                  {successMessage}
                </p>
                <Button
                  type="button"
                  variant="fill-white"
                  size="lg"
                  className="w-full"
                  onClick={handleSuccessModalClose}
                >
                  Continue
                </Button>
              </div>
            </div>
          </div>
        )}
      </div>
    </ProtectedRoute>
  );
}
