import type { MetaFunction } from "@remix-run/node";
import React, { useState, useEffect } from "react";
import { useNavigate } from "@remix-run/react";
import { ProtectedRoute } from "~/components/RouteProtection";
import { AppHeader } from "~/components/AppHeader";
import { Button } from "~/components/ui/button";
import { useLanguage } from "~/hooks/useLanguage";
import { plansAPI, type Plan, type Service } from "~/lib/api";

export const meta: MetaFunction = () => {
  return [
    { title: "Change Plan - Nomadics Studios" },
    {
      name: "description",
      content: "Change your membership plan.",
    },
  ];
};

const ChangePlan: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const [isVisible, setIsVisible] = useState(false);
  const [selectedTier, setSelectedTier] = useState<number | null>(null);
  const [selectedAddOns, setSelectedAddOns] = useState<number[]>([]);
  const [plans, setPlans] = useState<Plan[]>([]);
  const [service, setService] = useState<Service | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [currentPlanId, setCurrentPlanId] = useState<number | null>(null);

  useEffect(() => {
    setIsVisible(true);

    // Load current plan from localStorage
    let foundPlanId: number | null = null;
    if (typeof window !== "undefined") {
      try {
        const possibleKeys = ["userData", "userFullResponse"];
        for (const key of possibleKeys) {
          const stored = localStorage.getItem(key);
          if (stored) {
            const parsed = JSON.parse(stored);
            const userData = key === "userFullResponse" ? parsed.user : parsed;
            if (userData?.plan?.id) {
              foundPlanId = userData.plan.id;
              setCurrentPlanId(foundPlanId);
              console.log(
                "Current plan ID:",
                foundPlanId,
                "Name:",
                userData.plan.name
              );
              break;
            }
          }
        }
      } catch (error) {
        console.error("Error loading current plan:", error);
      }
    }

    const fetchPlans = async () => {
      try {
        setLoading(true);
        const response = await plansAPI.getAll();
        if (response.status && response.data) {
          setPlans(response.data.plans);
          setService(response.data.services);

          // If no plan found for user, default to Guest plan (index 0)
          if (!foundPlanId && response.data.plans.length > 0) {
            const guestPlanId = response.data.plans[0].id;
            setCurrentPlanId(guestPlanId);
            console.log(
              "No plan found for user, defaulting to Guest plan:",
              guestPlanId
            );
          }
        } else {
          setError("Failed to fetch plans");
        }
      } catch (err) {
        setError(err instanceof Error ? err.message : "Failed to fetch plans");
      } finally {
        setLoading(false);
      }
    };

    fetchPlans();
  }, []);

  const generateFeatures = (plan: Plan): string[] => {
    const features: string[] = [];
    if (plan.advance_duration > 0) {
      features.push(
        `Book up ${plan.advance_duration} ${plan.advance_frequency}${
          plan.advance_duration > 1 ? "s" : ""
        } in advance`
      );
    } else {
      features.push("Just one day/one by member");
    }
    if (plan.num_of_guest > 0) {
      features.push(
        `${
          plan.num_of_guest === 1 ? "One guest" : `${plan.num_of_guest} guests`
        }`
      );
    } else {
      features.push("Zero guests");
    }
    return features;
  };

  const handleAddOnToggle = (serviceId: number) => {
    setSelectedAddOns((prev) =>
      prev.includes(serviceId)
        ? prev.filter((id) => id !== serviceId)
        : [...prev, serviceId]
    );
  };

  const handleTierSelect = (index: number) => {
    setSelectedTier(index);
  };

  const handlePlanChange = () => {
    // Navigate to payment page with selected plan data
    if (selectedTier !== null) {
      navigate("/change-plan-payment", {
        state: {
          selectedPlan: plans[selectedTier],
        },
      });
    }
  };

  return (
    <ProtectedRoute>
      <div
        className="min-h-screen bg-[#121212]"
        style={{ fontFamily: "Clash Grotesk Variable, sans-serif" }}
      >
        <AppHeader theme="dark" />

        <main className="pt-20">
          <div className="min-h-screen bg-[#121212] relative overflow-hidden">
            {/* Main Content */}
            <div className="relative z-10 min-h-screen">
              {/* Hero Section */}
              <div className="w-full max-w-[1440px] mx-auto pt-16 sm:pt-24 md:pt-[100px] pb-12 sm:pb-16 md:pb-[60px] lg:pb-[80px] px-4 sm:px-6 md:px-[24px] lg:px-[56px]">
                <div className="text-center mb-6 sm:mb-8">
                  <h1 className="text-white text-[32px] sm:text-[40px] md:text-[48px] lg:text-[64px] xl:text-[80px] font-bold uppercase tracking-tight leading-[1.1] mb-3 sm:mb-4 md:mb-6">
                    CHANGE YOUR PLAN
                  </h1>
                  <p className="text-white/80 text-xs sm:text-sm md:text-base font-normal max-w-2xl mx-auto leading-relaxed mb-2 px-2 sm:px-4">
                    Select a new membership plan that fits your needs
                  </p>
                </div>

                {/* Back Button */}
                <div className="text-center mt-4 sm:mt-6 md:mt-8">
                  <button
                    onClick={() => navigate("/account-settings")}
                    className="px-4 sm:px-6 md:px-8 py-2 sm:py-2.5 md:py-3 border border-white text-white text-[10px] sm:text-xs md:text-sm uppercase tracking-wider font-medium hover:bg-white hover:text-black transition-all"
                  >
                    ← BACK TO ACCOUNT SETTINGS
                  </button>
                </div>
              </div>

              {/* Loading State */}
              {loading && (
                <div className="text-center py-16">
                  <div className="text-white/80">Loading plans...</div>
                </div>
              )}

              {/* Error State */}
              {error && (
                <div className="text-center py-16">
                  <div className="text-red-400">Error: {error}</div>
                </div>
              )}

              {/* Content - only show when not loading and no error */}
              {!loading && !error && (
                <>
                  {/* Plans Section - Row with 4 cards - Responsive */}
                  <div className="w-full max-w-[1440px] mx-auto border-t border-white/10 px-4 sm:px-6 md:px-[56px]">
                    <div className="flex flex-col md:flex-row md:flex-wrap gap-4 p-0">
                      {plans.map((plan, index) => {
                        const isPopular = plan.slug === "up-and-coming";
                        const isCurrentPlan = plan.id === currentPlanId;
                        const isSelected = selectedTier === index;
                        const shouldShowPopularStyle = isSelected;
                        return (
                          <div
                            key={plan.id}
                            className={`relative flex-1 flex flex-col w-full md:w-[300px] md:min-h-[510px] py-8 px-6 justify-between overflow-hidden ${
                              shouldShowPopularStyle
                                ? "bg-[#121212]"
                                : "bg-white"
                            } border-2 ${
                              shouldShowPopularStyle
                                ? "border-[#D64933]"
                                : "border-[#121212]"
                            } ${
                              isCurrentPlan ? "" : "cursor-pointer"
                            } transition-all`}
                            onClick={() =>
                              !isCurrentPlan && handleTierSelect(index)
                            }
                          >
                            {/* Current Plan Badge - Always show for current plan */}
                            {isCurrentPlan && (
                              <div className="absolute -top-0 left-0 right-0">
                                <div className="bg-[#52525B] text-white text-[11px] px-3 py-2 text-center uppercase tracking-wider font-semibold">
                                  CURRENT PLAN
                                </div>
                              </div>
                            )}

                            {/* Selected Badge - Only show for non-current plans when selected */}
                            {shouldShowPopularStyle && !isCurrentPlan && (
                              <div className="absolute -top-0 left-0 right-0">
                                <div className="bg-[#D64933] text-white text-[11px] px-3 py-2 text-center uppercase tracking-wider font-semibold">
                                  SELECTED
                                </div>
                              </div>
                            )}

                            {/* Card Content */}
                            <div
                              className={`flex flex-col justify-between flex-1 gap-8 ${
                                shouldShowPopularStyle || isCurrentPlan
                                  ? "mt-6"
                                  : ""
                              }`}
                            >
                              {/* Title */}
                              <h3
                                className={`${
                                  shouldShowPopularStyle
                                    ? "text-white"
                                    : "text-black"
                                } text-sm font-semibold uppercase tracking-wide`}
                              >
                                {plan.name}
                              </h3>

                              {/* Price */}
                              <div>
                                <div className="flex items-baseline gap-1">
                                  <span
                                    className={`text-[48px] font-bold ${
                                      shouldShowPopularStyle
                                        ? "text-white"
                                        : "text-black"
                                    } leading-none`}
                                  >
                                    ${plan.price}
                                  </span>
                                  <span
                                    className={`${
                                      shouldShowPopularStyle
                                        ? "text-white/60"
                                        : "text-black/60"
                                    } text-sm`}
                                  >
                                    /{plan.frequency}
                                  </span>
                                </div>
                              </div>

                              {/* Points */}
                              <div className="flex items-center gap-2">
                                <svg
                                  className="w-4 h-4"
                                  fill="none"
                                  viewBox="0 0 24 24"
                                  stroke="#D64933"
                                  strokeWidth={2.5}
                                >
                                  <path
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                    d="M5 13l4 4L19 7"
                                  />
                                </svg>
                                <span className="text-[#D64933] text-sm font-normal">
                                  Get {plan.points} Nomadic Points
                                </span>
                              </div>

                              {/* Features List */}
                              <ul
                                className={`flex-grow space-y-3 ${
                                  shouldShowPopularStyle
                                    ? "text-white"
                                    : "text-black"
                                } text-sm`}
                              >
                                {index === 0 ? (
                                  <>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>Attend as a guest of a member</span>
                                    </li>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>
                                        Book day-of studio time (subject to
                                        availability)
                                      </span>
                                    </li>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>Join workshops and events</span>
                                    </li>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>Upgrade anytime</span>
                                    </li>
                                    <li className="flex items-start gap-3 opacity-30">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>
                                        Cannot invite guests or refer members
                                      </span>
                                    </li>
                                  </>
                                ) : (
                                  <>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>
                                        Includes all Guest features and...
                                      </span>
                                    </li>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>
                                        {index === 1
                                          ? "Monthly creative balance for studio use and bookings"
                                          : index === 2
                                          ? "Expanded creative balance for frequent use"
                                          : "Premium balance for professionals and teams"}
                                      </span>
                                    </li>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>
                                        Book up to {plan.advance_duration}{" "}
                                        {plan.advance_frequency}
                                        {plan.advance_duration > 1
                                          ? "s"
                                          : ""}{" "}
                                        in advance
                                      </span>
                                    </li>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>
                                        {plan.num_of_guest} guest
                                        {plan.num_of_guest > 1 ? "s" : ""}{" "}
                                        included
                                      </span>
                                    </li>
                                    <li className="flex items-start gap-3">
                                      <div
                                        className={`w-4 h-4 mt-0.5 flex-shrink-0 border ${
                                          shouldShowPopularStyle
                                            ? "border-white"
                                            : "border-black"
                                        } flex items-center justify-center`}
                                      >
                                        <svg
                                          className={`w-3 h-3`}
                                          fill="currentColor"
                                          viewBox="0 0 24 24"
                                        >
                                          <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" />
                                        </svg>
                                      </div>
                                      <span>
                                        Usage valid for {plan.advance_duration}{" "}
                                        {plan.advance_frequency}
                                        {plan.advance_duration > 1 ? "s" : ""}
                                      </span>
                                    </li>
                                  </>
                                )}
                              </ul>

                              {/* Button */}
                              <button
                                onClick={(e) => {
                                  e.stopPropagation();
                                  if (!isCurrentPlan) {
                                    if (selectedTier === index) {
                                      // Navigate or trigger update
                                      handlePlanChange();
                                    } else {
                                      // Select the plan
                                      handleTierSelect(index);
                                    }
                                  }
                                }}
                                className={`w-full py-3 text-sm uppercase tracking-wider font-semibold border-2 transition-all cursor-pointer ${
                                  isCurrentPlan
                                    ? "bg-gray-400 text-white border-gray-400 cursor-not-allowed opacity-70"
                                    : shouldShowPopularStyle
                                    ? "bg-white text-black border-white hover:bg-white/90"
                                    : "bg-white text-black border-black hover:bg-black hover:text-white"
                                }`}
                                disabled={isCurrentPlan}
                              >
                                {isCurrentPlan
                                  ? "CURRENT PLAN"
                                  : selectedTier === index
                                  ? "UPDATE PLAN"
                                  : "SELECT"}
                              </button>
                            </div>
                          </div>
                        );
                      })}
                    </div>
                  </div>

                  {/* Services/Benefits Section */}
                  <div className="w-full bg-white py-[60px] sm:py-[120px] px-[24px] sm:px-[56px] mt-[-100px] sm:mt-[-230px]">
                    <div className="max-w-[1440px] mx-auto pt-[120px] sm:pt-[250px]">
                      {/* Header Section */}
                      <div className="flex flex-col lg:flex-row gap-[24px] mb-[60px] sm:mb-[120px]">
                        {/* Left - Title */}
                        <div className="lg:w-[538px]">
                          <p className="text-black/40 text-[10px] uppercase tracking-[0.2em] mb-4 sm:mb-6 font-medium">
                            [BENEFITS]
                          </p>
                          <h2 className="text-black text-[32px] sm:text-[44px] lg:text-[68px] font-medium uppercase leading-[1] lg:leading-[0.9] tracking-[-0.03em]">
                            MEMBERSHIP
                            <br />
                            THAT MOVES
                            <br />
                            WITH YOU
                          </h2>
                        </div>

                        {/* Right - Description */}
                        <div className="flex-1 flex items-center mt-6 lg:mt-0">
                          <div>
                            <p className="text-black text-[14px] sm:text-[16px] leading-[24px] sm:leading-[32px] mb-4 sm:mb-6 font-normal">
                              Nomadic Studios gives working creators flexible
                              access to{" "}
                              <span className="font-medium">
                                dedicated studio time, equipment, events,
                              </span>{" "}
                              and{" "}
                              <span className="font-medium">
                                technical teams
                              </span>{" "}
                              — available in every city within our network.
                            </p>
                            <p className="text-black text-[14px] sm:text-[16px] leading-[24px] sm:leading-[32px] font-normal">
                              Each plan includes a monthly creative balance{" "}
                              <span className="text-[#D64933] font-medium">
                                (Nomadic Points)
                              </span>{" "}
                              you can apply across disciplines and locations.
                            </p>
                          </div>
                        </div>
                      </div>

                      {/* Benefits Grid - 3 columns */}
                      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-[16px] sm:gap-[24px]">
                        {/* CREATE ANYWHERE */}
                        <div className="border border-black/30 p-6 sm:p-8">
                          <div className="w-8 h-8 sm:w-10 sm:h-10 mb-4 sm:mb-6">
                            <svg
                              width="100%"
                              height="100%"
                              viewBox="0 0 40 40"
                              fill="none"
                              xmlns="http://www.w3.org/2000/svg"
                            >
                              <path
                                d="M20 8V32M20 8L13 15M20 8L27 15"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                              <path
                                d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                            </svg>
                          </div>
                          <h3 className="text-black text-[14px] sm:text-[16px] font-medium uppercase tracking-wide mb-3 sm:mb-4 leading-tight">
                            CREATE ANYWHERE
                          </h3>
                          <p className="text-black/80 text-[13px] sm:text-[16px] leading-[20px] sm:leading-[24px] font-normal">
                            Book{" "}
                            <span className="font-medium">
                              dedicated studio time
                            </span>{" "}
                            for{" "}
                            <span className="font-medium">music, video</span>{" "}
                            and{" "}
                            <span className="font-medium">
                              streaming, fashion, art, photography,
                            </span>{" "}
                            or{" "}
                            <span className="font-medium">
                              immersive/maker projects
                            </span>{" "}
                            — all under one membership.
                          </p>
                        </div>

                        {/* FLEXIBLE ACCESS */}
                        <div className="border border-black/30 p-6 sm:p-8">
                          <div className="w-8 h-8 sm:w-10 sm:h-10 mb-4 sm:mb-6">
                            <svg
                              width="100%"
                              height="100%"
                              viewBox="0 0 40 40"
                              fill="none"
                              xmlns="http://www.w3.org/2000/svg"
                            >
                              <path
                                d="M20 8V32M20 8L13 15M20 8L27 15"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                              <path
                                d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                            </svg>
                          </div>
                          <h3 className="text-black text-[14px] sm:text-[16px] font-medium uppercase tracking-wide mb-3 sm:mb-4 leading-tight">
                            FLEXIBLE ACCESS
                          </h3>
                          <p className="text-black/80 text-[13px] sm:text-[16px] leading-[20px] sm:leading-[24px] font-normal">
                            Each session draws from your creative balance. The
                            cost of resources automatically adjusts based on
                            city and type.
                          </p>
                        </div>

                        {/* TIERED BENEFITS */}
                        <div className="border border-black/30 p-6 sm:p-8">
                          <div className="w-8 h-8 sm:w-10 sm:h-10 mb-4 sm:mb-6">
                            <svg
                              width="100%"
                              height="100%"
                              viewBox="0 0 40 40"
                              fill="none"
                              xmlns="http://www.w3.org/2000/svg"
                            >
                              <path
                                d="M20 8V32M20 8L13 15M20 8L27 15"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                              <path
                                d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                            </svg>
                          </div>
                          <h3 className="text-black text-[14px] sm:text-[16px] font-medium uppercase tracking-wide mb-3 sm:mb-4 leading-tight">
                            TIERED BENEFITS
                          </h3>
                          <p className="text-black/80 text-[13px] sm:text-[16px] leading-[20px] sm:leading-[24px] font-normal">
                            Higher tiers provide larger balances, extended
                            validity, earlier booking windows, and more included
                            guests.
                          </p>
                        </div>

                        {/* BRING GUESTS */}
                        <div className="border border-black/30 p-8">
                          <div className="w-10 h-10 mb-6">
                            <svg
                              width="40"
                              height="40"
                              viewBox="0 0 40 40"
                              fill="none"
                              xmlns="http://www.w3.org/2000/svg"
                            >
                              <path
                                d="M20 8V32M20 8L13 15M20 8L27 15"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                              <path
                                d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                            </svg>
                          </div>
                          <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                            BRING GUESTS
                          </h3>
                          <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                            Invite collaborators, clients, or assistants to your
                            booked sessions. Additional guests can join for a
                            small usage fee.
                          </p>
                        </div>

                        {/* WORKSHOPS & EVENTS */}
                        <div className="border border-black/30 p-8">
                          <div className="w-10 h-10 mb-6">
                            <svg
                              width="40"
                              height="40"
                              viewBox="0 0 40 40"
                              fill="none"
                              xmlns="http://www.w3.org/2000/svg"
                            >
                              <path
                                d="M20 8V32M20 8L13 15M20 8L27 15"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                              <path
                                d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                            </svg>
                          </div>
                          <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                            WORKSHOPS & EVENTS
                          </h3>
                          <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                            Apply your balance toward hands-on workshops and
                            community events that connect creators and expand
                            skills.
                          </p>
                        </div>

                        {/* REFERRAL-ONLY MEMBERSHIP */}
                        <div className="border border-black/30 p-8">
                          <div className="w-10 h-10 mb-6">
                            <svg
                              width="40"
                              height="40"
                              viewBox="0 0 40 40"
                              fill="none"
                              xmlns="http://www.w3.org/2000/svg"
                            >
                              <path
                                d="M20 8V32M20 8L13 15M20 8L27 15"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                              <path
                                d="M8 24V30C8 31.1046 8.89543 32 10 32H30C31.1046 32 32 31.1046 32 30V24"
                                stroke="black"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                            </svg>
                          </div>
                          <h3 className="text-black text-[16px] font-medium uppercase tracking-wide mb-4 leading-tight">
                            REFERRAL-ONLY
                            <br />
                            MEMBERSHIP
                          </h3>
                          <p className="text-black/80 text-[16px] leading-[24px] font-normal">
                            New members join by referral to ensure that every
                            addition strengthens the professional community.
                          </p>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* How It Works Section */}
                  <div className="w-full bg-[#121212] py-[60px] sm:py-[120px] px-[24px] sm:px-[56px]">
                    <div className="max-w-[1440px] mx-auto">
                      <div className="grid grid-cols-1 lg:grid-cols-2 gap-[32px] sm:gap-[56px] relative">
                        {/* Left Column */}
                        <div>
                          <p className="text-white/60 text-[12px] sm:text-[16px] uppercase tracking-[0.02em] mb-4 sm:mb-6 font-normal leading-[100%]">
                            [HOW IT WORKS]
                          </p>
                          <h2 className="text-white text-[32px] sm:text-[44px] md:text-[68px] font-medium uppercase leading-[1.05] md:leading-[1] tracking-[-0.03em] mb-6 sm:mb-8">
                            ONE MEMBERSHIP.
                            <br />
                            GLOBAL ACCESS
                          </h2>
                          <div className="space-y-4 sm:space-y-6">
                            <p className="text-white text-[16px] sm:text-[20px] lg:text-[24px] leading-[24px] sm:leading-[28px] lg:leading-[32px] font-normal">
                              Your membership includes a flexible creative
                              balance{" "}
                              <span className="text-[#D64933] font-medium">
                                (Nomadic Points)
                              </span>{" "}
                              you can use for studio time, equipment, events,
                              and technical or support teams.
                            </p>
                            <p className="text-white text-[16px] sm:text-[20px] lg:text-[24px] leading-[24px] sm:leading-[28px] lg:leading-[32px] font-normal">
                              The cost of each resource adjusts automatically to
                              reflect local studio rates — so you can create
                              confidently wherever you go.
                            </p>
                          </div>
                        </div>

                        {/* Center - Vertical Line with Dots */}
                        <div className="hidden lg:block absolute left-1/2 top-0 bottom-0 -translate-x-1/2">
                          <div className="relative h-full w-[2px]">
                            {/* Vertical Line */}
                            <div className="absolute inset-0 bg-[#D64933]"></div>
                            {/* Dots */}
                            {[0, 1, 2, 3, 4].map((index) => (
                              <div
                                key={index}
                                className="absolute left-1/2 -translate-x-1/2"
                                style={{ top: `${20 + index * 18}%` }}
                              >
                                <svg
                                  width="12"
                                  height="12"
                                  viewBox="0 0 12 12"
                                  fill="none"
                                >
                                  <rect width="12" height="12" fill="#D64933" />
                                </svg>
                              </div>
                            ))}
                            {/* Arrow at bottom */}
                            <div className="absolute bottom-0 left-1/2 -translate-x-1/2">
                              <svg
                                width="12"
                                height="20"
                                viewBox="0 0 12 20"
                                fill="none"
                              >
                                <path
                                  d="M6 0L6 16M6 16L1 11M6 16L11 11"
                                  stroke="#D64933"
                                  strokeWidth="2"
                                />
                              </svg>
                            </div>
                          </div>
                        </div>

                        {/* Right Column - Text Items */}
                        <div className="flex flex-col justify-center mt-8 lg:mt-0">
                          <div className="space-y-0">
                            {[
                              "BOOK STUDIOS, EQUIPMENT, WORKSHOPS, AND SUPPORT TEAMS DIRECTLY THROUGH YOUR DASHBOARD.",
                              "RESOURCE COSTS ARE AUTOMATICALLY CALCULATED PER CITY.",
                              "ADD MORE BALANCE ANYTIME AT YOUR MEMBER RATE.",
                              "HIGHER TIERS PROVIDE LONGER VALIDITY AND EARLY BOOKING PRIVILEGES.",
                              "APPLY YOUR BALANCE TOWARD EVENTS, WORKSHOPS, AND GUEST ACCESS.",
                            ].map((text, index) => (
                              <div
                                key={index}
                                className="py-4 sm:py-6 border-t border-white/10 first:border-t-0"
                              >
                                <p className="text-white text-[18px] sm:text-[24px] lg:text-[32px] leading-[24px] sm:leading-[32px] lg:leading-[40px] uppercase font-normal tracking-[-0.01em]">
                                  {text}
                                </p>
                              </div>
                            ))}
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* CTA Section */}
                  <div className="w-full bg-white relative overflow-hidden py-[80px] sm:py-[134px] px-[24px] sm:px-[56px]">
                    {/* Concentric circles background pattern */}
                    <div className="absolute inset-0 pointer-events-none overflow-hidden">
                      {/* Top-left circles */}
                      <div className="absolute -top-[400px] -left-[400px] hidden sm:block">
                        {[...Array(12)].map((_, i) => (
                          <div
                            key={`tl-${i}`}
                            className="absolute rounded-full border"
                            style={{
                              width: `${300 + i * 120}px`,
                              height: `${300 + i * 120}px`,
                              borderColor: `rgba(18, 18, 18, ${
                                0.08 - i * 0.005
                              })`,
                              borderWidth: "1px",
                            }}
                          />
                        ))}
                      </div>
                      {/* Bottom-right circles */}
                      <div className="absolute -bottom-[400px] -right-[400px] hidden sm:block">
                        {[...Array(12)].map((_, i) => (
                          <div
                            key={`br-${i}`}
                            className="absolute rounded-full border"
                            style={{
                              width: `${300 + i * 120}px`,
                              height: `${300 + i * 120}px`,
                              borderColor: `rgba(18, 18, 18, ${
                                0.08 - i * 0.005
                              })`,
                              borderWidth: "1px",
                            }}
                          />
                        ))}
                      </div>
                    </div>

                    {/* Content */}
                    <div className="relative z-10 max-w-[1100px] mx-auto text-center">
                      <div className="mb-6 sm:mb-8">
                        <h2 className="text-[#121212] text-[28px] sm:text-[40px] md:text-[68px] font-semibold uppercase tracking-[-0.03em] leading-[1.05] md:leading-[1.2] mb-0">
                          A CURATED NETWORK FOR
                        </h2>
                        <h2 className="text-[#121212] text-[28px] sm:text-[40px] md:text-[68px] font-semibold uppercase tracking-[-0.03em] leading-[1.05] md:leading-[1.2]">
                          SERIOUS CREATIVES
                        </h2>
                      </div>

                      <p className="text-[#52525B] text-[16px] sm:text-[20px] lg:text-[24px] leading-[24px] sm:leading-[28px] lg:leading-[32px] font-normal max-w-[651px] mx-auto mb-8 sm:mb-12 px-4 sm:px-0">
                        Nomadic Studios blends the accessibility of co-working
                        with the depth of professional production environments.
                        Membership is intentionally limited in each city to
                        ensure consistent access to top-tier facilities and
                        support.
                      </p>

                      <button
                        onClick={() => {
                          window.scrollTo({ top: 0, behavior: "smooth" });
                        }}
                        className="inline-flex items-center gap-2 sm:gap-3 bg-[#121212] text-white px-6 sm:px-10 py-3 sm:py-5 text-[14px] sm:text-[18px] uppercase tracking-[0.02em] font-medium hover:bg-[#121212]/90 transition-all"
                      >
                        VIEW PLANS AGAIN
                        <svg
                          className="w-4 h-4 sm:w-5 sm:h-5"
                          fill="none"
                          viewBox="0 0 24 24"
                          stroke="currentColor"
                          strokeWidth={2}
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            d="M17 8l4 4m0 0l-4 4m4-4H3"
                          />
                        </svg>
                      </button>
                    </div>
                  </div>
                </>
              )}
            </div>
          </div>
        </main>
      </div>
    </ProtectedRoute>
  );
};

export default ChangePlan;
