import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useParams, Link, useNavigate } from "@remix-run/react";
import { useState, useEffect } from "react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { useLanguage } from "~/hooks/useLanguage";
import { API_BASE_URL } from "~/lib/api";

export const meta: MetaFunction = () => {
  return [
    { title: "Equipment Checkout - Nomadic Studios" },
    {
      name: "description",
      content: "Equipment checkout process for your booking.",
    },
  ];
};

interface Equipment {
  id: number;
  location_id: number;
  name: string;
  photo: string;
  description: string;
  serial_number: string;
  manual: string | null;
  manual_link: string | null;
}

interface CheckoutResponse {
  status: boolean;
  data: {
    equipment: Equipment;
    CheckedOutData: any;
  };
}

export default function EquipmentCheckout() {
  const { bookingId, equipmentId } = useParams();
  const navigate = useNavigate();
  const { t } = useLanguage();
  const [equipment, setEquipment] = useState<Equipment | null>(null);
  const [loading, setLoading] = useState(true);
  const [submitting, setSubmitting] = useState(false);
  const [checkoutForm, setCheckoutForm] = useState({
    description: "",
    files: [] as File[],
  });

  // Fetch equipment checkout data
  useEffect(() => {
    const fetchCheckoutData = async () => {
      if (!bookingId || !equipmentId) return;

      try {
        const token =
          localStorage.getItem("authToken") ||
          localStorage.getItem("auth_token") ||
          localStorage.getItem("token");
        if (!token) {
          navigate("/login");
          return;
        }

        const response = await fetch(
          `${API_BASE_URL}/equipment-checkout/${bookingId}/${equipmentId}`,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );

        if (response.ok) {
          const result: CheckoutResponse = await response.json();
          if (result.status && result.data) {
            setEquipment(result.data.equipment);

            // If CheckedOutData exists, redirect back to booking confirmed page
            if (result.data.CheckedOutData) {
              navigate(`/booking-confirmed/${bookingId}`);
              return;
            }
          }
        } else {
          console.error("Failed to fetch checkout data");
        }
      } catch (error) {
        console.error("Error fetching checkout data:", error);
      } finally {
        setLoading(false);
      }
    };

    fetchCheckoutData();
  }, [bookingId, equipmentId, navigate]);

  // Handle file selection
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(e.target.files || []);
    setCheckoutForm((prev) => ({
      ...prev,
      files: [...prev.files, ...files],
    }));
  };

  // Remove file from selection
  const removeFile = (index: number) => {
    setCheckoutForm((prev) => ({
      ...prev,
      files: prev.files.filter((_, i) => i !== index),
    }));
  };

  // Handle drag and drop
  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    const files = Array.from(e.dataTransfer.files);
    setCheckoutForm((prev) => ({
      ...prev,
      files: [...prev.files, ...files],
    }));
  };

  // Handle checkout form submission
  const handleSubmitCheckout = async () => {
    if (!bookingId || !equipmentId) {
      alert("Missing booking or equipment information.");
      return;
    }

    if (!checkoutForm.description.trim()) {
      alert("Please enter a description.");
      return;
    }

    try {
      setSubmitting(true);
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");
      if (!token) {
        alert("Authentication required.");
        return;
      }

      const formData = new FormData();
      formData.append("booking_id", bookingId);
      formData.append("equipment_id", equipmentId);
      formData.append("description", checkoutForm.description);

      // Append files
      checkoutForm.files.forEach((file) => {
        formData.append("files", file);
      });

      const response = await fetch(`${API_BASE_URL}/equipment-checkout-save`, {
        method: "POST",
        headers: {
          Authorization: `Bearer ${token}`,
        },
        body: formData,
      });

      if (response.ok) {
        alert("Checkout submitted successfully!");
        navigate(`/booking-confirmed/${bookingId}`);
      } else {
        alert("Failed to submit checkout. Please try again.");
      }
    } catch (error) {
      console.error("Submit checkout error:", error);
      alert("Failed to submit checkout. Please try again.");
    } finally {
      setSubmitting(false);
    }
  };

  if (loading) {
    return (
      <ProtectedRoute>
        <div className="min-h-screen bg-white text-black">
          <AppHeader theme="dark" />
          <div className="max-w-7xl mx-auto px-4 py-20 text-center">
            <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-black"></div>
            <p className="mt-4 text-gray-600">Loading equipment checkout...</p>
          </div>
        </div>
      </ProtectedRoute>
    );
  }

  if (!equipment) {
    return (
      <ProtectedRoute>
        <div className="min-h-screen bg-white text-black">
          <AppHeader theme="dark" />
          <div className="max-w-7xl mx-auto px-4 py-20 text-center">
            <h1 className="text-2xl font-medium text-black mb-4">
              Equipment Not Found
            </h1>
            <p className="text-gray-600 mb-8">
              The equipment information could not be found.
            </p>
            <Link to={`/booking-confirmed/${bookingId}`}>
              <Button variant="fill-black" size="lg">
                Back to Booking
              </Button>
            </Link>
          </div>
        </div>
      </ProtectedRoute>
    );
  }

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />

        {/* Main Content */}
        <main className="max-w-7xl mx-auto px-4 md:px-6 lg:px-8 py-8 md:py-12">
          <div className="mb-8">
            <Link
              to={`/booking-confirmed/${bookingId}`}
              className="text-black hover:text-gray-600"
            >
              ← Back to Booking
            </Link>
          </div>

          <h1 className="text-4xl md:text-5xl font-light text-black mb-8">
            {t("booking.checkout.title")}
          </h1>

          {/* Equipment Details Section */}
          <div className="mb-8">
            <h2 className="text-3xl md:text-4xl font-medium text-black mb-6">
              {t("booking.checkout.equipmentDetails")}
            </h2>
            <div className="bg-white border-2 border-black p-6">
              <div className="flex items-center gap-6 mb-6">
                <img
                  src={
                    equipment.photo
                      ? `https://dashboard.nomadicstudios.net/storage/rental/${equipment.photo}`
                      : `https://dashboard.nomadicstudios.net/storage/rental/default-equipment.jpg`
                  }
                  alt={equipment.name}
                  className="w-64 h-48 object-cover"
                  loading="lazy"
                  onError={(e) => {
                    console.log(
                      "Equipment image failed to load:",
                      e.currentTarget.src
                    );
                    e.currentTarget.src =
                      "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjU2IiBoZWlnaHQ9IjE5MiIgdmlld0JveD0iMCAwIDI1NiAxOTIiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSIyNTYiIGhlaWdodD0iMTkyIiBmaWxsPSIjRjNGNEY2Ii8+Cjx0ZXh0IHg9IjEyOCIgeT0iOTYiIGZvbnQtZmFtaWx5PSJBcmlhbCIgZm9udC1zaXplPSIxNCIgZmlsbD0iIzlDQTNBRiIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZG9taW5hbnQtYmFzZWxpbmU9Im1pZGRsZSI+RXF1aXBtZW50PC90ZXh0Pgo8L3N2Zz4K";
                  }}
                  onLoad={() => {
                    console.log(
                      "Equipment image loaded successfully:",
                      equipment.photo
                    );
                  }}
                />
                <div className="flex-1">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-base md:text-lg text-gray-600 mb-1">
                        {t("booking.checkout.name")}
                      </p>
                      <p className="text-2xl md:text-3xl font-medium text-black">
                        {equipment.name}
                      </p>
                    </div>
                    <div>
                      <p className="text-base md:text-lg text-gray-600 mb-1">
                        {t("booking.checkout.serialNumber")}
                      </p>
                      <p className="text-xl md:text-2xl text-black">
                        {equipment.serial_number}
                      </p>
                    </div>
                    <div className="col-span-2">
                      <p className="text-sm text-gray-600 mb-1">Manual</p>
                      <p className="text-lg text-black">
                        {equipment.manual || equipment.manual_link
                          ? "Available"
                          : "Not Available"}
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Upload Photos Section */}
          <div className="mb-8">
            <h2 className="text-xl font-medium text-black mb-6">
              Upload Photos
            </h2>
            <div
              className="border-2 border-dashed border-gray-400 p-12 text-center bg-gray-50"
              onDragOver={handleDragOver}
              onDrop={handleDrop}
            >
              <input
                type="file"
                multiple
                accept=".png,.jpg,.jpeg,.gif"
                onChange={handleFileChange}
                className="hidden"
                id="file-upload"
              />
              <label htmlFor="file-upload" className="cursor-pointer">
                <span className="text-orange-500 font-medium">
                  Click to upload
                </span>{" "}
                or drag and drop
              </label>
              <p className="text-gray-500 text-sm mt-2">
                PNG, JPG or GIF (max. 800x400px)
              </p>
            </div>

            {checkoutForm.files.length > 0 && (
              <div className="mt-6">
                <p className="font-medium text-black mb-4">Selected Files:</p>
                <div className="space-y-2">
                  {checkoutForm.files.map((file, index) => (
                    <div
                      key={index}
                      className="flex items-center justify-between bg-white border border-gray-300 p-3 rounded"
                    >
                      <span className="text-gray-700">{file.name}</span>
                      <Button
                        onClick={() => removeFile(index)}
                        variant="fill-white"
                        size="sm"
                      >
                        Remove
                      </Button>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </div>

          {/* Description Section */}
          <div className="mb-8">
            <h2 className="text-xl font-medium text-black mb-6">Description</h2>
            <textarea
              value={checkoutForm.description}
              onChange={(e) =>
                setCheckoutForm((prev) => ({
                  ...prev,
                  description: e.target.value,
                }))
              }
              placeholder="Enter checkout description..."
              className="w-full h-32 px-4 py-3 border-2 border-black bg-white text-black placeholder-gray-500 focus:border-black focus:outline-none resize-none"
              rows={4}
            />
          </div>

          {/* Submit Button */}
          <div className="text-center">
            <Button
              onClick={handleSubmitCheckout}
              disabled={submitting}
              variant="fill-black"
              size="lg"
              className="px-8"
            >
              {submitting ? "Submitting..." : "Checkout"}
            </Button>
          </div>
        </main>
      </div>
    </ProtectedRoute>
  );
}
