import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useState, useEffect } from "react";
import { Link, useNavigate } from "@remix-run/react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { ProfileImage } from "~/components/shared";
// import { useLanguage } from "~/config/lang/useLanguage";
import { ArrowRight, ChevronRight } from "lucide-react";
import {
  newsAPI,
  eventsAPI,
  reservationsAPI,
  guestInvitationsAPI,
  getUserData,
  API_BASE_URL,
} from "~/lib/api";

export const meta: MetaFunction = () => {
  return [
    { title: "Home - Nomadics Studios" },
    {
      name: "description",
      content:
        "Connect creators with premium studio spaces, professional equipment, and a thriving creative community.",
    },
  ];
};

interface Reservation {
  id: string;
  booking_id: string;
  date: string;
  location: string;
  studio: string;
  price: number;
  image: string;
}

interface Invitation {
  id: string;
  name: string;
  date: string;
  location: string;
  image: string;
  myGuestOrBy: string; // "My Guest" or "My By"
  hostUuid: string;
}

interface NewsItem {
  id: string;
  title: string;
  description: string;
  image: string;
  author: string;
  date: string;
  slug: string;
}

interface Event {
  id: string;
  event_id?: number;
  title: string;
  location: string;
  date: string;
  time: string;
  image: string;
  interested_count?: number;
  going_count?: number;
  _originalData?: any;
}

interface Location {
  id: number;
  location: string;
  type: string | null;
  coming_soon: number;
  banner_picture: string | null;
  description: string | null;
}

export default function Home() {
  const navigate = useNavigate();
  // const { t } = useLanguage();
  const [bookingForm, setBookingForm] = useState({
    where: "",
    start: "",
    startTime: "09:00",
    end: "",
    endTime: "17:00",
  });
  const [advanceDateFlag, setAdvanceDateFlag] = useState<string | null>(null);

  // State for locations
  const [locations, setLocations] = useState<Location[]>([]);
  const [locationsLoading, setLocationsLoading] = useState(true);
  const [homeCityBanner, setHomeCityBanner] = useState<string>(""); ///images/my-home.png

  // Debug: Log whenever homeCityBanner changes
  useEffect(() => {
    console.log("🏠 homeCityBanner state changed to:", homeCityBanner);
  }, [homeCityBanner]);

  // Generate slug from title (same as news.tsx)
  const generateSlug = (title: string, id: number) => {
    return `${title
      .toLowerCase()
      .replace(/[^a-z0-9]+/g, "-")
      .replace(/(^-|-$)/g, "")}-${id}`;
  };

  // State for API data
  const [news, setNews] = useState<NewsItem[]>([]);
  const [events, setEvents] = useState<Event[]>([]);
  const [reservations, setReservations] = useState<Reservation[]>([]);
  const [invitations, setInvitations] = useState<Invitation[]>([]);
  const [showAllReservations, setShowAllReservations] = useState(false);
  const [showAllInvitations, setShowAllInvitations] = useState(false);
  const [userRole, setUserRole] = useState<string>("");
  const [currentUserUuid, setCurrentUserUuid] = useState<string>("");
  const [homeCityId, setHomeCityId] = useState<number | null>(null);
  const [loading, setLoading] = useState(true);
  const [newsLoading, setNewsLoading] = useState(true);
  const [eventsLoading, setEventsLoading] = useState(true);
  const [reservationsLoading, setReservationsLoading] = useState(true);
  const [invitationsLoading, setInvitationsLoading] = useState(true);

  // Get user data on mount
  useEffect(() => {
    const userData = getUserData();
    console.log("Full user data from getUserData():", userData);

    if (userData) {
      setUserRole(userData.role || "");
      setCurrentUserUuid(userData.uuid || "");
      const flagDate = userData.advance_date_flag || null;
      console.log("Advance date flag from user data:", flagDate);
      setAdvanceDateFlag(flagDate);

      // Extract home city ID for filtering news and events
      // Login response has 'home_city' field (not 'home_city_id')
      // Try both home_city and home_city_id for compatibility
      const homeCityIdValue = userData.home_city
        ? parseInt(userData.home_city.toString(), 10)
        : userData.home_city_id
        ? parseInt(userData.home_city_id.toString(), 10)
        : null;
      console.log(
        "User home city ID:",
        homeCityIdValue,
        "from field:",
        userData.home_city
          ? "home_city"
          : userData.home_city_id
          ? "home_city_id"
          : "none"
      );
      setHomeCityId(homeCityIdValue);

      // Pre-select user's home city in booking form
      if (homeCityIdValue) {
        setBookingForm((prev) => ({
          ...prev,
          where: homeCityIdValue.toString(),
        }));
      }
    } else {
      console.warn("No user data found - user may not be logged in");
    }
  }, []);

  // Fetch locations data
  useEffect(() => {
    const fetchLocations = async () => {
      try {
        setLocationsLoading(true);
        const token =
          localStorage.getItem("authToken") ||
          localStorage.getItem("auth_token") ||
          localStorage.getItem("token");
        if (!token) {
          console.error("No auth token found");
          setLocations([]);
          setLocationsLoading(false);
          return;
        }

        const response = await fetch(`${API_BASE_URL}/locations`, {
          method: "GET",
          headers: {
            Authorization: `Bearer ${token}`,
            Accept: "application/json",
          },
        });

        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();

        // API returns { status: true, data: [...] }
        if (result.status && Array.isArray(result.data)) {
          // Filter out duplicate locations by using a Map with location id as key
          // Priority: Keep entries with banner_picture data over null entries
          const locationMap = new Map<number, Location>();

          result.data.forEach((loc: Location) => {
            const existingLoc = locationMap.get(loc.id);

            // If no existing location, or existing has null banner but new one has banner, use the new one
            if (
              !existingLoc ||
              (!existingLoc.banner_picture && loc.banner_picture)
            ) {
              locationMap.set(loc.id, loc);
            }
            // If existing has banner but new one doesn't, keep the existing
            // If both have banners or both are null, keep the first one (existing)
          });

          const uniqueLocations = Array.from(
            locationMap.values()
          ) as Location[];
          console.log(
            "Fetched locations after deduplication:",
            uniqueLocations.map((l) => ({
              id: l.id,
              name: l.location,
              banner: l.banner_picture,
            }))
          );
          setLocations(uniqueLocations);
        } else {
          console.error("Unexpected response format:", result);
          setLocations([]);
        }
      } catch (error) {
        console.error("Failed to fetch locations:", error);
        setLocations([]);
      } finally {
        setLocationsLoading(false);
      }
    };

    fetchLocations();
  }, []);

  // Update banner when homeCityId or locations change
  useEffect(() => {
    console.log(
      "Banner update effect triggered - homeCityId:",
      homeCityId,
      "type:",
      typeof homeCityId,
      "locations length:",
      locations.length
    );

    if (homeCityId !== null && locations.length > 0) {
      console.log(
        "Looking for location with ID:",
        homeCityId,
        "Available locations:",
        locations.map((l) => ({
          id: l.id,
          type: typeof l.id,
          name: l.location,
          banner: l.banner_picture,
        }))
      );

      const homeCityLocation = locations.find((loc: Location) => {
        console.log(
          `Comparing location ID ${
            loc.id
          } (${typeof loc.id}) with homeCityId ${homeCityId} (${typeof homeCityId})`
        );
        // Use == for loose comparison to handle string vs number
        return loc.id == homeCityId;
      });

      console.log("Found home city location:", homeCityLocation);

      if (homeCityLocation) {
        // First, check if we have the banner in the location object
        if (homeCityLocation.banner_picture) {
          // Check if banner_picture is already a full URL or just a filename
          const isFullUrl = homeCityLocation.banner_picture.startsWith("http");
          const fullBannerUrl = isFullUrl
            ? homeCityLocation.banner_picture
            : `https://dashboard.nomadicstudios.net/storage/location/${homeCityLocation.banner_picture}`;
          console.log(
            "Setting home city banner from locations API:",
            fullBannerUrl
          );
          setHomeCityBanner(fullBannerUrl);
        } else {
          // If banner_picture is null, check localStorage cache first
          const cachedBanner = localStorage.getItem(
            `location_banner_${homeCityId}`
          );
          if (cachedBanner) {
            console.log(
              "Using cached banner for location:",
              homeCityId,
              cachedBanner
            );
            setHomeCityBanner(cachedBanner);
          } else {
            // If no cache, fetch the specific location's banner directly
            console.log(
              "Banner is null, fetching specific location data for ID:",
              homeCityId
            );
            fetchLocationBanner(homeCityId);
          }
        }
      } else {
        console.log(
          "No location found for home city ID:",
          homeCityId,
          "Available locations:",
          locations.map((l) => ({
            id: l.id,
            name: l.location,
            banner: l.banner_picture,
          }))
        );
        // Fallback to default banner
        setHomeCityBanner("/images/my-home.png");
      }
    } else {
      console.log(
        "Conditions not met - homeCityId:",
        homeCityId,
        "locations length:",
        locations.length
      );
    }
  }, [homeCityId, locations]);

  // Function to fetch specific location banner when locations API returns null
  const fetchLocationBanner = async (locationId: number) => {
    try {
      console.log("Fetching banner for location ID:", locationId);
      const token =
        localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token");

      if (!token) {
        console.error("No auth token found for banner fetch");
        setHomeCityBanner("/images/my-home.png");
        return;
      }

      // Try to fetch specific location details
      const response = await fetch(`${API_BASE_URL}/locations/${locationId}`, {
        method: "GET",
        headers: {
          Authorization: `Bearer ${token}`,
          Accept: "application/json",
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      console.log("Individual location API response:", result);

      if (result.status && result.data && result.data.banner_picture) {
        // Check if banner_picture is already a full URL or just a filename
        const isFullUrl = result.data.banner_picture.startsWith("http");
        const fullBannerUrl = isFullUrl
          ? result.data.banner_picture
          : `https://dashboard.nomadicstudios.net/storage/location/${result.data.banner_picture}`;
        console.log(
          "Found banner from individual location API:",
          fullBannerUrl
        );

        // Cache the banner URL for future use
        localStorage.setItem(`location_banner_${locationId}`, fullBannerUrl);

        setHomeCityBanner(fullBannerUrl);
      } else {
        console.warn(
          "No banner found in individual location API for ID:",
          locationId
        );
        setHomeCityBanner("/images/my-home.png");
      }
    } catch (error) {
      console.error("Failed to fetch individual location banner:", error);
      setHomeCityBanner("/images/my-home.png");
    }
  };

  // Listen for localStorage changes (when account settings are updated)
  useEffect(() => {
    const handleStorageChange = () => {
      console.log("Storage changed, reloading user data...");
      const userData = getUserData();
      console.log("User data from getUserData():", userData);

      if (userData) {
        const homeCityIdValue = userData.home_city
          ? parseInt(userData.home_city.toString(), 10)
          : userData.home_city_id
          ? parseInt(userData.home_city_id.toString(), 10)
          : null;

        console.log(
          "Extracted home city ID:",
          homeCityIdValue,
          "Current homeCityId:",
          homeCityId
        );

        if (homeCityIdValue !== homeCityId) {
          console.log(
            "Home city changed from",
            homeCityId,
            "to",
            homeCityIdValue
          );

          // Reset loading states to trigger fresh API calls
          setNewsLoading(true);
          setEventsLoading(true);

          setHomeCityId(homeCityIdValue);

          // Update booking form
          if (homeCityIdValue) {
            setBookingForm((prev) => ({
              ...prev,
              where: homeCityIdValue.toString(),
            }));
          }
        } else {
          console.log("Home city ID unchanged:", homeCityIdValue);
        }
      } else {
        console.log("No user data found in localStorage");
      }
    };

    // Listen for custom storage event
    window.addEventListener("storage", handleStorageChange);
    window.addEventListener("accountSettingsUpdated", handleStorageChange);

    return () => {
      window.removeEventListener("storage", handleStorageChange);
      window.removeEventListener("accountSettingsUpdated", handleStorageChange);
    };
  }, [homeCityId]);

  // Fetch reservations data
  useEffect(() => {
    const fetchReservations = async () => {
      try {
        setReservationsLoading(true);
        const limit = showAllReservations ? undefined : 4;
        const response = await reservationsAPI.getUpcoming(limit);

        if (response.status && response.data && Array.isArray(response.data)) {
          const mappedReservations: Reservation[] = response.data.map(
            (item: any) => {
              // Format date from start_date
              const formattedDate = item.start_date
                ? new Date(item.start_date).toLocaleDateString("en-US", {
                    year: "numeric",
                    month: "short",
                    day: "numeric",
                  })
                : "Date TBD";

              return {
                id: item.id?.toString() || Math.random().toString(),
                booking_id: item.booking_id?.toString() || "",
                date: formattedDate,
                location: item.loc?.location || "Location TBD",
                studio: item.booking_id || "Studio Booking",
                price: item.total_points || 0,
                image: item.loc_landing?.banner_picture
                  ? `https://dashboard.nomadicstudios.net/storage/location/${item.loc_landing.banner_picture}`
                  : "/images/studios/default.png",
              };
            }
          );
          setReservations(mappedReservations);
        } else {
          setReservations([]);
        }
      } catch (error) {
        console.error("Failed to fetch reservations:", error);
        setReservations([]);
      } finally {
        setReservationsLoading(false);
      }
    };

    fetchReservations();
  }, [showAllReservations]);

  // Fetch guest invitations data
  useEffect(() => {
    const fetchInvitations = async () => {
      try {
        setInvitationsLoading(true);
        const limit = showAllInvitations ? undefined : 3;
        const response = await guestInvitationsAPI.getAll(limit);

        if (response.status && response.data && Array.isArray(response.data)) {
          const mappedInvitations: Invitation[] = response.data.map(
            (item: any) => {
              // Determine if this is "My Guest" or "My By"
              const isMyGuest = item.host_uuid === currentUserUuid;
              const myGuestOrBy = isMyGuest ? "My Guest" : "My By";

              // Format date from invitation_date
              const formattedDate = item.invitation_date
                ? new Date(item.invitation_date).toLocaleDateString("en-US", {
                    year: "numeric",
                    month: "short",
                    day: "numeric",
                    hour: "2-digit",
                    minute: "2-digit",
                  })
                : "Date TBD";

              // Get guest name from guest object or use host info
              const guestName = isMyGuest
                ? `${item.guest?.fname || ""} ${
                    item.guest?.lname || ""
                  }`.trim() || "Guest"
                : `${item.host?.fname || ""} ${
                    item.host?.lname || ""
                  }`.trim() || "Host";

              return {
                id: item.id?.toString() || Math.random().toString(),
                name: guestName,
                date: formattedDate,
                location: item.location?.location || "Location TBD",
                image: "/images/team-member.png", // API doesn't provide image
                myGuestOrBy: myGuestOrBy,
                hostUuid: item.host_uuid || "",
              };
            }
          );
          setInvitations(mappedInvitations);
        } else {
          setInvitations([]);
        }
      } catch (error) {
        console.error("Failed to fetch invitations:", error);
        setInvitations([]);
      } finally {
        setInvitationsLoading(false);
      }
    };

    if (currentUserUuid) {
      fetchInvitations();
    }
  }, [showAllInvitations, currentUserUuid]);

  // Invitation actions
  const handleCancelInvitation = async (id: string) => {
    if (!confirm("Are you sure you want to cancel this invitation?")) return;
    try {
      await guestInvitationsAPI.cancel(id);
      // remove from list optimistically
      setInvitations((prev) => prev.filter((inv) => inv.id !== id));
    } catch (error) {
      console.error("Failed to cancel invitation:", error);
      alert("Failed to cancel invitation. Please try again.");
    }
  };

  const handleResendInvitation = async (id: string) => {
    try {
      await guestInvitationsAPI.resend(id);
      alert("Invitation resent successfully.");
    } catch (error) {
      console.error("Failed to resend invitation:", error);
      alert("Failed to resend invitation. Please try again.");
    }
  };

  // Fetch news data
  useEffect(() => {
    // Only fetch if we have a valid homeCityId and haven't loaded already
    if (homeCityId === null || newsLoading === false) return;

    const fetchNews = async () => {
      try {
        console.log("Fetching news with filterLocation:", homeCityId);
        // Filter by home city if available - API expects filterLocation query param
        const newsData = await newsAPI.getAll(
          undefined,
          undefined,
          homeCityId !== null ? homeCityId : undefined
        );

        console.log(
          "Fetched news with filterLocation:",
          homeCityId,
          "Result:",
          newsData
        );

        // Map API data to component interfaces
        const mappedNews: NewsItem[] = (newsData.data || []).map(
          (item: any) => {
            const title = item.name || "No title";
            const id = item.id || Math.random();
            return {
              id: id.toString(),
              title,
              description: item.heading_intro || "No description",
              image:
                item.banner ||
                item.thumbnail ||
                "https://via.placeholder.com/400x240/f3f4f6/9ca3af?text=News+Image",
              author: item.created_by
                ? `${item.created_by.fname} ${item.created_by.lname}`
                : "Unknown",
              date: item.created_at
                ? new Date(item.created_at).toLocaleDateString()
                : new Date().toLocaleDateString(),
              slug: generateSlug(title, id),
            };
          }
        );

        setNews(mappedNews);
      } catch (error) {
        console.error("Failed to fetch news:", error);
        setNews([]);
      } finally {
        setNewsLoading(false);
      }
    };

    fetchNews();
  }, [homeCityId]);

  // Fetch events data
  useEffect(() => {
    // Only fetch if we have a valid homeCityId and haven't loaded already
    if (homeCityId === null || eventsLoading === false) return;

    const fetchEvents = async () => {
      try {
        console.log("Fetching events for home city:", homeCityId);
        // Fetch all events (API response includes location_id field)
        const eventsData = await eventsAPI.getAll();

        console.log(
          "Fetched events, home city:",
          homeCityId,
          "All events:",
          eventsData
        );

        // Filter events by location_id matching user's home_city
        // Ensure both values are compared as numbers
        const filteredEvents =
          homeCityId !== null
            ? (eventsData.data || []).filter((item: any) => {
                const itemLocationId = parseInt(
                  item.location_id?.toString() || "0",
                  10
                );
                const userHomeCity = parseInt(homeCityId.toString(), 10);
                console.log(
                  `Comparing event ${item.event_id}: location_id=${itemLocationId} vs home_city=${userHomeCity}`
                );
                return itemLocationId === userHomeCity;
              })
            : eventsData.data || [];

        console.log(
          "Filtered events count:",
          filteredEvents.length,
          "Filtered events:",
          filteredEvents
        );

        // Map API data to component interfaces, keeping the original API data
        const mappedEvents: Event[] = filteredEvents.map((item: any) => ({
          id: item.event_id?.toString() || Math.random().toString(),
          event_id: item.event_id,
          title: item.event_title || "No title",
          location: item.address || "Location TBD",
          date: item.event_date
            ? new Date(item.event_date)
                .toLocaleDateString("en-US", {
                  weekday: "long",
                  year: "numeric",
                  month: "short",
                  day: "numeric",
                })
                .toUpperCase()
            : "Date TBD",
          time: item.event_time || "Time TBD",
          image:
            item.thumbnail ||
            item.picture ||
            "https://via.placeholder.com/400x240/1f2937/ffffff?text=Event+Image",
          interested_count: item.interested_count || 0,
          going_count: item.going_count || 0,
          // Keep the original API data for passing to detail page
          _originalData: item,
        }));

        setEvents(mappedEvents);
      } catch (error) {
        console.error("Failed to fetch events:", error);
        // Set empty array if API fails
        setEvents([]);
      } finally {
        setEventsLoading(false);
        setLoading(false);
      }
    };

    fetchEvents();
  }, [homeCityId]);

  const handleBookingSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    // Validate all required fields
    if (
      !bookingForm.where ||
      !bookingForm.start ||
      !bookingForm.startTime ||
      !bookingForm.end ||
      !bookingForm.endTime
    ) {
      alert("Please fill in all fields");
      return;
    }

    // HTML time input already gives us HH:mm format (24-hour)
    // Just need to add seconds
    const formatTime = (time: string) => {
      return `${time}:00`; // time is already in HH:mm format
    };

    // Find the selected location name
    const selectedLocation = locations.find(
      (loc) => loc.id.toString() === bookingForm.where
    );
    const locationName = selectedLocation
      ? selectedLocation.location
      : bookingForm.where;

    // Prepare booking data
    const bookingData = {
      location: bookingForm.where,
      location_name: locationName,
      start_date: bookingForm.start,
      end_date: bookingForm.end,
      start_time: formatTime(bookingForm.startTime),
      end_time: formatTime(bookingForm.endTime),
    };

    // Store in sessionStorage to pass to next page
    sessionStorage.setItem("bookingData", JSON.stringify(bookingData));

    // Navigate to booking studio selection page
    navigate("/book-studio");
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-black text-white">
        <AppHeader theme="dark" />

        <main className="pt-10">
          {/* Hero Section with Booking Form */}
          <section className="relative h-screen bg-cover bg-center bg-no-repeat">
            {/* Background Image */}
            <div className="absolute inset-0 bg-[#030303]" />
            <img
              src={homeCityBanner}
              alt=""
              className="absolute inset-0 h-full w-full object-cover"
              onLoad={() =>
                console.log("Banner image loaded successfully:", homeCityBanner)
              }
              onError={() =>
                console.error("Banner image failed to load:", homeCityBanner)
              }
            />
            <div className="absolute inset-0 bg-black/50" />
            <div className="relative z-10 flex flex-col items-center justify-center h-full px-4">
              {/* Hero Title */}
              <div className="text-center text-white mb-12">
                <h1 className="text-[32px] leading-[1.1] md:text-[56px] lg:text-heading-1 uppercase mb-4 font-bold tracking-[-0.02em] max-w-4xl mx-auto">
                  LET'S CREATE <br /> SOMETHING
                </h1>
              </div>

              {/* Booking Form - Only show if user role is not 'guest' */}
              {userRole !== "guest" && (
                <div className="w-full max-w-7xl mx-auto">
                  <div className="bg-white/95 backdrop-blur-sm p-6 md:p-8 rounded-none shadow-2xl">
                    <form onSubmit={handleBookingSubmit}>
                      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-6 gap-4 items-end">
                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider"
                            style={{ color: "#000000" }}
                          >
                            Where
                          </label>
                          <select
                            value={bookingForm.where}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                where: e.target.value,
                              })
                            }
                            required
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                          >
                            <option value="">Location</option>
                            {locationsLoading ? (
                              <option disabled>Loading...</option>
                            ) : (
                              locations.map((location) => (
                                <option key={location.id} value={location.id}>
                                  {location.location}
                                </option>
                              ))
                            )}
                          </select>
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider"
                            style={{ color: "#000000" }}
                          >
                            Start
                          </label>
                          <input
                            type="date"
                            value={bookingForm.start}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                start: e.target.value,
                              })
                            }
                            required
                            min={advanceDateFlag || undefined}
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                            placeholder="dd/mm/yyyy"
                          />
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider"
                            style={{ color: "#000000" }}
                          >
                            Start Time
                          </label>
                          <input
                            type="time"
                            value={bookingForm.startTime}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                startTime: e.target.value,
                              })
                            }
                            required
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                          />
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider"
                            style={{ color: "#000000" }}
                          >
                            End
                          </label>
                          <input
                            type="date"
                            value={bookingForm.end}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                end: e.target.value,
                              })
                            }
                            required
                            min={advanceDateFlag || undefined}
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                            placeholder="dd/mm/yyyy"
                          />
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider"
                            style={{ color: "#000000" }}
                          >
                            End Time
                          </label>
                          <input
                            type="time"
                            value={bookingForm.endTime}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                endTime: e.target.value,
                              })
                            }
                            required
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                          />
                        </div>

                        {/* Search Button */}
                        <div className="text-left">
                          <Button
                            type="submit"
                            variant="fill-black"
                            size="md"
                            className="w-full px-8 py-4 text-base font-medium uppercase tracking-wider transition-all duration-200 hover:shadow-lg"
                          >
                            Search
                          </Button>
                        </div>
                      </div>

                      {/* Booking minimum text */}
                      <div className="mt-4">
                        <p className="text-body-3" style={{ color: "#000000" }}>
                          Booking for a minimum of 2 hours
                        </p>
                      </div>
                    </form>
                  </div>
                </div>
              )}
            </div>
          </section>

          {/* Reservations & Invitations Section - White Background */}
          <section className="relative w-full bg-white">
            <div className="max-w-[1400px] mx-auto px-4 lg:px-14 py-14 lg:py-[120px]">
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
                {/* Left Column - Reservations */}
                <div>
                  <div className="flex justify-between items-center mb-6">
                    <h2 className="text-heading-5" style={{ color: "#000000" }}>
                      Upcoming reservations
                    </h2>
                    <button
                      onClick={() =>
                        setShowAllReservations(!showAllReservations)
                      }
                      className="flex items-center text-body-3 hover:opacity-80"
                      style={{ color: "#000000" }}
                    >
                      {showAllReservations ? "Show less" : "See all"}{" "}
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </button>
                  </div>

                  {reservationsLoading ? (
                    <div className="space-y-4">
                      {[...Array(3)].map((_, i) => (
                        <div
                          key={i}
                          className="animate-pulse flex items-center gap-4 p-4 border border-gray-200"
                        >
                          <div className="w-16 h-16 bg-gray-200"></div>
                          <div className="flex-1 space-y-2">
                            <div className="h-3 bg-gray-200 w-24"></div>
                            <div className="h-4 bg-gray-200 w-32"></div>
                            <div className="h-3 bg-gray-200 w-28"></div>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : reservations.length > 0 ? (
                    <div className="space-y-4">
                      {reservations.map((reservation) => (
                        <div
                          key={reservation.id}
                          className="flex items-center gap-4 p-4 border border-gray-200 hover:border-gray-300 transition-colors"
                        >
                          <img
                            src={reservation.image}
                            alt={reservation.studio}
                            className="w-16 h-16 object-cover"
                          />
                          <div className="flex-1">
                            <p
                              className="text-body-3 mb-1"
                              style={{ color: "#000000" }}
                            >
                              {reservation.date}
                            </p>
                            <p
                              className="text-body-2"
                              style={{ color: "#000000" }}
                            >
                              {reservation.location}
                            </p>
                            <p
                              className="text-body-3"
                              style={{ color: "#000000" }}
                            >
                              {reservation.studio}
                            </p>
                          </div>
                          <div className="text-right">
                            <p
                              className="text-body-2"
                              style={{ color: "#000000" }}
                            >
                              {reservation.price} pts
                            </p>
                            <Link
                              to={`/booking-confirmed/${reservation.booking_id}`}
                              className="text-body-3 flex items-center hover:opacity-80"
                              style={{ color: "#000000" }}
                            >
                              See details{" "}
                              <ChevronRight className="ml-1 h-3 w-3" />
                            </Link>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center py-8">
                      <p className="text-gray-500">No upcoming reservations</p>
                    </div>
                  )}
                </div>

                {/* Right Column - Guest Invitations */}
                <div>
                  <div className="flex justify-between items-center mb-6">
                    <h2 className="text-heading-5" style={{ color: "#000000" }}>
                      Guest invitations
                    </h2>
                    <button
                      onClick={() => setShowAllInvitations(!showAllInvitations)}
                      className="flex items-center text-body-3 hover:opacity-80"
                      style={{ color: "#000000" }}
                    >
                      {showAllInvitations ? "Show less" : "See all"}{" "}
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </button>
                  </div>

                  {invitationsLoading ? (
                    <div className="space-y-4">
                      {[...Array(3)].map((_, i) => (
                        <div
                          key={i}
                          className="animate-pulse flex items-center gap-4 p-4 border border-gray-200"
                        >
                          <div className="w-16 h-16 bg-gray-200 rounded-full"></div>
                          <div className="flex-1 space-y-2">
                            <div className="h-4 bg-gray-200 w-32"></div>
                            <div className="h-3 bg-gray-200 w-24"></div>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : invitations.length > 0 ? (
                    <div className="space-y-4">
                      {invitations.map((invitation) => (
                        <div
                          key={invitation.id}
                          className="flex items-center gap-4 p-4 border border-gray-200 hover:border-gray-300 transition-colors"
                        >
                          <ProfileImage
                            src={invitation.image}
                            alt={invitation.name}
                            className="w-16 h-16 rounded-full object-cover"
                          />
                          <div className="flex-1">
                            <p
                              className="text-body-3 mb-1"
                              style={{ color: "#000000" }}
                            >
                              Date: {invitation.date}
                            </p>
                            <p
                              className="text-body-3 mb-1"
                              style={{ color: "#000000" }}
                            >
                              Location: {invitation.location}
                            </p>
                            <p
                              className="text-body-3"
                              style={{ color: "#000000" }}
                            >
                              {invitation.myGuestOrBy}
                            </p>
                          </div>
                          <div className="text-right">
                            <p
                              className="text-body-2 mb-1"
                              style={{ color: "#000000" }}
                            >
                              {invitation.name}
                            </p>
                            <div className="flex flex-col items-end gap-2">
                              {/* <button
                                className="text-body-3 flex items-center mt-1 hover:opacity-80"
                                style={{ color: "#000000" }}
                              >
                                See details{" "}
                                <ChevronRight className="ml-1 h-3 w-3" />
                              </button> */}

                              <div className="flex gap-2 mt-2">
                                <button
                                  onClick={() =>
                                    handleResendInvitation(invitation.id)
                                  }
                                  className="text-[13px] px-3 py-1 border border-black/20 rounded uppercase hover:opacity-80 cursor-pointer"
                                  style={{ color: "#000000" }}
                                >
                                  Resend
                                </button>
                                <button
                                  onClick={() =>
                                    handleCancelInvitation(invitation.id)
                                  }
                                  className="text-[13px] px-3 py-1 border border-red-400 text-red-600 rounded uppercase hover:opacity-80 cursor-pointer"
                                >
                                  Cancel
                                </button>
                              </div>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center py-8">
                      <p className="text-gray-500">No guest invitations</p>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </section>

          {/* News Section - White Background */}
          <section className="relative w-full bg-white">
            <div className="max-w-[1400px] mx-auto px-4 lg:px-14 py-14">
              <div className="flex justify-between items-center mb-8">
                <h2 className="text-heading-4" style={{ color: "#000000" }}>
                  News
                </h2>
                <button
                  onClick={() => (window.location.href = "/news")}
                  className="flex items-center text-body-3 hover:opacity-80 transition-opacity"
                  style={{ color: "#000000" }}
                >
                  See more <ArrowRight className="ml-2 h-4 w-4" />
                </button>
              </div>

              {newsLoading ? (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {[...Array(6)].map((_, i) => (
                    <div key={i} className="animate-pulse">
                      <div className="w-full h-48 bg-gray-200 mb-4"></div>
                      <div className="h-4 bg-gray-200 mb-2"></div>
                      <div className="h-3 bg-gray-200 mb-2"></div>
                      <div className="flex items-center gap-2">
                        <div className="w-5 h-5 bg-gray-200 rounded-full"></div>
                        <div className="h-3 bg-gray-200 w-20"></div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : news.length > 0 ? (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {news.slice(0, 6).map((item) => {
                    // Format date like event cards
                    const formatCardDate = (dateString: string) => {
                      const date = new Date(dateString);
                      if (isNaN(date.getTime())) {
                        return { day: "TBD", month: "DATE" };
                      }
                      const day = date.getDate().toString();
                      const month = date
                        .toLocaleDateString("en-US", { month: "short" })
                        .toUpperCase();
                      return { day, month };
                    };

                    const { day, month } = formatCardDate(item.date);

                    return (
                      <Link
                        key={item.id}
                        to={`/news-details/${item.slug}`}
                        className="group cursor-pointer bg-white border border-gray-200 overflow-hidden hover:shadow-lg transition-all duration-300 block"
                      >
                        {/* News Image */}
                        <div className="relative overflow-hidden aspect-[426.67/240]">
                          <img
                            src={item.image}
                            alt={item.title}
                            className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                          />
                        </div>

                        {/* News Details */}
                        <div className="flex items-start gap-4 p-4 bg-white">
                          {/* Date Box */}
                          <div className="text-center min-w-[48px] flex-shrink-0">
                            <div
                              className="text-[24px] font-semibold leading-tight"
                              style={{ color: "#000000" }}
                            >
                              {day}
                            </div>
                            <div
                              className="text-[14px] font-normal uppercase"
                              style={{ color: "#000000" }}
                            >
                              {month}
                            </div>
                          </div>

                          {/* News Info */}
                          <div className="flex-1 min-w-0">
                            <h3
                              className="text-body-2 font-normal mb-1 line-clamp-2"
                              style={{ color: "#000000" }}
                            >
                              {item.title}
                            </h3>
                            <p
                              className="text-body-3 mb-1 line-clamp-2"
                              style={{ color: "#000000" }}
                            >
                              {item.description}
                            </p>
                            <p
                              className="text-body-3"
                              style={{ color: "#000000" }}
                            >
                              {item.author}
                            </p>
                          </div>
                        </div>
                      </Link>
                    );
                  })}
                </div>
              ) : (
                <div className="text-center py-12">
                  <p className="text-gray-500">
                    No news articles available at the moment.
                  </p>
                </div>
              )}
            </div>
          </section>

          {/* Upcoming Events Section - White Background */}
          <section className="relative w-full bg-white">
            <div className="max-w-[1400px] mx-auto px-4 lg:px-14 py-14 lg:py-[120px]">
              <div className="flex justify-between items-center mb-12">
                <h2 className="text-heading-4" style={{ color: "#000000" }}>
                  Upcoming events
                </h2>
                <button
                  onClick={() => (window.location.href = "/events")}
                  className="flex items-center text-body-3 hover:opacity-80 transition-opacity"
                  style={{ color: "#000000" }}
                >
                  See all <ArrowRight className="ml-2 h-4 w-4" />
                </button>
              </div>

              {eventsLoading ? (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {[...Array(3)].map((_, i) => (
                    <div key={i} className="animate-pulse">
                      <div className="w-full h-48 bg-gray-200 mb-4"></div>
                      <div className="h-4 bg-gray-200 mb-2"></div>
                      <div className="h-3 bg-gray-200 mb-2"></div>
                      <div className="h-3 bg-gray-200 mb-2"></div>
                    </div>
                  ))}
                </div>
              ) : events.length > 0 ? (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {events.slice(0, 3).map((event) => {
                    // Format date like event cards
                    const formatCardDate = (dateString: string) => {
                      const date = new Date(dateString);
                      if (isNaN(date.getTime())) {
                        return { day: "TBD", month: "DATE" };
                      }
                      const day = date.getDate().toString();
                      const month = date
                        .toLocaleDateString("en-US", { month: "short" })
                        .toUpperCase();
                      return { day, month };
                    };

                    const { day, month } = formatCardDate(event.date);

                    return (
                      <Link
                        key={event.id}
                        to={`/events-details/${event.event_id || event.id}`}
                        state={{ event: event._originalData }}
                        className="group cursor-pointer bg-white border border-gray-200 overflow-hidden hover:shadow-lg transition-all duration-300 block"
                      >
                        {/* Event Image */}
                        <div className="relative overflow-hidden aspect-[426.67/240]">
                          <img
                            src={event.image}
                            alt={event.title}
                            className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                          />
                        </div>

                        {/* Event Details */}
                        <div className="flex items-start gap-4 p-4 bg-white">
                          {/* Date Box */}
                          <div className="text-center min-w-[48px] flex-shrink-0">
                            <div
                              className="text-[24px] font-semibold leading-tight"
                              style={{ color: "#000000" }}
                            >
                              {day}
                            </div>
                            <div
                              className="text-[14px] font-normal uppercase"
                              style={{ color: "#000000" }}
                            >
                              {month}
                            </div>
                          </div>

                          {/* Event Info */}
                          <div className="flex-1 min-w-0">
                            <h3
                              className="text-body-2 font-normal mb-1 line-clamp-2"
                              style={{ color: "#000000" }}
                            >
                              {event.title}
                            </h3>
                            <p
                              className="text-body-3 mb-1"
                              style={{ color: "#000000" }}
                            >
                              {event.time}
                            </p>
                            <p
                              className="text-body-3 line-clamp-1"
                              style={{ color: "#000000" }}
                            >
                              {event.location}
                            </p>
                            {(event.interested_count || event.going_count) && (
                              <p
                                className="text-body-3 mt-2"
                                style={{ color: "#000000" }}
                              >
                                {(event.interested_count || 0) +
                                  (event.going_count || 0)}{" "}
                                interested • {event.going_count || 0} going
                              </p>
                            )}
                          </div>
                        </div>
                      </Link>
                    );
                  })}
                </div>
              ) : (
                <div className="text-center py-12">
                  <p className="text-gray-500">
                    No events available at your home location.
                  </p>
                </div>
              )}
            </div>
          </section>
        </main>
      </div>
    </ProtectedRoute>
  );
}
