import type { MetaFunction, LoaderFunctionArgs } from "@remix-run/node";
import { useLoaderData, useNavigate } from "@remix-run/react";
import {
  WelcomeSection,
  LocationInfo,
  StudiosSection,
  EventsSection,
  PlanYourVisitSection,
} from "~/components/pages/locations";
import { MembershipSection } from "~/components/shared/MembershipSection";
import { PeopleSection } from "~/components/shared/PeopleSection";
import { useLanguage } from "~/hooks/useLanguage";
import en from "~/config/lang/en.json";
import sp from "~/config/lang/sp.json";
import { API_BASE_URL } from "~/lib/api";
const translations = { en, sp };

export async function loader({ params }: LoaderFunctionArgs) {
  // get location from params
  const location = params.location;

  if (!location) {
    throw new Response("Location Not Found", { status: 404 });
  }

  // Fetch API data
  const apiRes = await fetch(
    `${API_BASE_URL}/locations/${location}`
  );
  const apiData = await apiRes.json();

  // Fetch studio details to get the slugs
  const studiosWithSlugs = await Promise.all(
    (apiData.data.studios || []).map(async (studio: any) => {
      try {
        // Fetch studio details from studios API by type
        const studioType = studio.studio_type?.toLowerCase() || 'music';
        const studiosRes = await fetch(`${API_BASE_URL}/studios?type=${studioType}`);
        const studiosData = await studiosRes.json();
        
        // Find matching studio by id
        const fullStudioData = studiosData.data?.studios?.find((s: any) => s.id === studio.id);
        
        if (fullStudioData && fullStudioData.studio_slug) {
          return {
            ...studio,
            studio_slug: fullStudioData.studio_slug,
            studio_name: fullStudioData.studio_name || studio.studio_name
          };
        }
      } catch (error) {
        console.error(`Failed to fetch slug for studio ${studio.id}:`, error);
      }
      
      // Fallback: return studio as is
      return studio;
    })
  );

  // Update apiData with studios that have slugs
  apiData.data.studios = studiosWithSlugs;

  // Fetch full event details from the main events API for each event
  const eventsWithDetails = await Promise.all(
    (apiData.data.events || []).map(async (event: any) => {
      try {
        // Try to fetch full event details from the events API
        const eventDetailsRes = await fetch(`${API_BASE_URL}/events`);
        const allEventsData = await eventDetailsRes.json();
        
        // Find the matching event in the full events list
        const fullEventData = allEventsData.data?.find((e: any) => e.event_id === event.id);
        
        if (fullEventData) {
          // Return the event with full details for state passing
          return {
            ...event,
            eventData: fullEventData
          };
        }
      } catch (error) {
        console.error(`Failed to fetch details for event ${event.id}:`, error);
      }
      
      // Fallback: create a basic event structure
      return {
        ...event,
        eventData: {
          event_id: event.id,
          event_title: event.title,
          event_date: event.event_date,
          event_time: event.event_time || event.time,
          thumbnail: event.image,
          picture: event.image,
          address: apiData.data.address || '',
          ticket_price: 0,
          description: event.description || '',
          interested_count: 0,
          going_count: 0,
          location_id: apiData.data.id,
          event_type: 'location',
          public_private: 'public',
        }
      };
    })
  );

  return Response.json({ location, apiData, transformedEvents: eventsWithDetails, translations });
}

export const meta: MetaFunction<typeof loader> = ({ data }) => {
  if (!data) {
    return [
      { title: "Location Not Found | Nomadics" },
      { name: "description", content: "The requested location was not found." },
    ];
  }

  // Use default English for meta tags since we can't access hooks in meta
  const locationData = data.translations.en[data.location];
  
  return [
    { title: locationData?.meta?.title || `${data.location} | Nomadics` },
    { name: "description", content: locationData?.meta?.description || `Explore Nomadics studio in ${data.location}` },
  ];
};

export default function LocationPage() {
  const { location, apiData, transformedEvents, translations } = useLoaderData<typeof loader>();
  const { currentLanguage } = useLanguage();
  
  // Get location data based on current language
  const locationData = translations[currentLanguage]?.[location] || translations.en[location];
  
  // Set background image from API
  if (locationData && apiData.data.banner_picture) {
    locationData.welcome.backgroundImage = apiData.data.banner_picture;
  }

  if (!locationData) {
    throw new Response("Location Not Found", { status: 404 });
  }

  const navigate = useNavigate();

  // Transform studios data with onClick handlers
  const studiosData = {
    sectionTitle: locationData.studios?.sectionTitle || "Our Studios",
    studios: (apiData.data.studios || []).map((studio: any) => {
      const slug = studio.studio_slug || studio.slug || `studio-${studio.id}`;
      const studioName = studio.studio_name || studio.name || studio.title || `Studio ${studio.id}`;
      
      return {
        id: studio.id,
        image: studio.image,
        title: studioName,
        studio_slug: slug,
        slug: slug,
      };
    })
  };

  return (
    <div>
      <WelcomeSection data={locationData.welcome} />
      <LocationInfo data={locationData.info} />
      <StudiosSection data={studiosData} />
      {apiData.data.resources?.length > 0 && (
        <PeopleSection
          title={locationData.people.title}
          eyebrow="MEET THE TEAM"
          buttonText="VIEW ALL PEOPLE"
          buttonHref="/people"
        />
      )}
      {transformedEvents.length > 0 && (
        <EventsSection
          title={locationData.events.sectionTitle}
          data={transformedEvents}
        />
      )}
      <PlanYourVisitSection
        title={locationData.planVisit.title}
        address={apiData.data.address}
        hours={apiData.data.opening_hours}
        phone={apiData.data.phone}
        email={apiData.data.email}
        additionalInfo={apiData.data.additional_info}
        location={apiData.data.location}
      />
      <MembershipSection theme="black" />
    </div>
  );
}
