import React, { useState, useEffect } from "react";
import type { MetaFunction } from "@remix-run/node";
import { Link, useParams, useLocation } from "@remix-run/react";
import { Button } from "~/components/ui/button";
import { PublicRoute } from "~/components/RouteProtection";
import { newsAPI, type News } from "~/lib/api";
import { useLanguage } from "~/hooks/useLanguage";

// CSS for line clamping
const styles = `
  .line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  .line-clamp-3 {
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
`;

export const meta: MetaFunction = ({ params }) => {
  return [
    { title: `${params.slug?.replace(/-/g, ' ')} - Nomadics Studios` },
    {
      name: "description",
      content: "Read the latest news and insights from the creative community at Nomadics Studios.",
    },
  ];
};

// Mock articles data - this would typically come from your API
const newsArticles = [
  {
    id: 1,
    title: "Travel as a failure",
    category: "Travel",
    author: {
      name: "Akanksha Bhartaure",
      avatar: "/team-member.png",
      bio: "Travel Writer & Creative Director"
    },
    publishedAt: "5 min ago",
    readTime: "5 min read",
    image: "/bw (1).png",
    slug: "travel-as-a-failure",
    excerpt: "A story about a trip to Barcelona and the lessons learned from unexpected challenges along the way.",
    content: `
      <h2>A story about travel</h2>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <h3>A story about a trip to Barcelona 2</h3>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <p>Travel opens our minds to new possibilities and perspectives that we might never encounter in our daily routines. When we step outside our comfort zones and immerse ourselves in different cultures, we gain invaluable insights that can transform our creative work.</p>
      
      <p>Sometimes, the most valuable lessons come from the moments when things don't go according to plan. Failed connections, missed opportunities, and unexpected detours often become the foundation for our most compelling stories and creative breakthroughs.</p>
    `,
    tags: ["Travel", "Lessons", "Barcelona", "Growth"]
  },
  {
    id: 2,
    title: "Travel as a fresh perspective",
    category: "Travel",
    author: {
      name: "Super Admin",
      avatar: "/team-member.png",
      bio: "Creative Director at Nomadics Studios"
    },
    publishedAt: "2 hours ago",
    readTime: "4 min read",
    image: "/image.png",
    slug: "travel-as-a-fresh-perspective",
    excerpt: "A story about travel and how it opens our minds to new creative possibilities and fresh perspectives.",
    content: `
      <h2>A story about travel</h2>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <h3>The Creative Impact of Travel</h3>
      
      <p>For creators, travel is more than just a break from routine—it's a source of inspiration that can revolutionize their approach to their craft. Whether you're a filmmaker seeking new locations, a photographer hunting for unique lighting, or a musician searching for fresh sounds, travel provides the raw material for creative breakthroughs.</p>
      
      <p>At Nomadics Studios, we've seen firsthand how our community members return from their travels with renewed energy and innovative ideas. The stories they share, the techniques they've learned, and the perspectives they've gained all contribute to the rich tapestry of our creative community.</p>
    `,
    tags: ["Travel", "Creativity", "Inspiration", "Community"]
  }
];

// Function to find article by slug
function findArticleBySlug(slug: string) {
  return newsArticles.find(article => article.slug === slug) || newsArticles[0];
}

// Related articles - get other articles from the main articles array
function getRelatedArticles(currentSlug: string) {
  return newsArticles
    .filter(article => article.slug !== currentSlug)
    .slice(0, 3);
}

export default function NewsDetailsPage() {
  const { slug } = useParams();
  const location = useLocation();
  const { t } = useLanguage();
  const [article, setArticle] = useState<News | null>(null);
  const [relatedArticles, setRelatedArticles] = useState<News[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Generate slug from title (same as in news.tsx)
  const generateSlug = (title: string, id: number) => {
    return `${title.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '')}-${id}`;
  };

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        setError(null);
        
        if (slug) {
          let foundArticle = null;
          
          // First try to get article by ID if slug ends with a number
          const idMatch = slug.match(/-(\d+)$/);
          if (idMatch) {
            const articleId = parseInt(idMatch[1]);
            try {
              const articleResponse = await newsAPI.getById(articleId);
              if (articleResponse.status && articleResponse.data) {
                foundArticle = articleResponse.data;
              }
            } catch (error) {
              console.log('Failed to fetch by ID, trying slug method');
            }
          }
          
          // If not found by ID, try by slug
          if (!foundArticle) {
            try {
              const slugResponse = await newsAPI.getBySlug(slug);
              if (slugResponse.status && slugResponse.data) {
                foundArticle = slugResponse.data;
              }
            } catch (error) {
              console.log('Failed to fetch by slug, trying to find in all news');
            }
          }
          
          // Fallback: fetch all news and find by matching criteria
          if (!foundArticle) {
            const allNewsResponse = await newsAPI.getAll();
            if (allNewsResponse.status && allNewsResponse.data) {
              foundArticle = allNewsResponse.data.find(item => {
                const itemSlug = item.name.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
                return itemSlug === slug || `${itemSlug}-${item.id}` === slug;
              });
            }
          }
          
          if (foundArticle) {
            setArticle(foundArticle);
            
            // Fetch all news for related articles (exclude current article)
            const allNewsResponse = await newsAPI.getAll();
            if (allNewsResponse.status && allNewsResponse.data) {
              const related = allNewsResponse.data.filter(item => item.id !== foundArticle.id).slice(0, 3);
              setRelatedArticles(related);
            }
          } else {
            setError('Article not found');
          }
        } else {
          setError('No article slug provided');
        }
      } catch (err) {
        setError('Failed to fetch article');
        console.error('Error fetching article:', err);
      } finally {
        setLoading(false);
      }
    };

    // Reset state when slug changes
    setArticle(null);
    setRelatedArticles([]);
    
    // Scroll to top when navigating to new article
    window.scrollTo({ top: 0, behavior: 'smooth' });
    
    fetchData();
  }, [slug]); // Only depend on slug, not location.state

  // Format date for display
  const formatNewsDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric'
    });
  };

  if (loading) {
    return (
      <PublicRoute>
        <div className="bg-white min-h-screen">
          <div className="flex justify-center items-center h-64">
            <div className="text-gray-500">{t("news.loading", "Loading article...")}</div>
          </div>
        </div>
      </PublicRoute>
    );
  }

  if (error || !article) {
    return (
      <PublicRoute>
        <div className="bg-white min-h-screen">
          <div className="flex justify-center items-center h-64">
            <div className="text-red-500">{t("news.error", "Error")}: {error || 'Article not found'}</div>
          </div>
        </div>
      </PublicRoute>
    );
  }
  
  return (
    <PublicRoute>
      <div className="relative bg-white min-h-screen">
        <style dangerouslySetInnerHTML={{ __html: styles }} />
        
        {/* Hero Section with Background Image */}
        <section className="relative h-[50vh] lg:h-[55vh] overflow-hidden">
          {/* Background Image */}
          <div className="absolute inset-0">
            <img
              src={article.banner || article.thumbnail}
              alt={article.name}
              className="h-full w-full object-cover"
              onError={(e) => {
                // Try the other image source if one fails
                const currentSrc = e.currentTarget.src;
                if (currentSrc === article.banner && article.thumbnail) {
                  e.currentTarget.src = article.thumbnail;
                } else if (currentSrc === article.thumbnail && article.banner) {
                  e.currentTarget.src = article.banner;
                } else {
                  e.currentTarget.src = "/bw.png";
                }
              }}
            />
            <div className="absolute inset-0 bg-gradient-to-b from-black/30 via-black/50 to-black/70" />
          </div>

          {/* Hero Content */}
          <div className="relative z-10 h-full flex items-end">
            <div className="mx-auto w-full max-w-[1000px] px-4 lg:px-8 pb-12 lg:pb-16">
              {/* Breadcrumb */}
              <nav className="mb-6">
                <Link 
                  to="/news" 
                  className="text-sm text-white/80 hover:text-white transition-colors uppercase tracking-wider flex items-center gap-2"
                >
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
                  </svg>
                  Back to News
                </Link>
              </nav>

              {/* Category */}
              <div className="mb-4">
                <span className="inline-block px-4 py-2 bg-white/20 backdrop-blur-sm text-white text-sm font-medium uppercase tracking-wider rounded">
                  {article.tags?.split(',')[0] || 'News'}
                </span>
              </div>

              {/* Title */}
              <h1 className="text-3xl lg:text-5xl font-bold text-white mb-6 leading-tight max-w-4xl">
                {article.name}
              </h1>

              {/* Meta Info */}
              <div className="flex flex-col sm:flex-row sm:items-center gap-4 text-white/90">
                <div className="flex items-center gap-3">
                  <img
                    src="/team-member.png"
                    alt={article.created_by.fname}
                    className="w-10 h-10 rounded-full object-cover bg-white/20"
                    onError={(e) => {
                      e.currentTarget.src = "/team-member.png";
                    }}
                  />
                  <div>
                    <div className="font-medium text-white">{article.created_by.fname} {article.created_by.lname}</div>
                    <div className="text-sm text-white/70">{formatNewsDate(article.created_at)}</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </section>

        {/* Article Content Section */}
        <section className="bg-white py-12 lg:py-16">
          <div className="mx-auto w-full max-w-[800px] px-4 lg:px-8">
            
            {/* Article Body */}
            <div className="prose prose-lg max-w-none mb-16">
              <div className="article-content">
                {/* Intro Section */}
                {article.heading_intro && (
                  <h2 className="text-heading-4 font-bold text-black mb-4 !text-black">{article.heading_intro}</h2>
                )}
                {article.para_intro && (
                  <div 
                    className="text-black/80 text-body-3 lg:text-body-2 leading-relaxed mb-8 !text-black"
                    dangerouslySetInnerHTML={{ __html: article.para_intro }}
                  />
                )}
                
                {/* Elaboration Section */}
                {article.heading_elaboration && (
                  <h2 className="text-heading-4 font-bold text-black mb-4 mt-8 !text-black">{article.heading_elaboration}</h2>
                )}
                {article.para_elaboration && (
                  <div 
                    className="text-black/80 text-body-3 lg:text-body-2 leading-relaxed mb-8 !text-black"
                    dangerouslySetInnerHTML={{ __html: article.para_elaboration }}
                  />
                )}
                
                {/* Conclusion Picture */}
                {article.conclusion_picture && (
                  <img 
                    src={article.conclusion_picture} 
                    alt="Article conclusion"
                    className="rounded-lg shadow-lg my-8 w-full"
                    onError={(e) => {
                      e.currentTarget.src = "/bw.png";
                    }}
                  />
                )}
                
                {/* Conclusion Section */}
                {article.heading_conclusion && (
                  <h2 className="text-2xl font-bold text-black mb-4 mt-8">{article.heading_conclusion}</h2>
                )}
                {article.para_conclusion && (
                  <div 
                    className="text-black/80 text-base lg:text-lg leading-relaxed mb-8"
                    dangerouslySetInnerHTML={{ __html: article.para_conclusion }}
                  />
                )}
              </div>
            </div>

            {/* Author Bio */}
            {/* <div className="bg-gray-50 rounded-2xl p-8 mb-12">
              <div className="flex items-start gap-4">
                <img
                  src="/team-member.png"
                  alt={article.created_by.fname}
                  className="w-16 h-16 rounded-full object-cover bg-gray-200"
                  onError={(e) => {
                    e.currentTarget.src = "/team-member.png";
                  }}
                />
                <div>
                  <h3 className="text-xl font-bold text-black mb-2">{article.created_by.fname} {article.created_by.lname}</h3>
                  <p className="text-gray-600 mb-4">Content Creator</p>
                </div>
              </div>
            </div> */}

            {/* Tags */}
            {/* <div className="flex flex-wrap gap-3 mb-12 pb-8 border-b border-gray-200">
              <span className="text-sm font-medium text-gray-600 mr-2">Tags:</span>
              {article.tags?.split(',').filter(tag => tag.trim()).map((tag: string, index: number) => (
                <span
                  key={index}
                  className="px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-full hover:bg-gray-200 transition-colors cursor-pointer"
                >
                  {tag.trim()}
                </span>
              ))}
            </div> */}



            {/* Share Section */}
            {/* <div className="text-center mb-12">
              <h3 className="text-xl font-bold text-black mb-4">Share this article</h3>
              <div className="flex justify-center gap-3">
                <Button variant="outline-black" size="md" className="bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300">Twitter</Button>
                <Button variant="outline-black" size="md" className="bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300">LinkedIn</Button>
                <Button variant="outline-black" size="md" className="bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300">Facebook</Button>
                <Button variant="fill-black" size="md" arrowRight>
                  Subscribe
                </Button>
              </div>
            </div> */}
          </div>
        </section>

        {/* Related News Section */}
        <section className="bg-gray-50">
          <div className="mx-auto w-full max-w-[1200px] px-4 lg:px-8 py-12 lg:py-16">
            <div className="text-center mb-16">
              <h2 className="text-3xl lg:text-4xl font-bold text-black mb-4">
                Related news
              </h2>
              <div className="w-24 h-1 bg-black mx-auto"></div>
            </div>
            
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8 mb-12">
              {relatedArticles.map((relatedArticle) => {
                const relatedSlug = generateSlug(relatedArticle.name, relatedArticle.id);
                return (
                  <Link
                    key={relatedArticle.id}
                    to={`/news-details/${relatedSlug}`}
                    className="group block bg-white shadow-sm hover:shadow-lg transition-all duration-300 overflow-hidden hover:transform hover:scale-[1.02] rounded-lg"
                    replace={false}
                  >
                  <div className="aspect-[4/3] overflow-hidden bg-neutral-100">
                    <img
                      src={relatedArticle.banner || relatedArticle.thumbnail}
                      alt={relatedArticle.name}
                      className="h-full w-full object-cover transition-transform duration-500 group-hover:scale-105"
                      onError={(e) => {
                        e.currentTarget.src = "/image.png";
                      }}
                    />
                  </div>
                  <div className="p-6">
                    <div className="inline-block px-3 py-1 bg-gray-100 text-gray-600 text-xs font-medium uppercase tracking-wider rounded mb-3">
                      {relatedArticle.tags?.split(',')[0] || 'News'}
                    </div>
                    <h3 className="text-lg font-bold text-black mb-3 group-hover:text-black/80 transition-colors leading-tight">
                      {relatedArticle.name}
                    </h3>
                    <p className="text-gray-600 text-sm line-clamp-2 mb-4">
                      {relatedArticle.heading_intro.substring(0, 100)}...
                    </p>
                    <div className="flex items-center gap-2 text-xs text-gray-500">
                      <span>{relatedArticle.created_by.fname} {relatedArticle.created_by.lname}</span>
                      <span>•</span>
                      <span>{formatNewsDate(relatedArticle.created_at)}</span>
                    </div>
                  </div>
                </Link>
                );
              })}
            </div>

            <div className="text-center">
              <Button
                variant="outline-black"
                size="lg"
                arrowRight
                className="text-black"
                onClick={() => window.location.href = '/news'}
              >
                View All News
              </Button>
            </div>
          </div>
        </section>
      </div>
    </PublicRoute>
  );
}