import React, { useState, useEffect, useMemo } from "react";
import type { MetaFunction } from "@remix-run/node";
import { Link } from "@remix-run/react";
import { Button } from "~/components/ui/button";
import { SectionHeader } from "~/components/shared/SectionHeader";
import { PublicRoute } from "~/components/RouteProtection";
import { useLanguage } from "~/hooks/useLanguage";
import { newsAPI, type News, API_BASE_URL } from "~/lib/api";

interface Location {
  id: number;
  location: string;
  type: string | null;
  coming_soon: number;
  banner_picture: string | null;
  description: string | null;
}

// CSS for line clamping and input styling
const styles = `
  .line-clamp-3 {
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  .line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  
  /* Custom date input styling */
  input[type="date"]::-webkit-calendar-picker-indicator {
    opacity: 0;
    position: absolute;
    right: 8px;
    width: 20px;
    height: 20px;
    cursor: pointer;
  }
  
  input[type="date"] {
    color-scheme: light;
  }
`;

export const meta: MetaFunction = () => {
  return [
    { title: "News - Nomadics Studios" },
    {
      name: "description",
      content:
        "Stay updated with the latest news, insights, and stories from the creative community at Nomadics Studios.",
    },
  ];
};

interface NewsCardProps {
  article: News;
  featured?: boolean;
  formatNewsDate?: (dateString: string) => string;
  generateSlug?: (title: string, id: number) => string;
}

function NewsCard({
  article,
  featured = false,
  formatNewsDate,
  generateSlug,
}: NewsCardProps) {
  const slug = generateSlug
    ? generateSlug(article.name, article.id)
    : `news-${article.id}`;
  const formattedDate = formatNewsDate
    ? formatNewsDate(article.created_at)
    : article.created_at;

  // Format date like event cards (day and month)
  const formatCardDate = (dateString: string) => {
    const date = new Date(dateString);
    if (isNaN(date.getTime())) {
      return { day: "TBD", month: "DATE" };
    }
    const day = date.getDate().toString();
    const month = date
      .toLocaleDateString("en-US", { month: "short" })
      .toUpperCase();
    return { day, month };
  };

  const { day, month } = formatCardDate(article.created_at);
  const author = article.created_by
    ? `${article.created_by.fname} ${article.created_by.lname}`
    : "Unknown Author";

  return (
    <Link
      to={`/news-details/${slug}`}
      state={{ article }}
      className="group block bg-white border border-gray-200 overflow-hidden hover:shadow-lg transition-all duration-300"
    >
      {/* News Image */}
      <div className="relative overflow-hidden aspect-[426.67/240]">
        <img
          src={article.banner || article.thumbnail}
          alt={article.name}
          className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
          onError={(e) => {
            e.stopPropagation();
            const currentSrc = e.currentTarget.src;
            if (currentSrc === article.banner && article.thumbnail) {
              e.currentTarget.src = article.thumbnail;
            } else if (currentSrc === article.thumbnail && article.banner) {
              e.currentTarget.src = article.banner;
            } else {
              e.currentTarget.src = "/bw.png";
            }
          }}
        />
      </div>

      {/* News Details */}
      <div className="flex items-start gap-4 p-4 bg-white">
        {/* Date Box */}
        <div className="text-center min-w-[48px] flex-shrink-0">
          <div
            className="text-[24px] font-semibold leading-tight"
            style={{ color: "#000000" }}
          >
            {day}
          </div>
          <div
            className="text-[14px] font-normal uppercase"
            style={{ color: "#000000" }}
          >
            {month}
          </div>
        </div>

        {/* News Info */}
        <div className="flex-1 min-w-0">
          <h3
            className="text-body-2 font-normal mb-1 line-clamp-2"
            style={{ color: "#000000" }}
          >
            {article.name}
          </h3>
          <p
            className="text-body-3 mb-1 line-clamp-2"
            style={{ color: "#000000" }}
          >
            {article.heading_intro}
          </p>
          <p className="text-body-3" style={{ color: "#000000" }}>
            {author}
          </p>
        </div>
      </div>
    </Link>
  );
}

export default function NewsPage() {
  const { t } = useLanguage();
  const [newsArticles, setNewsArticles] = useState<News[]>([]);
  const [selectedDate, setSelectedDate] = useState("");
  const [searchTags, setSearchTags] = useState("");
  const [selectedLocation, setSelectedLocation] = useState("");
  const [locations, setLocations] = useState<Location[]>([]);
  const [debouncedSearchTags, setDebouncedSearchTags] = useState("");
  const [loading, setLoading] = useState(true);
  const [locationsLoading, setLocationsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Format date for display
  const formatNewsDate = (dateString: string) => {
    const date = new Date(dateString);
    const now = new Date();
    const diffTime = Math.abs(now.getTime() - date.getTime());
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

    if (diffDays === 1) return "1 day ago";
    if (diffDays < 7) return `${diffDays} days ago`;
    if (diffDays < 30) return `${Math.ceil(diffDays / 7)} weeks ago`;
    return date.toLocaleDateString();
  };

  // Generate slug from title
  const generateSlug = (title: string, id: number) => {
    return `${title
      .toLowerCase()
      .replace(/[^a-z0-9]+/g, "-")
      .replace(/(^-|-$)/g, "")}-${id}`;
  };

  // Debounce search tags
  useEffect(() => {
    const timer = setTimeout(() => {
      setDebouncedSearchTags(searchTags);
    }, 1000); // 500ms delay

    return () => clearTimeout(timer);
  }, [searchTags]);

  // Fetch locations on mount
  useEffect(() => {
    const fetchLocations = async () => {
      try {
        setLocationsLoading(true);
        const token =
          localStorage.getItem("authToken") ||
          localStorage.getItem("auth_token") ||
          localStorage.getItem("token");

        if (!token) {
          console.error("No auth token found");
          setLocations([]);
          setLocationsLoading(false);
          return;
        }

        const response = await fetch(`${API_BASE_URL}/locations`, {
          method: "GET",
          headers: {
            Authorization: `Bearer ${token}`,
            Accept: "application/json",
          },
        });

        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();

        if (result.status && Array.isArray(result.data)) {
          // Remove duplicate locations by id
          const uniqueLocations = Array.from(
            new Map(result.data.map((loc: Location) => [loc.id, loc])).values()
          ) as Location[];
          setLocations(uniqueLocations);
        } else {
          console.error("Unexpected response format:", result);
          setLocations([]);
        }
      } catch (error) {
        console.error("Failed to fetch locations:", error);
        setLocations([]);
      } finally {
        setLocationsLoading(false);
      }
    };

    fetchLocations();
  }, []);

  useEffect(() => {
    const fetchNews = async () => {
      try {
        setLoading(true);
        const filterDate = selectedDate || undefined;
        const filterTags = debouncedSearchTags.trim() || undefined;
        const filterLocation = selectedLocation
          ? parseInt(selectedLocation, 10)
          : undefined;

        const response = await newsAPI.getAll(
          filterDate,
          filterTags,
          filterLocation
        );

        if (response.status && response.data) {
          setNewsArticles(response.data);
        } else {
          setError("Failed to fetch news");
        }
      } catch (err) {
        setError("An error occurred while fetching news");
        console.error("Error fetching news:", err);
      } finally {
        setLoading(false);
      }
    };

    fetchNews();
  }, [selectedDate, debouncedSearchTags, selectedLocation]);

  // Filter articles based on search criteria (this is now done server-side via API)
  const filteredArticles = useMemo(() => {
    return newsArticles; // API already handles filtering
  }, [newsArticles]);

  return (
    <PublicRoute>
      <div className="bg-white min-h-screen">
        <style dangerouslySetInnerHTML={{ __html: styles }} />

        {loading ? (
          <div className="flex justify-center items-center h-64">
            <div className="text-gray-500">
              {t("news.loading", "Loading news...")}
            </div>
          </div>
        ) : error ? (
          <div className="flex justify-center items-center h-64">
            <div className="text-red-500">
              {t("news.error", "Error")}: {error}
            </div>
          </div>
        ) : (
          <>
            {/* Hero Section with Background Image */}
            <section className="relative h-[30vh] lg:h-[35vh] overflow-hidden">
              {/* Background Image */}
              <div className="absolute inset-0">
                <img
                  src="/bw.png"
                  alt="Discover our news"
                  className="h-full w-full object-cover"
                />
                <div className="absolute inset-0 bg-gradient-to-b from-black/20 via-black/40 to-black/60" />
              </div>

              {/* Hero Content */}
              <div className="relative z-10 h-full flex items-end">
                <div className="max-w-[1400px] mx-auto px-4 lg:px-14 pb-12 lg:pb-16">
                  <div className="max-w-4xl">
                    <h1 className="text-[32px] leading-[1.1] md:text-[56px] lg:text-heading-1 2xl:text-[80px] uppercase font-bold tracking-[-0.02em] text-white mb-6">
                      Discover our news
                    </h1>

                    {/* Search and Filter Bar */}
                    <div className="flex flex-col sm:flex-row gap-4 mt-8 mb-4">
                      <div className="flex-1 relative">
                        <select
                          value={selectedLocation}
                          onChange={(e) => setSelectedLocation(e.target.value)}
                          className="w-full px-4 py-3 bg-white backdrop-blur-sm border border-white/20 text-black focus:outline-none focus:bg-white focus:border-gray-300 text-sm rounded-lg transition-all appearance-none"
                          disabled={locationsLoading}
                        >
                          <option value="">All Locations</option>
                          {locations.map((location) => (
                            <option key={location.id} value={location.id}>
                              {location.location}
                            </option>
                          ))}
                        </select>
                        <div className="absolute right-3 top-1/2 transform -translate-y-1/2 pointer-events-none">
                          <svg
                            className="w-4 h-4 text-gray-500"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"
                            />
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"
                            />
                          </svg>
                        </div>
                      </div>
                      <div className="flex-1 relative">
                        <input
                          type="date"
                          value={selectedDate}
                          onChange={(e) => setSelectedDate(e.target.value)}
                          placeholder="mm/dd/yyyy"
                          className="w-full px-4 py-3 bg-white backdrop-blur-sm border border-white/20 text-black placeholder:text-gray-500 focus:outline-none focus:bg-white focus:border-gray-300 text-sm rounded-lg transition-all"
                        />
                        <div className="absolute right-3 top-1/2 transform -translate-y-1/2 pointer-events-none">
                          <svg
                            className="w-4 h-4 text-gray-500"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"
                            />
                          </svg>
                        </div>
                      </div>
                      <div className="flex-1 relative">
                        <input
                          type="text"
                          value={searchTags}
                          onChange={(e) => setSearchTags(e.target.value)}
                          placeholder="Search By Tags"
                          className="w-full px-4 py-3 bg-white backdrop-blur-sm border border-white/20 text-black placeholder:text-gray-500 focus:outline-none focus:bg-white focus:border-gray-300 text-sm rounded-lg transition-all"
                        />
                        <div className="absolute right-3 top-1/2 transform -translate-y-1/2 pointer-events-none">
                          <svg
                            className="w-4 h-4 text-gray-500"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
                            />
                          </svg>
                        </div>
                      </div>
                    </div>

                    {/* Search Results Info */}
                    {(debouncedSearchTags ||
                      selectedDate ||
                      selectedLocation) && (
                      <div className="mt-4 text-white/80 text-sm">
                        {filteredArticles.length === 0
                          ? "No news found matching your search criteria"
                          : `Showing ${filteredArticles.length} news ${
                              filteredArticles.length !== 1 ? "items" : "item"
                            }`}
                        {debouncedSearchTags && (
                          <span className="ml-2">
                            for "{debouncedSearchTags}"
                          </span>
                        )}
                        {(debouncedSearchTags ||
                          selectedDate ||
                          selectedLocation) && (
                          <button
                            onClick={() => {
                              setSearchTags("");
                              setDebouncedSearchTags("");
                              setSelectedDate("");
                              setSelectedLocation("");
                            }}
                            className="ml-3 text-white/60 hover:text-white underline"
                          >
                            Clear filters
                          </button>
                        )}
                      </div>
                    )}
                  </div>
                </div>
              </div>
            </section>

            {/* News Grid Section */}
            <section className="bg-neutral-50 py-12 lg:py-16">
              <div className="max-w-[1400px] mx-auto px-4 lg:px-14">
                {/* News Grid */}
                {filteredArticles.length > 0 ? (
                  <>
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-16">
                      {filteredArticles.map((article) => (
                        <NewsCard
                          key={article.id}
                          article={article}
                          formatNewsDate={formatNewsDate}
                          generateSlug={generateSlug}
                        />
                      ))}
                    </div>

                    {/* Load More - only show if not filtered and there are more news */}
                    {!debouncedSearchTags &&
                      !selectedDate &&
                      !selectedLocation &&
                      filteredArticles.length >= 4 && (
                        <div className="text-center">
                          <Button
                            variant="outline-black"
                            size="lg"
                            arrowRight
                            className="text-black"
                          >
                            Load More News
                          </Button>
                        </div>
                      )}
                  </>
                ) : (
                  /* No Results State */
                  <div className="text-center py-16">
                    <div className="max-w-md mx-auto">
                      <svg
                        className="w-16 h-16 text-gray-300 mx-auto mb-6"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={1}
                          d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                        />
                      </svg>
                      <h3 className="text-xl font-semibold text-gray-700 mb-2">
                        No news found
                      </h3>
                      <p className="text-gray-500 mb-6">
                        Try adjusting your search criteria or browse all news
                      </p>
                      <Button
                        variant="outline-black"
                        size="md"
                        className="bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300"
                        onClick={() => {
                          setSearchTags("");
                          setDebouncedSearchTags("");
                          setSelectedDate("");
                          setSelectedLocation("");
                        }}
                      >
                        Show All News
                      </Button>
                    </div>
                  </div>
                )}
              </div>
            </section>
          </>
        )}
      </div>
    </PublicRoute>
  );
}
