import type { MetaFunction, LoaderFunctionArgs } from "@remix-run/node";
import { useLoaderData, useNavigate } from "@remix-run/react";
import { useEffect, useState } from "react";
import { API_BASE_URL, IMAGE_BASE_URL } from "~/lib/api";

export async function loader({ params, request }: LoaderFunctionArgs) {
  const slug = params.slug;

  if (!slug) {
    throw new Response("Studio Not Found", { status: 404 });
  }

  // Return just the slug, we'll fetch data client-side
  return Response.json({ slug });
}

export const meta: MetaFunction<typeof loader> = ({ data }) => {
  if (!data) {
    return [
      { title: "Studio Not Found | Nomadic Studios" },
      { name: "description", content: "The requested studio was not found." },
    ];
  }

  return [
    { title: `Studio Details | Nomadic Studios` },
    { name: "description", content: "Explore our studio at Nomadic Studios" },
  ];
};

export default function StudioDetailPage() {
  const { slug } = useLoaderData<typeof loader>();
  const navigate = useNavigate();
  const [studioData, setStudioData] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchStudioData = async () => {
      try {
        const token =
          localStorage.getItem("authToken") ||
          localStorage.getItem("auth_token") ||
          localStorage.getItem("token") ||
          "";

        console.log("Fetching studio with slug:", slug);
        console.log("Using token:", token ? "Token found" : "No token");

        // Use the correct API endpoint
        const apiUrl = `https://dashboard.nomadicstudios.net/api/studio/${slug}`;
        console.log("API URL:", apiUrl);

        const response = await fetch(apiUrl, {
          headers: {
            Authorization: `Bearer ${token}`,
            Accept: "application/json",
          },
        });

        console.log("Response status:", response.status);

        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const data = await response.json();
        console.log("API Response:", data);

        if (!data.status) {
          throw new Error("Invalid response from server");
        }

        setStudioData(data.data);
      } catch (err) {
        console.error("Error fetching studio data:", err);
        setError("Failed to load studio details");
      } finally {
        setLoading(false);
      }
    };

    fetchStudioData();
  }, [slug]);

  if (loading) {
    return (
      <div className="min-h-screen bg-[#121212] flex items-center justify-center">
        <div className="text-center">
          <div className="w-12 h-12 border-4 border-white/20 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
          <p className="text-white/60 text-lg">Loading studio details...</p>
        </div>
      </div>
    );
  }

  if (error || !studioData) {
    return (
      <div className="min-h-screen bg-[#121212] flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-white text-4xl font-bold mb-4">500</h1>
          <p className="text-white/60 text-lg">
            {error || "Failed to load studio"}
          </p>
        </div>
      </div>
    );
  }

  const { lp, crews = [], resources = [], equipments = [] } = studioData;

  const bannerUrl = lp.banner_picture
    ? `${IMAGE_BASE_URL}/images/studios/${lp.banner_picture}`
    : "/images/studios/default-banner.jpg";

  return (
    <div className="min-h-screen bg-[#121212]">
      {/* Hero Banner with Title Overlay */}
      <section className="relative w-full h-[60vh] min-h-[500px]">
        <div
          className="absolute inset-0 bg-cover bg-center"
          style={{ backgroundImage: `url(${bannerUrl})` }}
        >
          <div className="absolute inset-0 bg-black/40" />
        </div>

        <div className="relative z-10 h-full flex items-center justify-start px-8 lg:px-20">
          <div className="max-w-4xl">
            <h1 className="text-white text-heading-2 lg:text-heading-1 2xl:text-[120px] font-medium uppercase tracking-tight font-clash-grotesk mb-2 !text-black">
              {lp.location?.location || "Location"}
            </h1>
            <h2 className="text-white text-heading-3 lg:text-heading-2 2xl:text-[100px] font-light tracking-tight font-clash-grotesk !text-black">
              {lp.studio?.name || "Studio"}
            </h2>
          </div>
        </div>
      </section>

      {/* Description Section */}
      {lp.description && (
        <section className="w-full py-16 px-8 lg:px-20">
          <div className="max-w-6xl">
            <div
              className="text-black/80 text-body-2 lg:text-body-1 2xl:text-[28px] leading-relaxed"
              dangerouslySetInnerHTML={{ __html: lp.description }}
            />
          </div>
        </section>
      )}

      {/* Gallery Section */}
      {lp.gallery && lp.gallery.length > 0 && (
        <section className="w-full py-16 px-8 lg:px-20">
          <h3 className="text-white text-xs uppercase tracking-[0.2em] mb-8 font-medium">
            Thumbnail Slider trial version
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {lp.gallery.map((image: any, index: number) => {
              const imageUrl = image.image_url
                ? `${IMAGE_BASE_URL}/images/galleries/${image.image_url}`
                : "/images/studios/default-gallery.jpg";

              // Make first image larger
              const isFirst = index === 0;

              return (
                <div
                  key={image.id}
                  className={`relative overflow-hidden ${
                    isFirst ? "md:col-span-2 md:row-span-2" : ""
                  }`}
                >
                  <img
                    src={imageUrl}
                    alt={`Gallery image ${index + 1}`}
                    className="w-full h-full object-cover hover:scale-105 transition-transform duration-500"
                  />
                </div>
              );
            })}
          </div>
        </section>
      )}

      {/* Our Crew Section */}
      {resources && resources.length > 0 && (
        <section className="w-full py-16 px-8 lg:px-20 bg-[#1a1a1a]">
          <div className="max-w-7xl">
            <div className="flex flex-col lg:flex-row gap-12">
              {/* Left Side - Crew Types */}
              <div className="lg:w-1/3">
                <h3 className="text-white text-4xl lg:text-5xl font-medium mb-8 font-clash-grotesk">
                  Our crew
                </h3>
                <div className="space-y-4">
                  {crews.map((crew: any) => (
                    <p
                      key={crew.id}
                      className="text-white/50 text-lg lg:text-xl"
                    >
                      {crew.name}
                    </p>
                  ))}
                </div>
              </div>

              {/* Right Side - Crew Members */}
              <div className="lg:w-2/3 grid grid-cols-1 sm:grid-cols-2 gap-8">
                {resources.map((resource: any) => {
                  const profilePicUrl = resource.profile_pic
                    ? `${IMAGE_BASE_URL}/images/users/${resource.profile_pic}`
                    : "/images/users/default-profile.jpg";

                  return (
                    <div
                      key={resource.uuid}
                      onClick={() => navigate(`/profile/${resource.uuid}`)}
                      className="flex flex-col items-center cursor-pointer group"
                    >
                      <div className="w-full aspect-[3/4] overflow-hidden mb-4 shadow-lg">
                        <img
                          src={profilePicUrl}
                          alt={`${resource.fname} ${resource.lname}`}
                          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
                        />
                      </div>
                      <h4 className="text-white text-2xl font-medium mb-1 font-clash-grotesk">
                        {resource.fname} {resource.lname}
                      </h4>
                      <p className="text-white/60 text-sm">{resource.name}</p>
                    </div>
                  );
                })}
              </div>
            </div>
          </div>
        </section>
      )}

      {/* Equipment Section */}
      {equipments && equipments.length > 0 && (
        <section className="w-full py-16 px-8 lg:px-20">
          <div className="max-w-7xl">
            <h3 className="text-white text-4xl lg:text-5xl font-medium mb-12 font-clash-grotesk">
              We have all you need
              <br />
              Equipments
            </h3>
            <div className="flex flex-wrap gap-8 lg:gap-16">
              {equipments.map((equipment: any) => (
                <div key={equipment.id}>
                  <p className="text-white text-lg lg:text-xl">
                    {equipment.cat_name}
                  </p>
                </div>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* Book Now Button */}
      <section className="w-full py-16 px-8 lg:px-20 flex justify-center">
        <button
          onClick={() => navigate("/book-studio")}
          className="group relative inline-flex items-center gap-3 text-white text-lg lg:text-xl font-medium uppercase tracking-wider hover:gap-5 transition-all duration-300"
        >
          <span>Book Now</span>
          <svg
            width="20"
            height="20"
            viewBox="0 0 20 20"
            fill="none"
            className="group-hover:translate-x-2 transition-transform duration-300"
          >
            <path
              d="M4 10H16M16 10L11 5M16 10L11 15"
              stroke="currentColor"
              strokeWidth="2"
              strokeLinecap="round"
              strokeLinejoin="round"
            />
          </svg>
        </button>
      </section>
    </div>
  );
}
