import type { MetaFunction, LoaderFunctionArgs } from "@remix-run/node";
import { useLoaderData, useNavigate } from "@remix-run/react";
import { useEffect, useState } from "react";
import { API_BASE_URL, IMAGE_BASE_URL } from "~/lib/api";
import { Button } from "~/components/ui/button";

export async function loader({ params, request }: LoaderFunctionArgs) {
  const slug = params.slug;

  if (!slug) {
    throw new Response("Studio Not Found", { status: 404 });
  }

  // Return just the slug, we'll fetch data client-side
  return Response.json({ slug });
}

export const meta: MetaFunction<typeof loader> = ({ data }) => {
  if (!data) {
    return [
      { title: "Studio Not Found | Nomadic Studios" },
      { name: "description", content: "The requested studio was not found." },
    ];
  }

  return [
    { title: `Studio Details | Nomadic Studios` },
    { name: "description", content: "Explore our studio at Nomadic Studios" },
  ];
};

export default function StudioDetailPage() {
  const { slug } = useLoaderData<typeof loader>();
  const navigate = useNavigate();
  const [studioData, setStudioData] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [filters, setFilters] = useState({
    role: "",
  });

  useEffect(() => {
    const fetchStudioData = async () => {
      try {
        const token =
          localStorage.getItem("authToken") ||
          localStorage.getItem("auth_token") ||
          localStorage.getItem("token") ||
          "";

        console.log("Fetching studio with slug:", slug);
        console.log("Using token:", token ? "Token found" : "No token");

        // Use the API_BASE_URL from config
        const apiUrl = `${API_BASE_URL}/studio/${slug}`;
        console.log("API URL:", apiUrl);

        const response = await fetch(apiUrl, {
          headers: {
            Authorization: `Bearer ${token}`,
            Accept: "application/json",
          },
        });

        console.log("Response status:", response.status);

        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const data = await response.json();
        console.log("API Response:", data);

        if (!data.status) {
          throw new Error("Invalid response from server");
        }

        setStudioData(data.data);
      } catch (err) {
        console.error("Error fetching studio data:", err);
        setError("Failed to load studio details");
      } finally {
        setLoading(false);
      }
    };

    fetchStudioData();
  }, [slug]);

  if (loading) {
    return (
      <div className="min-h-screen bg-[#121212] flex items-center justify-center">
        <div className="text-center">
          <div className="w-12 h-12 border-4 border-white/20 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
          <p className="text-white/60 text-lg">Loading studio details...</p>
        </div>
      </div>
    );
  }

  if (error || !studioData) {
    return (
      <div className="min-h-screen bg-[#121212] flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-white text-4xl font-bold mb-4">500</h1>
          <p className="text-white/60 text-lg">
            {error || "Failed to load studio"}
          </p>
        </div>
      </div>
    );
  }

  const { lp, crews = [], resources = [], equipments = [] } = studioData;

  const bannerUrl = lp.banner_picture
    ? `${IMAGE_BASE_URL}/storage/location/${lp.banner_picture}`
    : "/images/studios/default-banner.jpg";

  // Get unique crew types for filtering
  const crewRoles = Array.from(new Set(crews.map((crew: any) => crew.name))).filter(Boolean);

  // Filter resources based on selected role
  const filteredResources = resources.filter((resource: any) => {
    if (!filters.role) return true;
    return resource.name === filters.role;
  });

  const handleRoleChange = (role: string) => {
    setFilters({
      role: filters.role === role ? "" : role,
    });
  };

  return (
    <div className="min-h-screen bg-white">
      {/* Hero Banner with Title Overlay */}
      <section className="relative w-full h-[60vh] min-h-[500px]">
        <div
          className="absolute inset-0 bg-cover bg-center"
          style={{ backgroundImage: `url(${bannerUrl})` }}
        >
          <div className="absolute inset-0 bg-black/40" />
        </div>

        <div className="relative z-10 h-full flex items-center justify-start px-8 lg:px-20">
          <div className="max-w-4xl">
            <h1 className="text-white text-5xl lg:text-7xl 2xl:text-8xl font-medium uppercase tracking-tight font-clash-grotesk mb-2">
              {lp.location?.location || "Location"}
            </h1>
            <h2 className="text-white text-4xl lg:text-6xl 2xl:text-7xl font-light tracking-tight font-clash-grotesk">
              {lp.studio?.name || "Studio"}
            </h2>
          </div>
        </div>
      </section>

      {/* Description Section */}
      {lp.description && (
        <section className="w-full py-16 px-8 lg:px-20">
          <div className="max-w-6xl">
            <div
              className="text-black/80 text-lg lg:text-xl 2xl:text-2xl leading-relaxed"
              dangerouslySetInnerHTML={{ __html: lp.description }}
            />
          </div>
        </section>
      )}

      {/* Gallery Section */}
      {lp.gallery && lp.gallery.length > 0 && (
        <section className="w-full py-16 px-8 lg:px-20">
          <h3 className="text-black text-xs uppercase tracking-[0.2em] mb-8 font-medium">
            {/* Thumbnail Slider trial version */}
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {lp.gallery.map((image: any, index: number) => {
              const imageUrl = image.image_url
                ? `${IMAGE_BASE_URL}/storage/location/${image.image_url}`
                : "/images/studios/default-gallery.jpg";

              // Make first image larger
              const isFirst = index === 0;

              return (
                <div
                  key={image.id}
                  className={`relative overflow-hidden ${
                    isFirst ? "md:col-span-2 md:row-span-2" : ""
                  }`}
                >
                  <img
                    src={imageUrl}
                    alt={`Gallery image ${index + 1}`}
                    className="w-full h-full object-cover hover:scale-105 transition-transform duration-500"
                  />
                </div>
              );
            })}
          </div>
        </section>
      )}

      {/* Our Crew Section */}
      {resources && resources.length > 0 && (
        <section className="w-full py-16 px-8 lg:px-20">
          <div className="max-w-7xl mx-auto">
            <div className="mb-12">
              <h3 className="text-black text-4xl lg:text-5xl font-medium mb-8 font-clash-grotesk">
                Our crew
              </h3>

              {/* Profession Filter */}
              <div className="mb-6">
                <label className="block mb-4 uppercase font-medium text-black">
                  PROFESSION
                </label>
                <div className="flex flex-wrap gap-3">
                  <button
                    className={`px-6 py-2.5 rounded-full font-normal transition-all text-sm lg:text-base ${
                      !filters.role
                        ? "bg-black border-2 border-black text-white"
                        : "bg-white border-2 border-black text-black hover:bg-gray-100"
                    }`}
                    onClick={() => handleRoleChange("")}
                  >
                    ALL CREW
                  </button>
                  {crewRoles.map((role: any) => (
                    <button
                      key={role}
                      className={`px-6 py-2.5 rounded-full font-normal transition-all text-sm lg:text-base ${
                        filters.role === role
                          ? "bg-black border-2 border-black text-white"
                          : "bg-white border-2 border-black text-black hover:bg-gray-100"
                      }`}
                      onClick={() => handleRoleChange(role)}
                    >
                      {role.toUpperCase()}
                    </button>
                  ))}
                </div>
              </div>
            </div>

            {/* Crew Members Grid */}
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3 md:gap-4 lg:gap-6">
              {filteredResources.length > 0 ? (
                filteredResources.map((resource: any) => {
                  const profilePicUrl = resource.profile_pic
                    ? `${IMAGE_BASE_URL}/storage/profile-pic/${resource.profile_pic}`
                    : "/images/users/default-profile.jpg";

                  return (
                    <div
                      key={resource.uuid}
                      onClick={() => navigate(`/profile/${resource.uuid}`)}
                      className="group block overflow-hidden transition-all duration-300 hover:scale-[1.02] cursor-pointer"
                    >
                      {/* Photo - Same aspect ratio as people cards */}
                      <div className="aspect-[313/419] overflow-hidden bg-neutral-100 mb-2 lg:mb-4">
                        <img
                          src={profilePicUrl}
                          alt={`${resource.fname} ${resource.lname}`}
                          className="h-full w-full object-cover transition-transform duration-500 group-hover:scale-105 grayscale group-hover:grayscale-0"
                          loading="lazy"
                        />
                      </div>
                      {/* Person Info */}
                      <div className="space-y-0.5 lg:space-y-1">
                        <h4 className="lg:text-heading-4 tracking-[-0.01em] uppercase leading-tight text-black font-medium">
                          {resource.fname} {resource.lname}
                        </h4>
                        <p className="text-button-small lg:text-body-3 uppercase text-black/60 font-normal">
                          {resource.name}
                        </p>
                      </div>
                    </div>
                  );
                })
              ) : (
                <div className="col-span-full flex justify-center items-center py-16">
                  <div className="text-gray-500">No crew members found matching your filter.</div>
                </div>
              )}
            </div>
          </div>
        </section>
      )}

      {/* Equipment Section */}
      {equipments && equipments.length > 0 && (
        <section className="w-full py-16 px-8 lg:px-20">
          <div className="max-w-7xl">
            <h3 className="text-black text-4xl lg:text-5xl font-medium mb-12 font-clash-grotesk">
              We have all you need
              <br />
              Equipments
            </h3>
            <div className="flex flex-wrap gap-8 lg:gap-16">
              {equipments.map((equipment: any) => (
                <div key={equipment.id}>
                  <p className="text-black text-lg lg:text-xl">
                    {equipment.cat_name}
                  </p>
                </div>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* Book Now Button */}
      <section className="w-full py-16 px-8 lg:px-20 flex justify-center">
        <Button
          variant="outline-black"
          size="lg"
          className="border-black/30 hover:border-black/50 uppercase tracking-wider font-medium !text-black"
          arrowRight
          onClick={() => navigate("/book-studio")}
        >
          Book Now
        </Button>
      </section>
    </div>
  );
}
