import type { MetaFunction, LoaderFunctionArgs } from "@remix-run/node";
import { useLoaderData, useNavigate } from "@remix-run/react";
import {
  WelcomeSection,
  StudiosSection,
  PhotoSection,
  BookSection,
} from "~/components/pages/studios";
import { MembershipSection } from "~/components/shared/MembershipSection";
import { PeopleSection } from "~/components/shared/PeopleSection";
import { useLanguage } from "~/hooks/useLanguage";
import { API_BASE_URL, IMAGE_BASE_URL } from "~/lib/api";
import type { ImageGridItem } from "~/components/shared/ImageGrid";

// Helper function to process URLs in translation data
const processTranslationUrls = (data: any): any => {
  if (typeof data === 'string') {
    // If it's just a path starting with /images, prepend IMAGE_BASE_URL
    if (data.startsWith('/images') && !data.includes('http')) {
      return IMAGE_BASE_URL + data;
    }
    return data;
  } else if (Array.isArray(data)) {
    return data.map(processTranslationUrls);
  } else if (data && typeof data === 'object') {
    const result: any = {};
    for (const [key, value] of Object.entries(data)) {
      result[key] = processTranslationUrls(value);
    }
    return result;
  }
  return data;
};

export async function loader({ params }: LoaderFunctionArgs) {
  const type = params.type;

  if (!type) {
    throw new Response("Studio Type Not Found", { status: 404 });
  }

  // Import translations dynamically
  const enModule = await import("~/config/lang/en.json");
  const spModule = await import("~/config/lang/sp.json");
  
  // Process URLs in translations
  const processedTranslations = {
    en: processTranslationUrls(enModule.default || enModule),
    sp: processTranslationUrls(spModule.default || spModule),
  };

  // Fetch API data
  const apiRes = await fetch(
    `${API_BASE_URL}/studios?type=${type}`
  );
  const apiData = await apiRes.json();
  console.log(apiData);
  
  return Response.json({ type, apiData, translations: processedTranslations });
}

export const meta: MetaFunction<typeof loader> = ({ data }) => {
  if (!data) {
    return [
      { title: "Studio Not Found | Nomadics" },
      {
        name: "description",
        content: "The requested studio type was not found.",
      },
    ];
  }

  // Use default English for meta tags since we can't access hooks in meta
  const studioData = data.translations?.en?.[data.type];
  
  return [
    { title: studioData?.meta?.title || `${data.type} Studio | Nomadics` },
    { name: "description", content: studioData?.meta?.description || `Explore ${data.type} studios at Nomadics` },
  ];
};

export default function StudioPage() {
  const { type, apiData, translations } = useLoaderData<typeof loader>();
  const { currentLanguage, t } = useLanguage();
  const navigate = useNavigate();
  
  // Get studio data based on current language (URLs already processed in loader)
  const studioData = translations[currentLanguage]?.[type] || translations.en[type];

  if (!studioData || !studioData.welcome) {
    throw new Response("Studio Type Not Found", { status: 404 });
  }

  // Add onClick handlers to existing API data items if they exist
  if (apiData?.items && Array.isArray(apiData.items)) {
    console.log('API Data items:', apiData.items);
    apiData.items = apiData.items.map((item: any) => {
      console.log('Individual item:', item);
      return {
        ...item,
        onClick: () => {
          // Extract slug from the item - try multiple possible field names
          const slug = item.studio_slug || item.slug || item.studioSlug || item.studio_name?.toLowerCase().replace(/\s+/g, '-') || item.title?.toLowerCase().replace(/\s+/g, '-');
          console.log('Navigating with slug:', slug, 'from item:', item);
          if (slug) {
            navigate(`/studio/${slug}`);
          } else {
            console.error('No slug found for item:', item);
          }
        }
      };
    });
  } else {
    console.log('No items found in apiData:', apiData);
  }

  return (
    <div>
      <WelcomeSection data={studioData.welcome} />
      <StudiosSection data={studioData.studios} />
      <PhotoSection data={studioData.gallery} />
      <BookSection
        title={studioData.book.title}
        description={studioData.book.description}
        data={apiData}
      />
      <PeopleSection 
        title={t('people.ourCrew', 'OUR CREW')}
        eyebrow="MEET THE TEAM"
        buttonText={t('people.viewAll', 'VIEW ALL PEOPLE')}
        buttonHref="/people"
      />
      <MembershipSection theme="black" />
    </div>
  );
}
