# Remix Documentation

## Overview

Remix is a full-stack web framework built on web standards that helps you build better websites. It focuses on creating modern, resilient user experiences with web fundamentals.

## Quick Start

### Create a New Remix Project

```bash
npx create-remix@latest
```

This command initializes a new Remix project using the create-remix CLI. It guides you through project setup via terminal prompts.

### Manual Setup

If you prefer to set up manually:

```bash
mkdir my-remix-app
cd my-remix-app
npm init -y

# install runtime dependencies
npm i @remix-run/node @remix-run/react @remix-run/serve isbot@4 react@18 react-dom@18

# install dev dependencies
npm i -D @remix-run/dev vite
```

### Basic Vite Configuration

Create a `vite.config.js` file:

```javascript
import { vitePlugin as remix } from "@remix-run/dev";
import { defineConfig } from "vite";

export default defineConfig({
  plugins: [remix()],
});
```

### Start Development Server

```bash
npm run dev
```

## Project Structure

### Basic File Structure

A typical Remix project includes:
- `app/` - Your application code
  - `routes/` - Route modules
  - `root.tsx` - Root layout component
  - `entry.client.tsx` - Browser entry point
  - `entry.server.tsx` - Server entry point
- `public/` - Static assets
- `build/` - Build output (generated)

### Routes

Remix uses file-based routing. Files in the `app/routes` directory automatically become routes:

- `app/routes/_index.tsx` → `/`
- `app/routes/about.tsx` → `/about`
- `app/routes/posts.$slug.tsx` → `/posts/:slug`
- `app/routes/posts._index.tsx` → `/posts`

## Core Concepts

### Data Loading

Use `loader` functions to fetch data on the server:

```typescript
import { json } from "@remix-run/node";
import { useLoaderData } from "@remix-run/react";

export async function loader() {
  const data = await fetchYourData();
  return json(data);
}

export default function MyRoute() {
  const data = useLoaderData<typeof loader>();
  return <div>{/* Use your data */}</div>;
}
```

### Data Mutations

Use `action` functions to handle form submissions:

```typescript
import { Form } from "@remix-run/react";
import { redirect } from "@remix-run/node";

export async function action({ request }) {
  const formData = await request.formData();
  // Process form data
  return redirect("/success");
}

export default function MyForm() {
  return (
    <Form method="post">
      <input name="title" type="text" />
      <button type="submit">Submit</button>
    </Form>
  );
}
```

### Forms

Remix leverages HTML forms with progressive enhancement:

```tsx
import { Form } from "@remix-run/react";

export default function ContactForm() {
  return (
    <Form method="post" action="/contact">
      <label>
        Name: <input name="name" type="text" />
      </label>
      <label>
        Email: <input name="email" type="email" />
      </label>
      <button type="submit">Send</button>
    </Form>
  );
}
```

## Deployment

### Build for Production

```bash
npm run build
```

### Start Production Server

```bash
npm start
```

### Custom Express Server

Install Express and the Remix adapter:

```bash
npm i express@4 @remix-run/express cross-env
```

Create `server.js`:

```javascript
import { createRequestHandler } from "@remix-run/express";
import express from "express";
import * as build from "./build/server/index.js";

const app = express();
app.use(express.static("build/client"));

app.all("*", createRequestHandler({ build }));

app.listen(3000, () => {
  console.log("App listening on http://localhost:3000");
});
```

Update `package.json` scripts:

```json
{
  "scripts": {
    "dev": "node ./server.js",
    "start": "cross-env NODE_ENV=production node ./server.js"
  }
}
```

## Database Integration

### Using Prisma

Install Prisma:

```bash
npm install --save-dev prisma
npm install @prisma/client
```

Initialize Prisma with SQLite:

```bash
npx prisma init --datasource-provider sqlite
```

Example model fetching:

```typescript
import { prisma } from "~/db.server";

export async function getPosts() {
  return prisma.post.findMany();
}
```

## Styling

Remix supports various styling approaches:
- CSS imports
- CSS Modules
- Tailwind CSS
- CSS-in-JS libraries

## Authentication

For secure authentication, install bcrypt:

```bash
npm install bcryptjs
npm install --save-dev @types/bcryptjs
```

## Templates and Examples

Create from templates:

```bash
# Basic template
npx create-remix@latest --template remix-run/examples/basic

# Indie Stack (with auth and database)
npx create-remix@latest --template remix-run/indie-stack

# Express template
npx create-remix@latest --template remix-run/remix/templates/express
```

## Platform-Specific Deployments

### Cloudflare Workers

```bash
npx create-remix@latest --template remix-run/remix/templates/cloudflare-workers
```

Build and deploy:

```bash
npm run build
npm run deploy
```

### Fly.io

```bash
flyctl auth signup
flyctl launch
npm run deploy
```

### Deno

```bash
npx create-remix@latest --template remix-run/remix/templates/deno
```

## Development Tools

### Reveal Entry Files

To customize entry files:

```bash
npx remix reveal
```

### TypeScript Support

Remix has built-in TypeScript support. Use `.ts` and `.tsx` files throughout your application.

### Development Dependencies

For TypeScript execution:

```bash
npm install --save-dev ts-node tsconfig-paths
```

## Best Practices

1. **Use Web Standards**: Remix is built on web standards - leverage native browser APIs
2. **Progressive Enhancement**: Forms work without JavaScript by default
3. **Server-Side Rendering**: All routes are server-rendered by default
4. **Type Safety**: Use TypeScript for better developer experience
5. **Error Boundaries**: Implement error boundaries for graceful error handling
6. **Optimistic UI**: Use Remix's built-in optimistic UI features for better UX

## Common Commands

- `npm run dev` - Start development server
- `npm run build` - Build for production
- `npm start` - Start production server
- `npx prisma db push` - Push database schema changes
- `npx prisma db seed` - Seed database with initial data

## Resources

- [Official Remix Documentation](https://remix.run/docs)
- [Remix GitHub Repository](https://github.com/remix-run/remix)
- [Remix Examples](https://github.com/remix-run/examples)
- [Remix Discord Community](https://rmx.as/discord)